package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.util.Event;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.io.Serializable;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.function.Predicate;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.RawCommand;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSFilterTelemetry;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSImageNameEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSettingsAppliedEvent;

/**
 * An implementation of MCMLayer which uses the CCS buses to communicate with an
 * MCM running in a separate process.
 *
 * @author tonyj
 */
class MCMCCSLayer implements MCMLayer {

    private final Subsystem subsystem;
    private final CCS ccs;
    private final OCSBridgeConfig config;

    MCMCCSLayer(Subsystem subsystem, CCS ccs, OCSBridgeConfig config) {
        this.subsystem = subsystem;
        this.ccs = ccs;
        this.config = config;

        Predicate<BusMessage<? extends Serializable, ?>> stateChangeFilter
                = BusMessageFilterFactory.messageOrigin(config.getMCMName()).and(BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class));

        subsystem.getMessagingAccess().addStatusMessageListener((msg) -> handleMCMStatusChangeMessage(msg), stateChangeFilter);

        Predicate<BusMessage<? extends Serializable, ?>> eventFilter
                = BusMessageFilterFactory.messageOrigin(config.getMCMName()).and(BusMessageFilterFactory.embeddedObjectClass(KeyValueData.class));

        subsystem.getMessagingAccess().addStatusMessageListener((msg) -> handleMCMEvent(msg), eventFilter);
    }

    @Override
    public CCSCommand.CCSCommandResponse execute(CCSCommand ccsCommand) {
        CCSExecutor executor = new CCSBusesExecutor(ccsCommand);
        return new CCSCommand.CCSCommandResponse(executor);
    }

    @Override
    public void addStateChangeListener(State.StateChangeListener<Enum> stateChangeListener) {
        ccs.addStateChangeListener(stateChangeListener);
    }

    @Override
    public void removeStateChangeListener(State.StateChangeListener<Enum> stateChangeListener) {
        ccs.removeStateChangeListener(stateChangeListener);
    }

    @Override
    public void addEventListener(Event.EventListener eventListener) {
        ccs.addEventListener(eventListener);
    }

    @Override
    public void removeEventListener(Event.EventListener eventListener) {
        ccs.removeEventListener(eventListener);
    }

    private void handleMCMStatusChangeMessage(StatusMessage msg) {
        StatusStateChangeNotification statusChange = (StatusStateChangeNotification) msg;
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        System.out.println("Changes: " + changedStates.getAllStatesAsStrings());
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            ccs.getAggregateStatus().add(new State(value));
        });
    }

    private void handleMCMEvent(StatusMessage msg) {
        Object object = msg.getObject();
        if (object instanceof KeyValueData) {
            KeyValueData data = (KeyValueData) object;
            switch (data.getKey()) {
                case "AvailableFilters":
                    List<String> filters = Arrays.asList(((String) data.getValue()).split(","));
                    ccs.fireEvent(new CCSAvailableFiltersEvent(filters));
                    break;
                case "ImageName": {
                    List<Object> args = (List<Object>) data.getValue();
                    ccs.fireEvent(new CCSImageNameEvent((String) args.get(0), (int) args.get(1), (String) args.get(2), (Integer) args.get(3), (Long) args.get(4), (Double) args.get(5)));
                    break;
                }
                case "SetFilter": {
                    List<Object> args = (List<Object>) data.getValue();
                    ccs.fireEvent(new CCSSetFilterEvent((String) args.get(0), (Boolean) args.get(1)));
                    break;
                }
                case "FilterTelemetry":
                    int id = ((Integer) data.getValue());
                    ccs.fireEvent(new CCSFilterTelemetry((short) id));
                    break;
                case "SettingsApplied": {
                    List<Object> args = (List<Object>) data.getValue();
                    ccs.fireEvent(new CCSSettingsAppliedEvent((String) args.get(0), (Double) args.get(1)));
                    break;
                }
                default:
                    //throw new IllegalArgumentException("Unknown event: " + data.getKey());
            }
        }
    }

    private class CCSBusesExecutor extends CCSExecutor {

        private final ArrayBlockingQueue queue = new ArrayBlockingQueue(2);

        public CCSBusesExecutor(CCSCommand ccsCommand) {
            CommandOriginator originator = new CommandOriginator() {
                @Override
                public void processAck(CommandAck ack) {
                    queue.offer(ack);
                }

                @Override
                public void processResult(CommandResult result) {
                    queue.offer(result);
                }

                @Override
                public void processNack(CommandNack nack) {
                    queue.offer(nack);
                }
            };
            CommandRequest request = new CommandRequest(config.getMCMName(), new RawCommand(ccsCommand.getCommand(), ccsCommand.getArguments()));
            subsystem.getMessagingAccess().sendCommandRequest(request, originator);
        }

        @Override
        protected Duration testPreconditions() throws CCSCommand.CCSPreconditionsNotMet {
            try {
                Object result = queue.take();
                if (result instanceof CommandAck) {
                    return ((CommandAck) result).getTimeout();
                } else if (result instanceof CommandNack) {
                    throw new CCSCommand.CCSPreconditionsNotMet(((CommandNack) result).getReason());
                } else {
                    throw new RuntimeException("Unexpected object received while waiting for ack/nack");
                }
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting for command ack/nack", ex);
            }
        }

        @Override
        protected void execute() throws Exception {
            Object result = queue.take();
            if (result instanceof CommandResult) {
                CommandResult cr = (CommandResult) result;
                if (!cr.wasSuccessful()) {
                    throw new RuntimeException("Command failed: " + cr.toString());
                }
            } else {
                throw new RuntimeException("Unexpected object received while waiting for command result");
            }
        }
    }

}
