package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Base class for all CCSCommands. This class is abstract and is extended by
 * specific implementations for each supported command. Note that this class
 * only encapsulates the command and its arguments, it does not include any code
 * for actually executing the command.
 *
 * @author tonyj
 */
public abstract class CCSCommand {

    /**
     * Gets the command name.
     *
     * @return The command name.
     */
    abstract String getCommand();

    /**
     * Gets the command arguments.
     *
     * @return The command arguments.
     */
    abstract List<Object> getArguments();

    public static class CCSInitImageCommand extends CCSCommand {

        private final double deltaT;

        public CCSInitImageCommand(double deltaT) {
            this.deltaT = deltaT;
        }

        public double getDeltaT() {
            return deltaT;
        }

        @Override
        String getCommand() {
            return "initImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(deltaT);
        }

    }

    public static class CCSTakeImagesCommand extends CCSCommand {

        private final int numImages;
        private final double expTime;
        private final boolean shutter;
        private final boolean science;
        private final boolean wfs;
        private final boolean guide;
        private final String imageSequenceName;

        public CCSTakeImagesCommand(double expTime, int numImages, boolean shutter, boolean science, boolean wfs, boolean guide, String imageSequenceName) {
            this.numImages = numImages;
            this.expTime = expTime;
            this.shutter = shutter;
            this.science = science;
            this.wfs = wfs;
            this.guide = guide;
            this.imageSequenceName = imageSequenceName;
        }

        public int getNumImages() {
            return numImages;
        }

        public double getExpTime() {
            return expTime;
        }

        public boolean isShutter() {
            return shutter;
        }

        public boolean isScience() {
            return science;
        }

        public boolean isWfs() {
            return wfs;
        }

        public boolean isGuide() {
            return guide;
        }

        public String getImageSequenceName() {
            return imageSequenceName;
        }

        @Override
        String getCommand() {
            return "takeImages";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(expTime, numImages, shutter, science, wfs, guide, imageSequenceName);
        }
    }

    public static class CCSSetFilterCommand extends CCSCommand {

        private final String filterName;

        public CCSSetFilterCommand(String filterName) {
            this.filterName = filterName;
        }

        public String getFilterName() {
            return filterName;
        }

        @Override
        String getCommand() {
            return "setFilter";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(filterName);
        }
    }

    public static class CCSInitGuidersCommand extends CCSCommand {

        private final String roiSpec;

        public CCSInitGuidersCommand(String roiSpec) {
            this.roiSpec = roiSpec;
        }

        public String getRoiSpec() {
            return roiSpec;
        }

        @Override
        String getCommand() {
            return "initGuiders";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(roiSpec);
        }
    }

    public static class CCSStartImageCommand extends CCSCommand {

        private final String imageSequenceName;
        private final boolean shutterOpen;
        private final boolean science;
        private final boolean wavefront;
        private final boolean guide;
        private final double timeout;

        public CCSStartImageCommand(String imageSequenceName, boolean shutterOpen, boolean science, boolean wavefront, boolean guide, double timeout) {
            this.imageSequenceName = imageSequenceName;
            this.shutterOpen = shutterOpen;
            this.science = science;
            this.wavefront = wavefront;
            this.guide = guide;
            this.timeout = timeout;
        }

        public String getImageSequenceName() {
            return imageSequenceName;
        }

        public boolean isShutterOpen() {
            return shutterOpen;
        }

        public boolean isScience() {
            return science;
        }

        public boolean isWavefront() {
            return wavefront;
        }

        public boolean isGuide() {
            return guide;
        }

        public double getTimeout() {
            return timeout;
        }

        @Override
        String getCommand() {
            return "startImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(imageSequenceName, shutterOpen, science, wavefront, guide, timeout);
        }
    }

    public static class CCSStartCommand extends CCSCommand {

        private final String configuration;

        public CCSStartCommand(String configuration) {
            this.configuration = configuration;
        }

        public String getConfiguration() {
            return configuration;
        }

        @Override
        public String toString() {
            return "CCSStartCommand{" + "configuration=" + configuration + '}';
        }

        @Override
        String getCommand() {
            return "start";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(configuration);
        }
    }

    static abstract class NoArgCCSCommand extends CCSCommand {

        private String commandName;

        NoArgCCSCommand(String commandName) {
            this.commandName = commandName;
        }

        @Override
        String getCommand() {
            return commandName;
        }

        @Override
        List<Object> getArguments() {
            return Collections.emptyList();
        }
    }
    
    public static class CCSEnableCalibrationCommand extends NoArgCCSCommand {

        public CCSEnableCalibrationCommand() {
            super("enableCalibration");
        }

    }
    
    public static class CCSDisableCalibrationCommand extends NoArgCCSCommand {

        public CCSDisableCalibrationCommand() {
            super("disableCalibration");
        }

    }   
    
    public static class CCSEndImageCommand extends NoArgCCSCommand {

        public CCSEndImageCommand() {
            super("endImage");
        }

    }

    public static class CCSSetAvailableCommand extends NoArgCCSCommand {

        public CCSSetAvailableCommand() {
            super("setAvailable");
        }
    }

    public static class CCSRevokeAvailableCommand extends NoArgCCSCommand {

        public CCSRevokeAvailableCommand() {
            super("revokeAvailable");
        }
    }

    public static class CCSSimulateFaultCommand extends NoArgCCSCommand {

        public CCSSimulateFaultCommand() {
            super("simulateFault");
        }
    }

    public static class CCSClearFaultCommand extends NoArgCCSCommand {

        public CCSClearFaultCommand() {
            super("clearFault");
        }
    }

    public static class CCSDiscardRowsCommand extends CCSCommand {

        private final int nRows;

        public CCSDiscardRowsCommand(int nRows) {
            this.nRows = nRows;
        }

        public int getnRows() {
            return nRows;
        }

        @Override
        String getCommand() {
            return "discardRows";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(nRows);
        }
    }

    public static class CCSClearCommand extends CCSCommand {

        private final int nClears;

        public CCSClearCommand(int nClears) {
            this.nClears = nClears;
        }

        public int getNClears() {
            return nClears;
        }

        @Override
        String getCommand() {
            return "clear";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(nClears);
        }

    }

    public static class CCSCommandResponse {

        private final CCSExecutor executor;

        public CCSCommandResponse(CCSExecutor executor) {
            this.executor = executor;
        }

        public CCSAckOrNack waitForAckOrNack() {
            try {
                Duration duration = executor.testPreconditions();
                return new CCSAckOrNack(duration);
            } catch (CCSPreconditionsNotMet ex) {
                return new CCSAckOrNack(ex.getReason());
            }
        }

        public void waitForCompletion() throws Exception {
            executor.execute();
        }

    }

    public static class CCSAckOrNack {

        private final boolean ack;
        private final Duration duration;
        private String reason;

        private CCSAckOrNack(Duration duration) {
            this.ack = true;
            this.reason = null;
            this.duration = duration;
        }

        private CCSAckOrNack(String reason) {
            this.ack = false;
            this.reason = reason;
            this.duration = null;
        }

        public boolean isNack() {
            return !ack;
        }

        public String getReason() {
            return reason;
        }

        public Duration getDuration() {
            return duration;
        }

    }

    public static class CCSPreconditionsNotMet extends Exception {

        private static final long serialVersionUID = 7463370916615596884L;

        private final String reason;

        public CCSPreconditionsNotMet(String reason) {
            super("Reason : " + reason);
            this.reason = reason;
        }

        private String getReason() {
            return reason;
        }
    }
}
