package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.subsystem.comcam.filterchanger.data.EndSetFilterData;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.states.FcsPowerState;
import org.lsst.ccs.subsystem.ocsbridge.states.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class FilterChangerSimulation implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(FilterChangerSimulation.class.getName());

    // This should all become (simulation only) configuration
    private static final Map<String, FilterType> mainCameraFilters = new LinkedHashMap<>();
    static {
        // These are filter available to be installed in the camera as of April 2025
        mainCameraFilters.put("u_24", FilterType.U);
        mainCameraFilters.put("g_6", FilterType.G);
        mainCameraFilters.put("r_57", FilterType.R);
        mainCameraFilters.put("i_39", FilterType.I);
        mainCameraFilters.put("z_20", FilterType.Z);
        mainCameraFilters.put("y_10", FilterType.Y);
        mainCameraFilters.put("ef_43", FilterType.EF);
        mainCameraFilters.put("ph_5", FilterType.PH);
        mainCameraFilters.put("NONE", FilterType.NONE);
    }

    private static final List<String> mainCameraInstalledFilters = new ArrayList<>();
    static {
        mainCameraInstalledFilters.add("u_24");
        mainCameraInstalledFilters.add("g_6");
        mainCameraInstalledFilters.add("r_57");
        mainCameraInstalledFilters.add("i_39");
        mainCameraInstalledFilters.add("z_20");
        mainCameraInstalledFilters.add("NONE");
    }

    private static final Map<String, FilterType> comCamFilters = new LinkedHashMap<>();

    static {
        // These are taken from comcam-fcs configuration
        comCamFilters.put("u_05", FilterType.U);
        comCamFilters.put("g_07", FilterType.G);
        comCamFilters.put("g_01", FilterType.G);
        comCamFilters.put("r_03", FilterType.R);
        comCamFilters.put("i_06", FilterType.I);
        comCamFilters.put("z_03", FilterType.Z);
        comCamFilters.put("z_02", FilterType.Z);
        comCamFilters.put("y_04", FilterType.Y);
        comCamFilters.put("pinhole", FilterType.OTHER);
        comCamFilters.put("NONE", FilterType.NONE);
    }

    private static final List<String> comCamInstalledFilters = new ArrayList<>();
    static {
        comCamInstalledFilters.add("i_06");
        comCamInstalledFilters.add("r_03");
        comCamInstalledFilters.add("g_07");
    }

    static final Duration LOAD_TIME = Duration.ofMillis(15000);
    static final Duration ROTATION_TIME_PER_DEGREE = Duration.ofMillis(100);
    static final Duration UNLOAD_TIME = Duration.ofMillis(15000);

    private final Map<String, FilterType> allFilters;
    private final List<String> installedFilters;

    private final MCMConfig config;
    private String currentFilter;
    private int currentRotationPosition = 0;
    private final CCS ccs;
    private final State<FilterState> filterState;
    private final State<FcsPowerState> powerState;
    private boolean powerSaveAllowed = true;

    /* Last published FCSState and lock */
    private volatile KeyValueDataList lastPublishedFCStateData = null;
    private final Object lastPublishedFCStateDataLock = new Object();

    private Agent agent;

    private final Random random = new Random();

    FilterChangerSimulation(CCS ccs, MCMConfig config) {

        this.ccs = ccs;
        this.config = config;
        switch (config.getCameraType()) {
            case MAIN_CAMERA -> {
                allFilters = mainCameraFilters;
                installedFilters = mainCameraInstalledFilters;
                currentFilter = mainCameraInstalledFilters.get(0);
            }
            case COMCAM -> {
                allFilters = comCamFilters;
                installedFilters = comCamInstalledFilters;
                currentFilter = comCamInstalledFilters.get(0);
            }
            default -> throw new RuntimeException("Filter changer not supported for " + config.getCameraType());
        }
        filterState = new State<FilterState>(FilterState.LOADED);
        powerState = new State<FcsPowerState>(FcsPowerState.LOW_POWER);
        ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), filterState);
        ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), powerState);
    }

    @Override
    public List<String> getAvailableFilters() {
        return Collections.unmodifiableList(installedFilters);
    }

    @Override
    public List<String> getInstalledFilters() {
        return Collections.unmodifiableList(installedFilters);
    }

    @Override
    public String getFilterType(String filterName) {
        return allFilters.get(filterName).getName();
    }

    @Override
    public void start(String configName) {
        /// No Op
    }

    @Override
    public String getCurrentFilter() {
        return currentFilter;
    }

    @Override
    public void setFilter(String filterName) {
        try {
            int targetRotation = -1;
            int position = getInstalledFilters().indexOf(filterName);
            if (position < 0) {
                throw new IllegalArgumentException("Invalid or uninstalled filter: " + filterName);
            }

            if (filterName.equals(currentFilter)) {
                return;
            }

            if ( config.getCameraType().equals(Camera.MAIN_CAMERA) && powerState.isInState(FcsPowerState.LOW_POWER)) {
                wakeFilterChanger(1);
            }

            FilterType requestedFilterType = allFilters.get(filterName);
            FilterType currentFilterType = allFilters.get(currentFilter);
            ccs.fireEvent(new CCSSetFilterEvent(filterName, requestedFilterType.getName()));

            if (!currentFilterType.equals(FilterType.NONE)) {
                filterState.setState(FilterState.UNLOADING);
                Future<Void> waitForUnloaded = ccs.waitForStatus(FilterState.UNLOADED);
                ccs.schedule(UNLOAD_TIME, () -> {
                    filterState.setState(FilterState.UNLOADED);
                    currentFilter = filterName;
                });
                waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            }

            if (!requestedFilterType.equals(FilterType.NONE)) {
                targetRotation = (position + 1) * 360 / 5;
                if (this.currentRotationPosition != targetRotation) {
                    int randomSockets = getRandomRelativeCarouselSocket();
                    int degreesToRotate = Math.abs(currentRotationPosition - randomSockets * targetRotation) % 360;
                    filterState.setState(FilterState.ROTATING);
                    Future<Void> waitForRotation = ccs.waitForStatus(FilterState.UNLOADED);
                    Duration rotationTime = ROTATION_TIME_PER_DEGREE.multipliedBy(degreesToRotate);
                    ccs.schedule(rotationTime, () -> {
                        filterState.setState(FilterState.UNLOADED);
                    });
                    waitForRotation.get(rotationTime.toMillis() * 2, TimeUnit.MILLISECONDS);
                    this.currentRotationPosition = targetRotation;
                }
                filterState.setState(FilterState.LOADING);
                Future<Void> waitForLoaded = ccs.waitForStatus(FilterState.LOADED);
                ccs.schedule(LOAD_TIME, () -> {
                    filterState.setState(FilterState.LOADED);
                    currentFilter = filterName;
                });
                waitForLoaded.get(LOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            }

            if ( config.getCameraType().equals(Camera.COMCAM) ) {
                String fname = getCurrentFilter(); 
                String ftype = getFilterType(fname);
                int fslot = 1;
                int fpos = getInstalledFilters().indexOf(fname);
                EndSetFilterData data = new EndSetFilterData(fname, ftype, 
                                                         fslot, fpos);
                KeyValueDataList kvd = new KeyValueDataList();
                kvd.addData(EndSetFilterData.endKey,data);
                publishCurrentFCState(kvd);
            }
            
            ccs.fireEvent(new CCSSetFilterEvent(filterName, requestedFilterType.getName(), position, targetRotation));


            if ( config.getCameraType().equals(Camera.MAIN_CAMERA) && powerSaveAllowed ) {
                wakeFilterChanger(0);
            }
        } catch (InterruptedException | ExecutionException | TimeoutException ex) {
            throw new RuntimeException("Error while changing filter", ex);
        }
    }

    private int getRandomRelativeCarouselSocket() {
        int randomStep;
        do {
            // Generate a random integer between -2 and 2 (inclusive)
            randomStep = random.nextInt(5) - 2;
        } while (randomStep == 0); // Repeat if the value is zero
        return randomStep;
    }

    @Override
    public Duration getDurationForFastFilterChange(String filterName) {
        return ROTATION_TIME_PER_DEGREE.multipliedBy(360).plus(LOAD_TIME).plus(UNLOAD_TIME);
    }

    @Override
    public void wakeFilterChanger(int mode) throws ExecutionException {
        Duration timeout = getDurationForWakeUp(mode);
        if ( mode == 2 ) {
            powerSaveAllowed = false;
        } else {
            powerSaveAllowed = true;
        }
        switch (mode) {
            case 0 -> {
                try {
                    if (powerState.isInState(FcsPowerState.REGULAR)) {
                        Future<Void> waitForTimeout = ccs.waitForStatus(FcsPowerState.LOW_POWER);
                        ccs.schedule(timeout, () -> {
                            powerState.setState(FcsPowerState.LOW_POWER);
                        });
                        waitForTimeout.get(timeout.toMillis() * 2, TimeUnit.MILLISECONDS);
                    } else {
                        // Do nothing
                    }
                } catch (InterruptedException | ExecutionException | TimeoutException ex) {
                    throw new ExecutionException("Error while FES going to sleep", ex);
                }
            }
            case 1, 2 -> {
                try {
                    if (powerState.isInState(FcsPowerState.LOW_POWER)) {
                        Future<Void> waitForTimeout = ccs.waitForStatus(FcsPowerState.REGULAR);
                        ccs.schedule(timeout, () -> {
                            powerState.setState(FcsPowerState.REGULAR);
                        });
                        waitForTimeout.get(timeout.toMillis() * 2, TimeUnit.MILLISECONDS);
                    } else {
                        // Do nothing
                    }
                } catch (InterruptedException | ExecutionException | TimeoutException ex) {
                    throw new ExecutionException("Error while FES waking up", ex);
                }
            }
            default -> throw new RuntimeException("Unsupported wake up mode");
        }
    }

    @Override
    public Duration getDurationForWakeUp(int mode) {
        return switch (mode) {
            case 0 -> Duration.ofMillis(2000);
            case 1, 2 -> Duration.ofMillis(4000);
            default -> throw new RuntimeException("Unsupported wake up mode");
        };
    }
    
    private void publishCurrentFCState(KeyValueDataList kvd) {
        synchronized(lastPublishedFCStateDataLock) {
            if ( kvd != null ) {
                lastPublishedFCStateData = kvd;
            }
            LOG.log(Level.FINE,"Publishing current FCS State: "+lastPublishedFCStateData);
            if (lastPublishedFCStateData != null) {
                if ( agent != null ) {
                    agent.publishSubsystemDataOnStatusBus(lastPublishedFCStateData);
                }
            }
        }
    }
    
    void setTopLevelAgent(Agent agent) {
        this.agent = agent;
    }
    
    void publishDataProviderCurrentData() {
        publishCurrentFCState(null);
    }

}
