package org.lsst.ccs.subsystem.ocsbridge;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.subsystem.imagehandling.data.HeaderServiceEnabled;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData;
import org.lsst.ccs.subsystem.imagehandling.data.ImageHeaderData.Header;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.sal.SALEvent;

/**
 * This class is used to receive header service events in the ocs-bridge, and forward them to the image
 * handlers for inclusion in the FITS headers. The class is instantiated from groovy and registered as 
 * a SALEventListener in OCSBridge.postStart.
 * 
 * @see 
 * @author tonyj
 */
public class HeaderServiceEventHandler implements EventListener<SALEvent>, HasLifecycle, HasDataProviderInfos {

    private static final Logger LOG = Logger.getLogger(HeaderServiceEventHandler.class.getName());
    private final ObjectMapper mapper;
    private final TypeReference<Map<String, List<Header>>> yamlType = new TypeReference<Map<String, List<Header>>>() {
    };

    private volatile boolean headerServiceEnabled = true;
    private final Object hseLock = new Object();
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsystem;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dataProviderDictionaryService;

    public HeaderServiceEventHandler() {
        mapper = new ObjectMapper(new YAMLFactory());
        mapper.findAndRegisterModules();
        mapper.addMixIn(Header.class,HeaderMixIn.class);
    }
    
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(HeaderServiceEnabled.class, HeaderServiceEnabled.EVENT_KEY);
        dataProviderDictionaryService.registerClass(ImageHeaderData.class, ImageHeaderData.EVENT_KEY);
    }
    
    
    @Override
    public void eventFired(SALEvent event) {
        if (event instanceof org.lsst.sal.atheader.event.LargeFileObjectAvailableEvent lfoe) {
            String url = lfoe.getUrl();
            handleEvent(url);
        } else if (event instanceof org.lsst.sal.ccheader.event.LargeFileObjectAvailableEvent lfoe) {
            String url = lfoe.getUrl();
            handleEvent(url);
        } else if (event instanceof org.lsst.sal.mtheader.event.LargeFileObjectAvailableEvent lfoe) {
            String url = lfoe.getUrl();
            handleEvent(url);
        } else if (event instanceof org.lsst.sal.atheader.states.SummaryStateEvent sse) {
            LOG.log(Level.INFO, "Header service state changed {0}", sse);
            HeaderServiceEnabled hse = new HeaderServiceEnabled(sse.getSubstate() == org.lsst.sal.atheader.states.SummaryStateEvent.SummaryState.ENABLED); 
            processHeaderServiceEnabled(hse);
        } else if (event instanceof org.lsst.sal.ccheader.states.SummaryStateEvent sse) {
            LOG.log(Level.INFO, "Header service state changed {0}", sse);
            HeaderServiceEnabled hse = new HeaderServiceEnabled(sse.getSubstate() == org.lsst.sal.ccheader.states.SummaryStateEvent.SummaryState.ENABLED); 
            processHeaderServiceEnabled(hse);
        } else if (event instanceof org.lsst.sal.mtheader.states.SummaryStateEvent sse) {
            LOG.log(Level.INFO, "Header service state changed {0}", sse);
            HeaderServiceEnabled hse = new HeaderServiceEnabled(sse.getSubstate() == org.lsst.sal.mtheader.states.SummaryStateEvent.SummaryState.ENABLED); 
            processHeaderServiceEnabled(hse);
        }
    }
    
    //When the provided HeaderServiceEnabled object is null, then we
    //publish the last stored value. This is triggered by an ImageHandler
    //subsystem joining the cluster.
    private void processHeaderServiceEnabled(HeaderServiceEnabled hse) {
        synchronized(hseLock) {
            if ( hse != null ) {
                headerServiceEnabled = hse.isEnabled();
            } else {
                hse = new HeaderServiceEnabled(headerServiceEnabled);
            }
            KeyValueData kvd = new KeyValueData(HeaderServiceEnabled.EVENT_KEY, hse);
            subsystem.publishSubsystemDataOnStatusBus(kvd);
        }
    }
    
    void handleEvent(String url) {
        try {
            LOG.log(Level.INFO, "Header service got {0}", url);
            
            URL yamlUrl = new URL(url);
            //Code to convert yaml ".nan" into "NaN"
            String content;
            try (InputStream in = yamlUrl.openStream()) {
                content = new String(in.readAllBytes(), StandardCharsets.UTF_8).replace(".nan","NaN");
            }
            Map<String, List<Header>> header = mapper.readValue(content, yamlType);
            
            //NaN is converted to a String Object by the mapper.
            //We scan for such Headers and replace the value with Double.NaN
            for (String headerName : header.keySet() ) {                
                ListIterator<Header> iter = header.get(headerName).listIterator();                
                while(iter.hasNext()) {
                    Header h = iter.next();
                    Object value = h.getValue();
                    if ( value != null ) {
                        if (value.getClass() == String.class && "NaN".equals(value)) {
                            Header newHeader = new Header(h.getKeyword(),Double.NaN,h.getComment());
                            iter.remove();
                            iter.add(newHeader);
                        }
                    }
                }
            }
            
            
            
            sendPrimaryHeaderViaCCS(header.get("PRIMARY"));
        } catch (IOException ex) {
            LOG.log(Level.SEVERE, "Unexpected exception handling header service URL", ex);
        }
    }

    void sendPrimaryHeaderViaCCS(List<Header> data) {
        KeyValueData kvd = new KeyValueData(ImageHeaderData.EVENT_KEY, new ImageHeaderData(data));
        subsystem.publishSubsystemDataOnStatusBus(kvd);
        LOG.log(Level.FINE, "Sent {0}", kvd);
    }

    /// Used to teach Jackson how to create Header class
    public static abstract class HeaderMixIn {

        @JsonCreator
        public HeaderMixIn(@JsonProperty("keyword") String keyword, @JsonProperty("value") Object value, @JsonProperty("comment") String comment) {

        }
    }

    @Override
    public void publishDataProviderCurrentData(AgentInfo... agents) {
        //Publish the data only for image handlers
        for ( AgentInfo a : agents ) {
            if ( a.getAgentProperties().getProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"").equals(AgentCategory.IMAGE_HANDLER.name()) ) {
                processHeaderServiceEnabled(null);
                break;                
            }            
        }        
    }

}
