package org.lsst.ccs.subsystem.ocsbridge.util;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
// specific CCSCommands - 
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDefinePlaylistCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPlayCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStandbyCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSWakeFilterChangerCommand;

import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.command.ClearCommand;
import org.lsst.sal.camera.command.DisableCalibrationCommand;
import org.lsst.sal.camera.command.DiscardRowsCommand;
import org.lsst.sal.camera.command.EnableCalibrationCommand;
import org.lsst.sal.camera.command.EndImageCommand;
import org.lsst.sal.camera.command.InitGuidersCommand;
import org.lsst.sal.camera.command.InitImageCommand;
import org.lsst.sal.camera.command.PlayCommand;
import org.lsst.sal.camera.command.PlaylistCommand;
import org.lsst.sal.camera.command.SetFilterCommand;
import org.lsst.sal.camera.command.StandbyCommand;
import org.lsst.sal.camera.command.StartCommand;
import org.lsst.sal.camera.command.StartImageCommand;
import org.lsst.sal.camera.command.TakeImagesCommand;
import org.lsst.sal.camera.command.WakeFilterChangerCommand;

/**
 *
 * @author Farrukh Azfar
 */
public class OCSCommandConverter {

    private final Map< Class<? extends CameraCommand>, Class<? extends CCSCommand>> ocsToCcsCommandMap;
    private final DelimitedStringSplitJoin delimitedStringSplitJoin = new DelimitedStringSplitJoin();

    public OCSCommandConverter() {
        this.ocsToCcsCommandMap = new HashMap<>();

        ocsToCcsCommandMap.put(SetFilterCommand.class, CCSSetFilterCommand.class);
        ocsToCcsCommandMap.put(ClearCommand.class, CCSClearCommand.class);
        ocsToCcsCommandMap.put(EndImageCommand.class, CCSEndImageCommand.class);
        ocsToCcsCommandMap.put(InitGuidersCommand.class, CCSInitGuidersCommand.class);
        ocsToCcsCommandMap.put(StartImageCommand.class, CCSStartImageCommand.class);
        ocsToCcsCommandMap.put(TakeImagesCommand.class, CCSTakeImagesCommand.class);
        ocsToCcsCommandMap.put(StartCommand.class, CCSStartCommand.class);
        ocsToCcsCommandMap.put(StandbyCommand.class, CCSStandbyCommand.class);
        ocsToCcsCommandMap.put(DisableCalibrationCommand.class, CCSDisableCalibrationCommand.class);
        ocsToCcsCommandMap.put(DiscardRowsCommand.class, CCSDiscardRowsCommand.class);
        ocsToCcsCommandMap.put(InitImageCommand.class, CCSInitImageCommand.class);
        ocsToCcsCommandMap.put(EnableCalibrationCommand.class, CCSEnableCalibrationCommand.class);
        ocsToCcsCommandMap.put(PlayCommand.class, CCSPlayCommand.class);
        ocsToCcsCommandMap.put(PlaylistCommand.class, CCSDefinePlaylistCommand.class);
        ocsToCcsCommandMap.put(WakeFilterChangerCommand.class, CCSWakeFilterChangerCommand.class);

    }

    /**
     * Convert the given OCS command to the equivalent CCS command
     * @param myOCSCommand The command to convert
     * @return The converted command
     * @throws CommandConversionException If the conversion fails.
     */
    public CCSCommand convert(CameraCommand myOCSCommand) throws CommandConversionException {

        // get ocs command 
        // get its instance 
        // make a copy of the same command with fields appropriately made
        // construct a CCS command with the same fields
        // return a Camera command8
        // get the command to issue - 
        Class<? extends CCSCommand> ccsCommandFromMap = ocsToCcsCommandMap.get(myOCSCommand.getClass());
        if (ccsCommandFromMap == null) {
            throw new CommandConversionException("Unsupported command "+myOCSCommand.toString());
        }

        try {
            // Doesn't this assume a no-args constructor -- yes, should change to use constuctor
            CCSCommand ccsCommand = ccsCommandFromMap.getDeclaredConstructor().newInstance();

            Field[] ocsCommandFields = myOCSCommand.getClass().getDeclaredFields();
            Field[] ccsCommandFields = ccsCommand.getClass().getDeclaredFields();

            for (Field ocsCommandField : ocsCommandFields) {
                if (ocsCommandField.isSynthetic()) {
                    continue;
                }
                ocsCommandField.setAccessible(true);
                boolean matchFound = false;
                for (Field ccsCommandField : ccsCommandFields) {
                    if (ccsCommandField.isSynthetic()) {
                        continue;
                    }
                    ccsCommandField.setAccessible(true);
                    if (ocsCommandField.getName().equals(ccsCommandField.getName())) {
                        if (!ocsCommandField.getType().equals(ccsCommandField.getType())) {
                            // Special case, for string arguments which represent a list
                            if (ocsCommandField.getType() == String.class && ccsCommandField.getType() == String[].class) {
                                ccsCommandField.set(ccsCommand, delimitedStringSplitJoin.split((String) ocsCommandField.get((myOCSCommand))));
                                matchFound = true;
                                break;
                            } else {
                                throw new CommandConversionException("OCSCommandConverter: OCS and CCS Commands contain the same variable names with different types");
                            }
                        } else {
                            ccsCommandField.set(ccsCommand, ocsCommandField.get(myOCSCommand));
                            matchFound = true;
                            break;
                        }
                    }
                }
                if (!matchFound) {
                    throw new CommandConversionException("OCSCommandCoverter : no match found between OCS CCS fields, OCSCommand:  "
                            + myOCSCommand.toString() + " CCSCommand : " + ccsCommand.toString() + " for field " + ocsCommandField.getName());
                }
            }
            return ccsCommand;

        } catch (ReflectiveOperationException ex) {
            throw new CommandConversionException("Error converting command : ", ex);
        }
    }
    
    public static class CommandConversionException extends Exception {
        private CommandConversionException(String msg) {
            super(msg);
        }
        private CommandConversionException(String msg, Throwable t) {
            super(msg, t);
        }
    }

}
