package org.lsst.ccs.subsystem.ocsbridge;

import java.util.Properties;
import java.util.UUID;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.camera.Camera;
import static org.lsst.ccs.camera.Camera.AUXTEL;
import static org.lsst.ccs.camera.Camera.COMCAM;
import static org.lsst.ccs.camera.Camera.MAIN_CAMERA;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.sal.SAL;
import org.lsst.sal.SALCommand;
import org.lsst.sal.SALEvent;
import org.lsst.sal.SALTelemetry;
import org.lsst.sal.atcamera.ATCamera;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.sal.camera.SALCamera;
import org.lsst.sal.cccamera.CCCamera;

/**
 *
 * @author Farrukh Azfar. Configuration class to hold parameters for Camera,
 * AUXTel, ComCam -
 */
@SuppressWarnings("FieldMayBeFinal")
public class OCSBridgeConfig {

    @ConfigurationParameter
    private volatile Camera device = Camera.MAIN_CAMERA;
    
    @ConfigurationParameter
    private final boolean hasFilterChanger = true;
    
    @ConfigurationParameter
    private volatile String mcmName = "mcm";
    
    @ConfigurationParameter(description = "Userid to be used to MCM subsystem", isFinal = true)
    private volatile String userId = "ocs";

    @ConfigurationParameter(description = "Properties file with lfa credentials", isFinal = true)
    private volatile String lfaCredentialsFile = "lfa-test.properties";
        
    private final Properties salProperties;
    
    public OCSBridgeConfig() {
        // Initially empty, filled in createSALProperties
        salProperties = new Properties();
    }
  
    /**
     * Subsystems we should listen to for events
     */
    @ConfigurationParameter(maxLength = 50)
    @SuppressWarnings("VolatileArrayField")
    private volatile String[] listenSubsystems = new String[0];

    public boolean hasFilterChanger() {
        return hasFilterChanger;
    }

    public String getMCMName() {
        return mcmName;
    }

    public Camera getDevice() {
        return device;
    }
    
    public String getLfaCredentialsFile() {
        return lfaCredentialsFile;
    }

    public String[] getListenSubsystems() {
        return listenSubsystems;
    }

    String getUserID() {
        return userId;
    }
    
    private Properties createSALProperties() {
        if (salProperties.isEmpty()) {
            salProperties.put("AGENT_NAME", getAgentName());
            salProperties.put("CAMERA_CSC", getCameraCSC());
            salProperties.put("AGENT_TYPE", "CCS");
            salProperties.put("AGENT_UUID", UUID.randomUUID());      
        }
        return salProperties;
    }
    
    SAL<CameraCommand, CameraEvent, CameraTelemetry> getSALManager() {
        return createSALManager(device, createSALProperties());
    }

    static SAL<CameraCommand, CameraEvent, CameraTelemetry> createSALManager(Camera device, Properties properties) {
        return switch (device) {
            case MAIN_CAMERA -> SALCamera.create(properties);
            case AUXTEL -> ATCamera.create(properties);
            case COMCAM -> CCCamera.create(properties);
            default -> throw new UnsupportedOperationException("Device not yet supported: " + device);
        };
    }
    
    String getCameraCSC() {
        return switch (device) {
            case MAIN_CAMERA -> "MTCamera";
            case AUXTEL -> "ATCamera";
            case COMCAM -> "CCCamera";
            default -> throw new UnsupportedOperationException("Device not yet supported: " + device);
        };
    }

    String getAgentName() {
        return switch (device) {
            case MAIN_CAMERA -> "ocs-bridge";
            case AUXTEL -> "ats-ocs-bridge";
            case COMCAM -> "comcam-ocs-bridge";
            default -> throw new UnsupportedOperationException("Device not yet supported: " + device);
        };
    }
    
    SAL<SALCommand, SALEvent, SALTelemetry> getHeaderServiceManager() {
        return switch (device) {
            case MAIN_CAMERA -> org.lsst.sal.mtheader.SALMain.create(createSALProperties());
            case AUXTEL -> org.lsst.sal.atheader.SALMain.create(createSALProperties());
            case COMCAM -> org.lsst.sal.ccheader.SALMain.create(createSALProperties());
            default -> throw new UnsupportedOperationException("Device not yet supported: " + device);
        };
    }
    
    SAL<SALCommand, SALEvent, SALTelemetry> getMTRotatorManager() {
        return switch (device) {
            case MAIN_CAMERA -> org.lsst.sal.mtrotator.SALMain.create(createSALProperties());
            default -> null;
        };    
    }

    SAL<SALCommand, SALEvent, SALTelemetry> getMTMountManager() {
        return switch (device) {
            case MAIN_CAMERA -> org.lsst.sal.mtmount.SALMain.create(createSALProperties());
            default -> null;
        };    
    }

    SAL<SALCommand, SALEvent, SALTelemetry> getWatcherManager() {
        return org.lsst.sal.watcher.SALMain.create(createSALProperties());  
    }
    
    static OCSBridgeConfig createDefaultConfig(Camera camera) {
        OCSBridgeConfig config = new OCSBridgeConfig();
        config.device = camera;
        return config;
    }

    
    @ConfigurationParameterChanger(propertyName = "listenSubsystems")
    public void setListenSubsystems(String[] listenSubsystems) {
        for ( String sub : listenSubsystems ) {
            if ( sub == null ) {
                throw new RuntimeException("Illegal value for listened subsystem. It cannot be null.");
            } else if ( sub.isEmpty() ) {
                throw new RuntimeException("Illegal value for listened subsystem. It cannot be empty.");                
            } else if ( sub.contains(" ") ) {
                throw new RuntimeException("Illegal value for listened subsystem. It cannot contain blanks: *"+sub+"*" );                
            }
        }
        this.listenSubsystems = listenSubsystems;
    }    
}
