package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;
import org.lsst.ccs.subsystem.imagehandling.data.JsonFile;
import org.lsst.ccs.subsystem.ocsbridge.events.ShutterMotionProfileFitResult;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.utilities.scheduler.Scheduler;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to the main camera shutter subsystem.
 *
 * @author tonyj
 */
class MainCameraShutterSubsystemLayer extends ControlledSubsystem implements ShutterInterface {

    private static final Logger LOG = Logger.getLogger(MainCameraShutterSubsystemLayer.class.getName());
    private final static Map<Enum<org.lsst.ccs.subsystem.shutter.common.PhysicalState>, Enum<ShutterState>> CAMERA_SHUTTER_TO_SHUTTER_STATE = new HashMap<>();
    private volatile CountDownLatch motionDoneCountDown;
    private volatile ImageName currentImageName;

    static final String MOTOR_ENCODER_PATH = "shutter/motorEncoder/";
    static final String HALL_SENSOR_PATH = "shutter/hallSensor/";
    static final String MOTION_PROFILE_FIT_RESULT = "/motionProfileFitResult";
    
    private final Subsystem theMCM;

    static {
        // TODO: Camera has other states. We need to decide how to map them.
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.OPENED, ShutterState.OPEN);
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.OPENING, ShutterState.OPENING);
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.CLOSED, ShutterState.CLOSED);
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.CLOSING, ShutterState.CLOSING);
    }

    MainCameraShutterSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getShutterSubsystemName(), ccs, config);
        this.theMCM = mcm;
    }

    public double getShutterOpenTime(CCSTimeStamp t1, CCSTimeStamp t2) {

        return t1.getTAIDouble() - t2.getTAIDouble();
    }

    @Override
    public void expose(ImageName imageName, Duration exposeTime) throws ExecutionException {
        // We should get two MotionDone publications as a result of the takeExposure command, one for each blade.
        try {
            motionDoneCountDown = new CountDownLatch(2);
            currentImageName = imageName;
            commandSender.sendCommand("takeExposure", exposeTime.toMillis() / 1000.);
            boolean success = motionDoneCountDown.await(exposeTime.toMillis() + 2000, TimeUnit.MILLISECONDS);
            if (!success) {
                throw new TimeoutException("Timed out waiting for shutter exposure to complete");
            }
        } catch (InterruptedException | TimeoutException x) {
            throw new ExecutionException("Error during expose", x);
        }
    }

    @Override
    public void open(ImageName imageName) throws ExecutionException {
        commandSender.sendCommand("openShutter");
    }

    @Override
    public void close() throws ExecutionException {
        commandSender.sendCommand("closeShutter");
    }

    @Override
    public void prepare() {
        // This is a NOOP for the main camera shutter, it is always ready
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        CCSTimeStamp when = statusChange.getStateTransitionTimestamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        String cause = statusChange.getCause();
        changedStates.getComponentStateBundle("statemachine").getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateCameraShutterStateToShutterState(when, value, cause);
        });
    }

    private volatile CompletableFuture<Double> openMotionFuture = null;
    
    @Override
    protected void onEvent(StatusMessage msg) {
        if (msg.getObject() instanceof KeyValueData && "MotionDone".equals(((KeyValueData) msg.getObject()).getKey()) && motionDoneCountDown != null) {
            LOG.log(Level.INFO, "Got Motion Done {0} ", msg);
            int currentCount = (int) motionDoneCountDown.getCount();
            if (currentCount > 0) {
                motionDoneCountDown.countDown();
                MotionDone motionDone = (MotionDone) ((KeyValueData) msg.getObject()).getValue();
                
                boolean isOpen = currentCount == 2;
                CompletableFuture<Double> scheduleMotionFuture = scheduleMotionDoneFit(currentImageName, isOpen, motionDone);
                if (isOpen) {
                    openMotionFuture = scheduleMotionFuture;
                } else {
                    final ImageName imageName = currentImageName;
                    // We can't compute the measured shutter time until both the open and close fits are done
                    openMotionFuture.thenAcceptBoth(scheduleMotionFuture, (openTime, closeTime) -> {
                        MeasuredShutterTime cst = new MeasuredShutterTime(imageName, (closeTime - openTime));
                        sendEvent(MeasuredShutterTime.EVENT_KEY, cst);
                    });
                }
            }
        }
    }

    private void translateCameraShutterStateToShutterState(CCSTimeStamp when, Enum value, String cause) {
        Enum<ShutterState> converted = CAMERA_SHUTTER_TO_SHUTTER_STATE.get(value);
        if (converted != null) {
            LOG.log(Level.INFO, "Got shutter state {0} ", value);
            ccs.getAggregateStatus().add(when, new State(converted, cause));
        }
    }

    @Override
    public void setImageSequence(boolean imageSequence) {
        // We don't care
    }

    /**
     * This method performs a fit to the MotionDone data, and sendEvents for the results.
     * The fit is done asynchronously. The completableFuture will return the midpoint time from 
     * the fit, once the fit is complete, or will complete exceptionally if the fit fails.
     * @param imageName The image being processed
     * @param isOpen True if this is an open operation
     * @param motionDone The MotionDone data to fit
     * @return A CompletableFuture which will contain the midpoint once the fit is complete
     */
    private CompletableFuture<Double> scheduleMotionDoneFit(ImageName imageName, boolean isOpen, MotionDone motionDone) {
        final Scheduler scheduler = theMCM.getScheduler();
        CompletableFuture<ShutterMotionProfileFitter> futureFit = ShutterMotionProfileFitter.fit(scheduler, motionDone);
        // This will run once the fit is complete, and publishes the events and extracts the mid-point time
        return futureFit.thenApply((ShutterMotionProfileFitter fitResult) -> {
            // We also need to send this info to trending (LCOBM-106) and send it to the LFA
            ShutterMotionProfileSender motionProfileSender = new ShutterMotionProfileSender(imageName, motionDone, isOpen, fitResult);
            JsonFile jsonProfile = motionProfileSender.getJsonFile();
            sendEvent(AdditionalFile.EVENT_KEY, jsonProfile);

            String pathPrefix = motionDone.side().toString() + "/" + (isOpen ? "open" : "close") + MOTION_PROFILE_FIT_RESULT;
            
            ShutterMotionProfileFitResult encoderFitResult = motionProfileSender.getEncoderFitResult();
            String encoderPath = MOTOR_ENCODER_PATH + pathPrefix;
            sendEvent(encoderPath, encoderFitResult);

            ShutterMotionProfileFitResult hallSensorFitResult = motionProfileSender.getHallSensorFitResult();
            String hallSensorPath = HALL_SENSOR_PATH + pathPrefix;
            sendEvent(hallSensorPath, hallSensorFitResult);

            final double timeAtMidPoint = motionProfileSender.getStartTime().getTAIDouble() + motionProfileSender.getMidPointTimeFromHallSensor() + hallSensorFitResult.getModelStartTime();
            return timeAtMidPoint;
        });
    }
}