package org.lsst.ccs.subsystem.ocsbridge;

import java.io.Serializable;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.lsst.ccs.imagenaming.ImageName;

/**
 * Base class for all CCSCommands. This class is abstract and is extended by
 * specific implementations for each supported command. Note that this class
 * only encapsulates the command and its arguments, it does not include any code
 * for actually executing the command.
 *
 * @author tonyj
 */
public abstract class CCSCommand {

    /**
     * Gets the command name.
     *
     * @return The command name.
     */
    abstract String getCommand();

    /**
     * Gets the command arguments.
     *
     * @return The command arguments.
     */
    abstract List<Object> getArguments();

    @Override
    public String toString() {
        return "CCSCommand: " + getCommand() + " " + getArguments().stream().map(Object::toString).collect(Collectors.joining(" "));
    }

    public static class CCSInitImageCommand extends CCSCommand {

        private final double deltaT;

        public CCSInitImageCommand() {
            this.deltaT = 0;
        }

        public CCSInitImageCommand(double deltaT) {
            this.deltaT = deltaT;
        }

        public double getDeltaT() {
            return deltaT;
        }

        @Override
        String getCommand() {
            return "initImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(deltaT);
        }

    }

    public static class CCSTakeImagesCommand extends CCSCommand {

        private final int numImages;
        private final double expTime;
        private final boolean shutter;
        private final String sensors;
        private final String obsNote;
        private final String keyValueMap;

        public CCSTakeImagesCommand() {
            this.numImages = 1;
            this.expTime = 1;
            this.shutter = false;
            this.sensors = "";
            this.obsNote = "";
            this.keyValueMap = "";
        }

        public CCSTakeImagesCommand(double expTime, int numImages, boolean shutter, String sensors, String keyValueMap, String obsNote) {
            this.numImages = numImages;
            this.expTime = expTime;
            this.shutter = shutter;
            this.sensors = sensors;
            this.keyValueMap = keyValueMap;
            this.obsNote = obsNote;
        }

        public int getNumImages() {
            return numImages;
        }

        public double getExpTime() {
            return expTime;
        }

        public boolean isShutter() {
            return shutter;
        }

        public String getSensors() {
            return sensors;
        }

        public String getObsNote() {
            return obsNote;
        }

        public String getKeyValueMap() {
            return keyValueMap;
        }

        @Override
        String getCommand() {
            return "takeImages";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(expTime, numImages, shutter, sensors, keyValueMap, obsNote);
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("CCSTakeImagesCommand{numImages=").append(numImages);
            sb.append(", expTime=").append(expTime);
            sb.append(", shutter=").append(shutter);
            sb.append(", sensors=").append(sensors);
            sb.append(", obsNote=").append(obsNote);
            sb.append(", keyValueMap=").append(keyValueMap);
            sb.append('}');
            return sb.toString();
        }
    }

    public static class CCSSetFilterCommand extends CCSCommand {

        private final String name;

        public CCSSetFilterCommand() {
            this.name = "Empty Constructor Default";
        }

        public CCSSetFilterCommand(String filterName) {
            this.name = filterName;
        }

        public String getFilterName() {
            return name;
        }

        @Override
        String getCommand() {
            return "setFilter";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(name);
        }
    }

    public static class CCSWakeFilterChangerCommand extends CCSCommand {

        private final int mode;

        public CCSWakeFilterChangerCommand() {
            this.mode = -1;
        }

        public CCSWakeFilterChangerCommand(int mode) {
            this.mode = mode;
        }

        public int getMode() {
            return mode;
        }

        @Override
        String getCommand() {
            return "wakeFilterChanger";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(mode);
        }
    }

    public static class CCSInitGuidersCommand extends CCSCommand {

        private final String roiSpec;

        /**
         *
         */
        public CCSInitGuidersCommand() {
            this.roiSpec = "will be overwritten";
        }

        public CCSInitGuidersCommand(String roiSpec) {
            this.roiSpec = roiSpec;
        }

        public String getRoiSpec() {
            return roiSpec;
        }

        @Override
        String getCommand() {
            return "initGuiders";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(roiSpec);
        }
    }

    public static class CCSStartImageCommand extends CCSCommand {

        private final boolean shutter;
        private final double timeout;
        private final String sensors;
        private final String keyValueMap;
        private final String obsNote;

        public CCSStartImageCommand() {
            this.shutter = false;
            this.sensors = "";
            this.keyValueMap = "";
            this.obsNote = "";
            this.timeout = 1.;
        }

        public CCSStartImageCommand(boolean shutter, String sensors, String keyValueMap, String obsNote, double timeout) {
            this.shutter = shutter;
            this.sensors = sensors;
            this.keyValueMap = keyValueMap;
            this.obsNote = obsNote;
            this.timeout = timeout;
        }

        public boolean isShutterOpen() {
            return shutter;
        }

        public String getSensors() {
            return sensors;
        }

        public String getKeyValueMap() {
            return keyValueMap;
        }

        public String getObsNote() {
            return obsNote;
        }

        public double getTimeout() {
            return timeout;
        }

        @Override
        String getCommand() {
            return "startImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(shutter, sensors, keyValueMap, obsNote, timeout);
        }

        @Override
        public String toString() {
            return "CCSStartImageCommand{" + "shutter=" + shutter + ", timeout=" + timeout + ", sensors=" + sensors + ", keyValueMap=" + keyValueMap + ", annotation=" + obsNote + '}';
        }
    }

    public static class CCSStartCommand extends CCSCommand {

        private final String configurationOverride;

        public CCSStartCommand() {

            this.configurationOverride = "will be oerwritten";
        }

        public CCSStartCommand(String configurationOverride) {
            this.configurationOverride = configurationOverride;
        }

        public String getConfiguration() {
            return configurationOverride;
        }

        @Override
        public String toString() {
            return "CCSStartCommand{" + "configuration=" + configurationOverride + '}';
        }

        @Override
        String getCommand() {
            return "start";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(configurationOverride);
        }
    }

    static class NoArgCCSCommand extends CCSCommand {

        private String commandName;

        NoArgCCSCommand(String commandName) {
            this.commandName = commandName;
        }

        @Override
        String getCommand() {
            return commandName;
        }

        @Override
        List<Object> getArguments() {
            return Collections.emptyList();
        }
    }

    public static class CCSStandbyCommand extends NoArgCCSCommand {

        public CCSStandbyCommand() {
            super("standby");
        }
    }

    public static class CCSEnableCalibrationCommand extends NoArgCCSCommand {

        public CCSEnableCalibrationCommand() {
            super("enableCalibration");
        }

    }

    public static class CCSDisableCalibrationCommand extends NoArgCCSCommand {

        public CCSDisableCalibrationCommand() {
            super("disableCalibration");
        }

    }

    public static class CCSEndImageCommand extends NoArgCCSCommand {

        public CCSEndImageCommand() {
            super("endImage");
        }

    }

    public static class CCSSetAvailableCommand extends NoArgCCSCommand {

        public CCSSetAvailableCommand() {
            super("setAvailable");
        }
    }

    public static class CCSRevokeAvailableCommand extends NoArgCCSCommand {

        public CCSRevokeAvailableCommand() {
            super("revokeAvailable");
        }
    }

    public static class CCSSimulateFaultCommand extends NoArgCCSCommand {

        public CCSSimulateFaultCommand() {
            super("simulateFault");
        }
    }

    public static class CCSClearFaultCommand extends NoArgCCSCommand {

        public CCSClearFaultCommand() {
            super("clearFault");
        }
    }

    public static class CCSLockMotionCommand extends NoArgCCSCommand {

        public CCSLockMotionCommand() {
            super("lockMotion");
        }
    }
    
    public static class CCSUnlockMotionCommand extends NoArgCCSCommand {

        public CCSUnlockMotionCommand() {
            super("unlockMotion");
        }
    }
    
    public static class CCSAllocateImageNameCommand extends NoArgCCSCommand {

        public CCSAllocateImageNameCommand() {
            super("allocateImageName");
        }
    }

    public static class CCSSetHeaderKeywordsCommand extends CCSCommand {

        private final Map<String, Serializable> headersMap;

        public CCSSetHeaderKeywordsCommand(Map<String, Serializable> headersMap) {
            this.headersMap = headersMap;
        }

        public Map<String, Serializable> getHeadersMap() {
            return headersMap;
        }

        @Override
        String getCommand() {
            return "setHeaderkeywords";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(headersMap);
        }
    }

    public static class CCSClearAndStartNamedIntegrationCommand extends CCSCommand {

        private final ImageName imageName;
        private final boolean shutter;
        private final int clears;
        private final String annotation;
        private final Set sensors;
        private final Map<String, Serializable> headersMap;

        public CCSClearAndStartNamedIntegrationCommand(ImageName imageName, boolean shutter, int clears, String annotation, Set sensors, Map<String, Serializable> headersMap) {
            this.imageName = imageName;
            this.shutter = shutter;
            this.clears = clears;
            this.annotation = annotation;
            this.sensors = sensors;
            this.headersMap = headersMap;
        }

        public ImageName getImageName() {
            return imageName;
        }

        public boolean isShutter() {
            return shutter;
        }

        public int getClears() {
            return clears;
        }

        public String getAnnotation() {
            return annotation;
        }

        public Set getSensors() {
            return sensors;
        }

        public Map<String, Serializable> getHeadersMap() {
            return headersMap;
        }

        @Override
        String getCommand() {
            return "clearAndStartNamedIntegration";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(imageName, shutter, clears, annotation, sensors, headersMap);
        }
    }

    public static class CCSTakeImageCommand extends CCSCommand {

        private final ImageName imageName;
        private final boolean shutter;
        private final double expTime;
        private final int clears;
        private final String annotation;
        private final Set sensors;
        private final Map<String, Serializable> headersMap;

        public CCSTakeImageCommand(ImageName imageName, boolean shutter, double expTime, int clears, String annotation, Set sensors, Map<String, Serializable> headersMap) {
            this.imageName = imageName;
            this.shutter = shutter;
            this.expTime = expTime;
            this.clears = clears;
            this.annotation = annotation;
            this.sensors = sensors;
            this.headersMap = headersMap;
        }

        public ImageName getImageName() {
            return imageName;
        }

        public boolean isShutter() {
            return shutter;
        }

        public double getExpTime() {
            return expTime;
        }

        public int getClears() {
            return clears;
        }

        public String getAnnotation() {
            return annotation;
        }

        public Set getSensors() {
            return sensors;
        }

        public Map<String, Serializable> getHeadersMap() {
            return headersMap;
        }

        @Override
        String getCommand() {
            return "takeImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(imageName, shutter, expTime, clears, annotation, sensors, headersMap);
        }
    }

    public static class CCSEndIntegrationCommand extends NoArgCCSCommand {

        public CCSEndIntegrationCommand() {
            super("endintegration");
        }
    }

    public static class CCSWaitForImageCommand extends NoArgCCSCommand {

        public CCSWaitForImageCommand() {
            super("waitForImage");
        }
    }

    public static class CCSOpenShutterCommand extends NoArgCCSCommand {

        public CCSOpenShutterCommand() {
            super("openShutter");
        }
    }

    public static class CCSResetFCSPLCCommand extends NoArgCCSCommand {

        public CCSResetFCSPLCCommand() {
            super("resetFESPLC");
        }
    }
    
    public static class CCSGetAvailableFiltersCommand extends NoArgCCSCommand {

        public CCSGetAvailableFiltersCommand() {
            super("getAvailableFilters");
        }
    }

    public static class CCSGetCurrentFilterCommand extends NoArgCCSCommand {

        public CCSGetCurrentFilterCommand() {
            super("getCurrentFilter");
        }
    }

    public static class CCSCloseShutterCommand extends NoArgCCSCommand {

        public CCSCloseShutterCommand() {
            super("closeShutter");
        }
    }

    public static class CCSDiscardRowsCommand extends CCSCommand {

        private final int nRows;

        public CCSDiscardRowsCommand() {

            this.nRows = 0;
        }

        public CCSDiscardRowsCommand(int nRows) {
            this.nRows = nRows;
        }

        public int getnRows() {
            return nRows;
        }

        @Override
        String getCommand() {
            return "discardRows";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(nRows);
        }
    }

    public static class CCSClearCommand extends CCSCommand {

        private final int nClears;

        public CCSClearCommand() {
            this.nClears = 0;
        }

        public CCSClearCommand(int nClears) {
            this.nClears = nClears;
        }

        public int getNClears() {
            return nClears;
        }

        @Override
        String getCommand() {
            return "clear";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(nClears);
        }

    }

    public static class CCSPlayCommand extends CCSCommand {

        private final String playlist;
        private final boolean repeat;

        public CCSPlayCommand() {
            this.playlist = "";
            this.repeat = false;
        }

        public CCSPlayCommand(String playlist, boolean repeat) {
            this.playlist = playlist;
            this.repeat = repeat;
        }

        public String getPlaylist() {
            return playlist;
        }

        public boolean isRepeat() {
            return repeat;
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(playlist, repeat);
        }

        @Override
        String getCommand() {
            return "play";
        }
    }

    public static class CCSDefinePlaylistCommand extends CCSCommand {

        private final String playlist;
        private final String folder;
        private final String[] images;

        public CCSDefinePlaylistCommand() {
            this.playlist = "";
            this.folder = "";
            this.images = new String[0];
        }

        public CCSDefinePlaylistCommand(String playlist, String folder, String... images) {
            this.playlist = playlist;
            this.folder = folder;
            this.images = images;
        }

        public String getPlaylist() {
            return playlist;
        }

        public String getFolder() {
            return folder;
        }

        public String[] getImages() {
            return images;
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(playlist, folder, images);
        }

        @Override
        String getCommand() {
            return "definePlaylist";
        }

    }

    public static class CCSCommandResponse<T extends Object> {

        private final CCSExecutorWithReturn<T> executor;

        public CCSCommandResponse(CCSExecutorWithReturn<T> executor) {
            this.executor = executor;
        }

        public CCSAckOrNack waitForAckOrNack() {
            try {
                Duration duration = executor.testPreconditions();
                return new CCSAckOrNack(duration);
            } catch (CCSPreconditionsNotMet ex) {
                return new CCSAckOrNack(ex.getReason());
            }
        }

        public T waitForCompletion() throws Exception {
            return executor.executeAndReturn();
        }

    }

    public static class CCSAckOrNack {

        private final boolean ack;
        private final Duration duration;
        private final String reason;

        private CCSAckOrNack(Duration duration) {
            this.ack = true;
            this.reason = null;
            this.duration = duration;
        }

        CCSAckOrNack(String reason) {
            this.ack = false;
            this.reason = reason;
            this.duration = null;
        }

        public boolean isNack() {
            return !ack;
        }

        public String getReason() {
            return reason;
        }

        public Duration getDuration() {
            return duration;
        }

    }

    public static class CCSPreconditionsNotMet extends Exception {

        private static final long serialVersionUID = 7463370916615596884L;

        private final String reason;

        public CCSPreconditionsNotMet(String reason) {
            super("Reason : " + reason);
            this.reason = reason;
        }

        private String getReason() {
            return reason;
        }
    }
    
    
    public static class CCSCreateTestAlertCommand extends CCSCommand {

        private final String message;
        private final String level;
        private final boolean cleared;

        CCSCreateTestAlertCommand(String message, String level, boolean cleared) {
            this.message = message;
            this.level = level;
            this.cleared = cleared;
        }

        public String getMessage() {
            return message;
        }

        public String getLevel() {
            return level;
        }

        @Override
        String getCommand() {
            return "createTestAlert";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(message, level);
        }

        boolean isCleared() {
            return cleared;
        }
    }
}
