package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.concurrent.ExecutionException;

/**
 * An interface to be implemented by all filter changers, real or simulated
 *
 * @author tonyj
 */
public interface FilterChangerInterface {

    public enum FilterType {
        U, G, R, I, Z, Y, NONE, OTHER,
        PH("other:pinhole"),
        EF("other:emptyframe");

        private final String name;

        private FilterType(String name) {
            this.name = name;
        }

        private FilterType() {
            this.name = this.name().toLowerCase();
        }
        
        public String getName() {
            return this.name;
        }
    };

    Map<String, FilterType> allFilterTypes = new HashMap<String, FilterType>() {
        {
            // Main camera filters
            put("ef_43", FilterType.EF);
            put("ph_5", FilterType.PH);
            put("u_24", FilterType.U);
            put("g_6", FilterType.G);
            put("r_57", FilterType.R);
            put("i_39", FilterType.I);
            put("z_20", FilterType.Z);
            put("y_10", FilterType.Y);
            put("NONE", FilterType.NONE);

            // ComCam filters
            put("u_05", FilterType.U);
            put("g_07", FilterType.G);
            put("g_01", FilterType.G);
            put("r_03", FilterType.R);
            put("i_06", FilterType.I);
            put("z_03", FilterType.Z);
            put("z_02", FilterType.Z);
            put("y_04", FilterType.Y);
            put("pinhole", FilterType.PH);
        }
    };

    /**
     * Get the filter type as string associated to the given filter 
     * @param filterName
     * @return 
     */
    default String getFilterType(String filterName) {
        return allFilterTypes.get(filterName).getName();
    };

    /**
     * Get a list of available filters, containing the (unique) name of each filter
     * @return 
     * @throws java.util.concurrent.ExecutionException 
     */
    List<String> getAvailableFilters() throws ExecutionException;

    /**
     * Get a list of currently installed filters
     * @return 
     * @throws java.util.concurrent.ExecutionException
     */
    List<String> getInstalledFilters() throws ExecutionException;

    /**
     * Start and lock the filter changer, with the specified configuration
     * @param configName 
     * @throws java.util.concurrent.ExecutionException 
     */
    void start(String configName) throws ExecutionException;

    /**
     * Get the currently installed filter name
     * @return The installed filter, or ? if none installed
     * @throws java.util.concurrent.ExecutionException
     */
    String getCurrentFilter() throws ExecutionException;

    /**
     * Set the currently installed filter 
     * @param filterName 
     * @throws java.util.concurrent.ExecutionException  if the command fails
     */
    void setFilter(String filterName) throws ExecutionException;

    /**
     * Get the expected max angle from vertical position
     * to switch to a given filter
     * @return the angle in degrees
     * @throws java.util.concurrent.ExecutionException 
     */
    default double getMaxAngleForFilterChange() throws ExecutionException {
        return 90;
    };

    /**
     * Get the expected max angle from vertical position
     * to switch to a given filter in optimal conditions
     * @return the angle in degrees
     * @throws java.util.concurrent.ExecutionException 
     */
    default double getMaxAngleForFastFilterChange() throws ExecutionException {
        return 90;
    };

    /**
     * Get the expected max duration to switch to a given filter
     * in bad conditions (above a certain angle)
     * @param filterName
     * @return The max time in seconds
     * @throws java.util.concurrent.ExecutionException 
     */
    default Duration getDurationForSlowFilterChange(String filterName) throws ExecutionException {
        return Duration.ofSeconds(240);
    };

    /**
     * Get the expected max duration to switch to a given filter
     * in best conditions (under a certain angle)
     * @param filterName
     * @return The max time in seconds
     * @throws java.util.concurrent.ExecutionException 
     */
    default Duration getDurationForFastFilterChange(String filterName) throws ExecutionException {
        return Duration.ofSeconds(240);
    };

    /**
     * Wake up the carousel of the filter exchange system from power save mode
     * @param mode
     * @throws java.util.concurrent.ExecutionException  if the command fails
     */
    void wakeFilterChanger(int mode) throws ExecutionException;

    /**
     * Get the expected max duration of filter exchange wake up
     * @param mode
     * @return The max time in seconds
     * @throws java.util.concurrent.ExecutionException 
     */
    default Duration getDurationForWakeUp(int mode) throws ExecutionException {
        return Duration.ofSeconds(10);
    };
}
