package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSFilterSwapEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.states.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.states.FcsPowerState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystems.fcs.states.AutochangerInclinationState;
import org.lsst.ccs.subsystems.fcs.states.AutochangerTrucksState;
import org.lsst.ccs.subsystems.fcs.states.CarouselPowerState;
import org.lsst.ccs.subsystems.fcs.states.FcsState;
import org.lsst.ccs.subsystems.fcs.states.ObservatoryFilterState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to the filter changer subsystem.
 *
 * @authors tonyj aubourg and aboucaud
 */
public class MainCameraFilterChangerSubsystemLayer extends ControlledSubsystem implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(ComCamFilterChangerSubsystemLayer.class.getName());
    private final static Map<ObservatoryFilterState, FilterState> FCS_TO_FILTER_STATE = new HashMap<>();
    private final static Map<CarouselPowerState, FcsPowerState> CAROUSEL_TO_FCS_POWER_STATE = new HashMap<>();
    private final AtomicBoolean isFirstFcsPublication = new AtomicBoolean(true);
    private volatile CCSSetFilterEvent ccsSetFilterEvent;
    private volatile CCSFilterSwapEvent ccsFilterSwapEvent;
    private String previousFilter = "";

    static {
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.LOADING, FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.UNLOADING, FilterState.UNLOADING);
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.UNLOADED, FilterState.UNLOADED);
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.LOADED, FilterState.LOADED);
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.ROTATING, FilterState.ROTATING);
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.SWAPPING, FilterState.SWAPPING);
        FCS_TO_FILTER_STATE.put(ObservatoryFilterState.UNKNOWN, FilterState.UNKNOWN);

        CAROUSEL_TO_FCS_POWER_STATE.put(CarouselPowerState.LOW_POWER, FcsPowerState.LOW_POWER);
        CAROUSEL_TO_FCS_POWER_STATE.put(CarouselPowerState.REGULAR, FcsPowerState.REGULAR);
    }

    private static class FilterStateConverter {

        private FilterState convertStateInstantaneous(ObservatoryFilterState state) {
            FilterState s = FCS_TO_FILTER_STATE.get(state);
            if (s == null) {
                s = FilterState.LOADED;
            }
            return s;
        }

        FilterState convertState(ObservatoryFilterState state) {
            FilterState s = convertStateInstantaneous(state);
            return s;
        }
    }

    private final static FilterStateConverter converter = new FilterStateConverter();

    @SuppressWarnings({"rawtypes", "unchecked"})
    private void translateFcsStateToFilterState(CCSTimeStamp when, ObservatoryFilterState value, String cause) {
        if (value == null) {
            value = ObservatoryFilterState.LOADED;
        }
        FilterState converted = converter.convertState(value);
        LOG.log(Level.INFO, "FilterState was converted to {0} ", converted);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted, cause));
        }
    }

    private static class PowerStateConverter {

        private FcsPowerState convertStateInstantaneous(CarouselPowerState state) {
            FcsPowerState s = CAROUSEL_TO_FCS_POWER_STATE.get(state);
            if (s == null) {
                s = FcsPowerState.LOW_POWER;
            }
            return s;
        }

        FcsPowerState convertState(CarouselPowerState state) {
            FcsPowerState s = convertStateInstantaneous(state);
            return s;
        }
    }

    private final static PowerStateConverter powerConverter = new PowerStateConverter();

    @SuppressWarnings({ "unchecked", "rawtypes" })
    private void translateCarouselPowerStateToFcsPowerState(CCSTimeStamp when, CarouselPowerState value, String cause) {
        if (value == null) {
            value = CarouselPowerState.LOW_POWER;
        }
        FcsPowerState converted = powerConverter.convertState(value);
        LOG.log(Level.INFO, "CarouselPowerState was converted to {0} ", converted);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted, cause));
        }
    }

    public MainCameraFilterChangerSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFilterChangerSubsystemName(), ccs, config);
    }

    private void waitUntilMoveComplete(FilterState finalState, Duration estimatedDurationForFilterChange) throws ExecutionException {
        Future<Void> waitForStatus = ccs.waitForStatus(finalState);
        try {
            waitForStatus.get(estimatedDurationForFilterChange.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException | TimeoutException ex) {
            throw new ExecutionException("Timeout waiting for filter change to complete", ex);
        }
    }

    @Override
    public void setFilter(String filter) throws ExecutionException {
        // This value is used by onEvent to filter events published by the FCS
        previousFilter = getCurrentFilter();
        // The following test avoids creating setFilter events if the requested filter is already set 
        // This cannot be handled by the CCSCommandPreconditions since the filter currently set is not known by the MCM
        if ( previousFilter.equals(filter) ) {
            return;
        }
        /* Trigger a StartSetFilterEvent */
        ccsSetFilterEvent = new CCSSetFilterEvent(filter, getFilterType(filter));
        ccs.fireEvent(ccsSetFilterEvent);
        LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
        
        Duration timeout = getDurationForSlowFilterChange(filter);

        commandSender.sendCommand(Void.TYPE, timeout, "setFilterByName", filter);
        waitUntilMoveComplete(FilterState.LOADED, timeout);
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<String> getAvailableFilters() throws ExecutionException {
        return commandSender.sendCommand(List.class, "getAvailableFilters");
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<String> getInstalledFilters() throws ExecutionException {
        return commandSender.sendCommand(List.class, "getInstalledFilters");
    }

    @Override
    public String getCurrentFilter() throws ExecutionException {
        return commandSender.sendCommand(String.class, "getOnlineFilterName");
    }

    @Override
    public double getMaxAngleForFilterChange() throws ExecutionException {
        return commandSender.sendCommand(Double.class, "getMaxAngleForFilterChange");
    }

    @Override
    public double getMaxAngleForFastFilterChange() throws ExecutionException {
        return commandSender.sendCommand(Double.class, "getMaxAngleForFastFilterChange");
    }

    @Override
    public Duration getDurationForSlowFilterChange(String filterName) throws ExecutionException {
        return commandSender.sendCommand(Duration.class, "getDurationForSlowFilterChange", filterName);
    }

    @Override
    public Duration getDurationForFastFilterChange(String filterName) throws ExecutionException {
        return commandSender.sendCommand(Duration.class, "getDurationForFastFilterChange", filterName);
    }

    @Override
    public void wakeFilterChanger(int mode) throws ExecutionException {
        commandSender.sendCommand(Void.TYPE, "wakeFilterChanger", mode);
    }

    @Override
    public Duration getDurationForWakeUp(int mode) throws ExecutionException {
        return commandSender.sendCommand(Duration.class, "getDurationForWakeUp", mode);
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Filter changer connected");
        FcsState state = initialState.getState(FcsState.class);
        LOG.log(Level.INFO, "Got initial FCS FcsState {0}", state);
        ObservatoryFilterState filterState = initialState.getState(ObservatoryFilterState.class);
        LOG.log(Level.INFO, "Got initial FCS ObservatoryFilterState {0}", filterState);
        AutochangerInclinationState inclinationState = initialState.getState(AutochangerInclinationState.class);
        LOG.log(Level.INFO, "Got initial FCS AutochangerInclinationState {0}", inclinationState);
        translateFcsStateToFilterState(CCSTimeStamp.currentTime(), filterState, null);
        CarouselPowerState powerState = initialState.getState(CarouselPowerState.class);
        LOG.log(Level.INFO, "Got initial FCS CarouselPowerState {0}", powerState);
        translateCarouselPowerStateToFcsPowerState(CCSTimeStamp.currentTime(), powerState, null);
    }

    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Filter changer disconnected");
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        CCSTimeStamp when = statusChange.getStateTransitionTimestamp();
        StateBundle changedStates = newStates.diffState(oldStates);
        String cause = statusChange.getCause();
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue())
                .forEachOrdered((value) -> {
                    if (value instanceof ObservatoryFilterState) {
                        LOG.log(Level.INFO, "Got ObservatoryFilterState {0}", value);
                        translateFcsStateToFilterState(when, (ObservatoryFilterState) value, cause);
                    } else if (value instanceof AutochangerInclinationState) {
                        LOG.log(Level.INFO, "Got AutochangerInclinationState {0}", value);
                    } else if (value instanceof CarouselPowerState) {
                        LOG.log(Level.INFO, "Got CarouselPowerState {0}", value);
                        translateCarouselPowerStateToFcsPowerState(when, (CarouselPowerState) value, cause);
                    }
                });
    }

    @Override
    protected void onEvent(@SuppressWarnings("rawtypes") StatusMessage msg) {
        Object data = msg.getObject();
        
        if ( data instanceof KeyValueData ) {
            final KeyValueData sentData = (KeyValueData) data;
            String dataKey = sentData.getKey();
            LOG.log(Level.FINE, "Got kvd {0} {1}", new Object[]{dataKey, sentData.getValue()});

            switch ( dataKey ) {
                // Filter exchange system data
                case "fcs/mcm" -> {
                    processFcsKvdl((KeyValueDataList) sentData.getValue());
                }
                // Filter loader swap data
                case "loader/mcm" -> {
                    processLoaderKvdl(msg.getCCSTimeStamp(), (KeyValueDataList) sentData.getValue());
                }
                default -> { return; }
            }
        }
    }

    private void processFcsKvdl(KeyValueDataList kvdl) {
        String filterName = null;
        String filterType = null;
        int slot = 0;
        int position = 0;
        double proximity = 0;
        AutochangerTrucksState location = AutochangerTrucksState.ONLINE;

        for ( KeyValueData kvd : kvdl ) {
            Object value = kvd.getValue();
            switch ( kvd.getKey() ) {
                case "filter_on_autochanger" -> {
                    filterName = value.toString();
                    filterType = getFilterType(filterName);
                }
                case "filter_previous_socketID" -> {
                    slot = ((Number) value).intValue();
                }
                case "autochanger_trucks_position" -> {
                    position = ((Number) value).intValue();
                }
                case "autochanger_trucks_state" -> {
                    location = AutochangerTrucksState.valueOf(value.toString());
                }
                case "proximity" -> {
                    /* More precise than the autochanger trucks position close to the ONLINE position */
                    proximity = ((Number) value).doubleValue();
                }
            }
        }
        /* Only publish info on the buses if this corresponds to the new filter at the end of the setFilter */
        boolean endOfExchange = location == AutochangerTrucksState.ONLINE;
        boolean isSameFilter = this.previousFilter.equals(filterName);

        if ( isFirstFcsPublication.getAndSet(false) ) {
            ccsSetFilterEvent = new CCSSetFilterEvent(filterName, filterType, slot, proximity == 0 ? position : proximity);
            ccs.fireEvent(ccsSetFilterEvent);
            LOG.log(Level.INFO, "First publication from FCS: {0}", ccsSetFilterEvent);
        } else if ( endOfExchange && !isSameFilter ) {
            ccsSetFilterEvent = new CCSSetFilterEvent(filterName, filterType, slot, proximity == 0 ? position : proximity);
            ccs.fireEvent(ccsSetFilterEvent);
            LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
        } else {
            return;
        }
    }

    private void processLoaderKvdl(CCSTimeStamp when, KeyValueDataList kvdl) {
        String filterName = null;
        String filterType = null;
        String loaderId = null;
        String action = null;

        for ( KeyValueData kvd : kvdl ) {
            Object value = kvd.getValue();
            switch ( kvd.getKey() ) {
                case "filter_name" -> {
                    filterName = value.toString();
                    filterType = getFilterType(filterName);
                }
                case "loader_id" -> {
                    loaderId = value.toString();
                }
                case "action" -> {
                    action = value.toString();
                }
            }
        }

        ccsFilterSwapEvent = new CCSFilterSwapEvent(when.getTAIDouble(), loaderId, action, filterName, filterType);
        ccs.fireEvent(ccsFilterSwapEvent);
        LOG.log(Level.INFO, "Sent: {0}", ccsFilterSwapEvent);
    }
    
    @Override
    protected void onRepublishServiceData() {
        if (ccsSetFilterEvent != null) {
            ccs.fireEvent(ccsSetFilterEvent);
            LOG.log(Level.INFO, "Resent: {0}", ccsSetFilterEvent);
        }
    }

}
