package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.subsystem.comcam.filterchanger.data.EndSetFilterData;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.states.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class FilterChangerSimulation implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(FilterChangerSimulation.class.getName());

    // This should all become (simulation only) configuration
    private static final Map<String, FilterType> mainCameraFilters = new LinkedHashMap<>();

    static {
        // These are filter changers installed for EO test on level 3, October 2024
        mainCameraFilters.put("NONE", FilterType.NONE);
        mainCameraFilters.put("ph_5", FilterType.OTHER);
        mainCameraFilters.put("y_10", FilterType.Y);
        mainCameraFilters.put("r_57", FilterType.R);
        mainCameraFilters.put("ef_43", FilterType.OTHER);
        mainCameraFilters.put("g_6", FilterType.G);
    }

    private static final List<String> mainCameraInstalledFilters = new ArrayList<>(mainCameraFilters.keySet());

//    static {
//        mainCameraInstalledFilters.remove("y-099");
//    }

    private static final Map<String, FilterType> comCamFilters = new LinkedHashMap<>();

    static {
        // These are taken from comcam-fcs configuration
        comCamFilters.put("u_05", FilterType.U);
        comCamFilters.put("g_07", FilterType.G);
        comCamFilters.put("g_01", FilterType.G);
        comCamFilters.put("r_03", FilterType.R);
        comCamFilters.put("i_06", FilterType.I);
        comCamFilters.put("z_03", FilterType.Z);
        comCamFilters.put("z_02", FilterType.Z);
        comCamFilters.put("y_04", FilterType.Y);
        comCamFilters.put("pinhole", FilterType.OTHER);
        comCamFilters.put("NONE", FilterType.NONE);
    }

    private static final List<String> comCamInstalledFilters = new ArrayList<>();
    static {
        comCamInstalledFilters.add("i_06");
        comCamInstalledFilters.add("r_03");
        comCamInstalledFilters.add("g_07");
    }

    static final Duration LOAD_TIME = Duration.ofMillis(15000);
    static final Duration ROTATION_TIME_PER_DEGREE = Duration.ofMillis(100);
    static final Duration UNLOAD_TIME = Duration.ofMillis(15000);

    private final Map<String, FilterType> allFilters;
    private final List<String> installedFilters;

    private final MCMConfig config;
    private String currentFilter;
    private int currentRotationPosition = 0;
    private final CCS ccs;
    private final State filterState;

    /* Last published FCSState and lock */
    private volatile KeyValueDataList lastPublishedFCStateData = null;
    private final Object lastPublishedFCStateDataLock = new Object();

    private Agent agent;
    
    FilterChangerSimulation(CCS ccs, MCMConfig config) {

        this.ccs = ccs;
        this.config = config;
        switch (config.getCameraType()) {
            case MAIN_CAMERA:
                allFilters = mainCameraFilters;
                currentFilter = "NONE";
                installedFilters = mainCameraInstalledFilters;
                break;
            case COMCAM:
                allFilters = comCamFilters;
                installedFilters = comCamInstalledFilters;
                currentFilter = comCamInstalledFilters.get(0);
                break;
            default:
                throw new RuntimeException("Filter changer not supported for " + config.getCameraType());
        }
        filterState = new State(FilterState.UNLOADED);
        ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), filterState);
    }

    @Override
    public List<String> getAvailableFilters() {
        return allFilters.entrySet().stream()
            .map(entry -> entry.getKey())
            .collect(Collectors.toList());
    }

    @Override
    public List<String> getInstalledFilters() {
        return Collections.unmodifiableList(installedFilters);
    }

    @Override
    public String getFilterType(String filterName) {
        return allFilters.get(filterName).getName();
    }

    @Override
    public void start(String configName) {
        /// No Op
    }

    @Override
    public String getCurrentFilter() {
        return currentFilter;
    }

    @Override
    public void setFilter(String filterName) {
        try {
            int position = getInstalledFilters().indexOf(filterName);
            if (position < 0) {
                throw new IllegalArgumentException("Invalid or uninstalled filter: " + filterName);
            }
            FilterType requestedFilterType = allFilters.get(filterName);
            ccs.fireEvent(new CCSSetFilterEvent(filterName, requestedFilterType.getName()));

            int targetRotation = -1;
            if (filterName.equals(currentFilter)) {
                // No-op? -- actually comcam-fcs rejects this, maybe we should too?
            } else if (requestedFilterType == FilterType.NONE) {
                filterState.setState(FilterState.UNLOADING);
                Future<Void> waitForUnloaded = ccs.waitForStatus(FilterState.UNLOADED);
                ccs.schedule(UNLOAD_TIME, () -> {
                    filterState.setState(FilterState.UNLOADED);
                    currentFilter = filterName;
                });
                waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            } else {
                if (allFilters.get(currentFilter) != FilterType.NONE) {
                    filterState.setState(FilterState.UNLOADING);
                    Future<Void> waitForUnloaded = ccs.waitForStatus(FilterState.UNLOADED);
                    ccs.schedule(UNLOAD_TIME, () -> {
                        filterState.setState(FilterState.UNLOADED);
                        currentFilter = filterName;
                    });
                    waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
                }
                targetRotation = (position - 1) * 360 / 5;
                if (currentRotationPosition != targetRotation) {
                    // FIXME: Always rotates in same direction, probably not realistic
                    int degreesToRotate = Math.abs(currentRotationPosition - targetRotation) % 360;
                    filterState.setState(FilterState.ROTATING);
                    Future<Void> waitForRotation = ccs.waitForStatus(FilterState.UNLOADED);
                    Duration rotationTime = ROTATION_TIME_PER_DEGREE.multipliedBy(degreesToRotate);
                    ccs.schedule(rotationTime, () -> {
                        filterState.setState(FilterState.UNLOADED);
                    });
                    waitForRotation.get(rotationTime.toMillis() * 2, TimeUnit.MILLISECONDS);
                    currentRotationPosition = targetRotation;
                }
                filterState.setState(FilterState.LOADING);
                Future<Void> waitForLoaded = ccs.waitForStatus(FilterState.LOADED);
                ccs.schedule(LOAD_TIME, () -> {
                    filterState.setState(FilterState.LOADED);
                    currentFilter = filterName;
                });
                waitForLoaded.get(LOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            }
            

            if ( config.getCameraType().equals(Camera.COMCAM) ) {
                String fname = getCurrentFilter(); 
                String ftype = getFilterType(fname);
                //FIXME: WHERE IS THE SLOT DEFINED?
                int fslot = 1;
                //FIXME: IS THIS CORRECT FOR THE FILTER POSITION?
                int fpos = getInstalledFilters().indexOf(fname);
                EndSetFilterData data = new EndSetFilterData(fname, ftype, 
                                                         fslot, fpos);
                KeyValueDataList kvd = new KeyValueDataList();
                kvd.addData(EndSetFilterData.endKey,data);
                publishCurrentFCState(kvd);
            }
            
            ccs.fireEvent(new CCSSetFilterEvent(filterName, requestedFilterType.getName(), position, targetRotation));
        } catch (InterruptedException | ExecutionException | TimeoutException ex) {
            throw new RuntimeException("Error while changing filter", ex);
        }
    }

    @Override
    public Duration getDurationForFastFilterChange(String filterName) {
        return ROTATION_TIME_PER_DEGREE.multipliedBy(360).plus(LOAD_TIME).plus(UNLOAD_TIME);
    }
    
    private void publishCurrentFCState(KeyValueDataList kvd) {
        synchronized(lastPublishedFCStateDataLock) {
            if ( kvd != null ) {
                lastPublishedFCStateData = kvd;
            }
            LOG.log(Level.FINE,"Publishing current FCS State: "+lastPublishedFCStateData);
            if (lastPublishedFCStateData != null) {
                if ( agent != null ) {
                    agent.publishSubsystemDataOnStatusBus(lastPublishedFCStateData);
                }
            }
        }
    }
    
    void setTopLevelAgent(Agent agent) {
        this.agent = agent;
    }
    
    void publishDataProviderCurrentData() {
        publishCurrentFCState(null);
        
    }
}
