package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.lsst.ccs.Agent;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;

/**
 * The filter changer used by the MCM
 *
 * @author tonyj
 */
public class FilterChanger {

    private static final Logger LOG = Logger.getLogger(FilterChanger.class.getName());

    private List<String> availableFilters;
    private List<String> installedFilters;
    private final CCS ccs;
    private FilterChangerInterface filterChanger;
    private final MCMConfig config;

    FilterChanger(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        this.filterChanger = new FilterChangerSimulation(ccs, config);
    }

    public void start(String configName) throws ExecutionException {
        filterChanger.start(configName);
        // filterChanger is locked at this point, so installed filters cannot change
        availableFilters = filterChanger.getAvailableFilters();
        installedFilters = filterChanger.getInstalledFilters();
        List<String> availableFilterTypes = availableFilters.stream()
            .map(f -> filterChanger.getFilterType(f))
            .collect(Collectors.toList());

        double maxAngle = filterChanger.getMaxAngleForFilterChange();
        double maxFastAngle = filterChanger.getMaxAngleForFastFilterChange();

        int n = availableFilterTypes.size();
        double[] slowChangeTime = new double[n], fastChangeTime = new double[n];
        for (int i=0; i<n; i++) {
            String filter = availableFilters.get(i);
            slowChangeTime[i] = filterChanger.getDurationForSlowFilterChange(filter).getSeconds(); 
            fastChangeTime[i] = filterChanger.getDurationForFastFilterChange(filter).getSeconds();
        }

        LOG.log(Level.INFO, "Available filters {0}", availableFilters);
        ccs.fireEvent(new CCSAvailableFiltersEvent(availableFilters, availableFilterTypes, maxAngle, maxFastAngle, slowChangeTime, fastChangeTime));
    }

    boolean filterIsAvailable(String filter) {
        return availableFilters.contains(filter);
    }

    boolean filterIsInstalled(String filter) {
        return installedFilters.contains(filter);
    }

    List<String> getAvailableFilters() {
        return availableFilters;
    }

    String getCurrentFilter() throws ExecutionException {
        return filterChanger.getCurrentFilter();
    }

    void setFilter(String filter) throws ExecutionException {
        filterChanger.setFilter(filter);
    }

    Duration getEstimatedTimeForChange(String filterName) throws ExecutionException {
        return filterChanger.getDurationForSlowFilterChange(filterName);
    }

    ControlledSubsystem registerMCMSubsystem(MCMSubsystem mcm) {
        final FilterChangerInterface controlledSubsystem = config.getCameraType() == Camera.MAIN_CAMERA ? 
                new MainCameraFilterChangerSubsystemLayer(mcm, ccs, config) : new ComCamFilterChangerSubsystemLayer(mcm, ccs, config);
        filterChanger = controlledSubsystem;
        return (ControlledSubsystem) controlledSubsystem;
    }
    
    void setTopLevelAgent(Agent agent) {
        if ( filterChanger instanceof FilterChangerSimulation ) {
            ((FilterChangerSimulation)filterChanger).setTopLevelAgent(agent);
        }
    }

    void publishDataProviderCurrentData() {
        if ( filterChanger instanceof FilterChangerSimulation ) {
            ((FilterChangerSimulation)filterChanger).publishDataProviderCurrentData();
        }
    }
}
