package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.subsystem.comcam.filterchanger.data.EndSetFilterData;
import org.lsst.ccs.subsystem.comcam.filterchanger.data.SetFilterData;
import org.lsst.ccs.subsystem.comcam.filterchanger.states.ComCamFCState;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.states.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to a real comcam filter changer subsystem.
 *
 * @author tonyj
 */
public class ComCamFilterChangerSubsystemLayer extends ControlledSubsystem implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(ComCamFilterChangerSubsystemLayer.class.getName());
    private final static Map<Enum<ComCamFCState>, Enum<FilterState>> COMCAM_TO_FILTER_STATE = new HashMap<>();

    static {
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_MOVING, FilterState.LOADING);
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_FILTER1, FilterState.LOADED);
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_FILTER2, FilterState.LOADED);
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_FILTER3, FilterState.LOADED);
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_LOAD1, FilterState.UNLOADING);
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_LOAD3, FilterState.UNLOADING);
        COMCAM_TO_FILTER_STATE.put(ComCamFCState.POSITION_OTHER, FilterState.UNLOADED);
    }

    public ComCamFilterChangerSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFilterChangerSubsystemName(), ccs, config);
    }

    @Override
    public void setFilter(String filter) throws ExecutionException {
        commandSender.sendCommand("goToFilter " + filter);
        waitUntilMoveComplete(getDurationForSlowFilterChange(filter));
    }

    @Override
    public List<String> getAvailableFilters() throws ExecutionException {
        Map<String, String> l = commandSender.sendCommand(Map.class, "getAllFilters");
        return l.entrySet().stream()
            .map(e -> e.getKey())
            .collect(Collectors.toList());
    }

    @Override
    public List<String> getInstalledFilters() throws ExecutionException {
        return commandSender.sendCommand(List.class, "getInstalledFilters");
    }

    @Override
    public String getCurrentFilter() throws ExecutionException {
        return commandSender.sendCommand(String.class, "getFilter", String.class);
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Filter changer connected");
        ComCamFCState state = initialState.getState(ComCamFCState.class);
        translateComCamStateToFilterState(CCSTimeStamp.currentTime(), state, null);
    }

    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Filter changer disconnected");
        // TOOD: Deal with initial state                        
//        StateBundle result = commandSender.sendCommand("getState", StateBundle.class);
//        FocalPlaneState state = result.getState(FocalPlaneState.class);
//        translateFocalPlaneStateToRaftsState(state);
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        CCSTimeStamp ts = statusChange.getCCSTimeStamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle changedStates = newStates.diffState(oldStates);
        String cause = statusChange.getCause();
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateComCamStateToFilterState(when, value, cause);
        });
    }

    private void waitUntilMoveComplete(Duration estimatedDurationForFilterChange) throws ExecutionException {
        // TOOD: Do we need to wait for it to start moving first?
        Future<Void> waitForStatus = ccs.waitForStatus(FilterState.LOADED);
        try {
            waitForStatus.get(estimatedDurationForFilterChange.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException | TimeoutException ex) {
            throw new ExecutionException("Timeout waiting for filter change to complete", ex);
        }
    }

    private void translateComCamStateToFilterState(CCSTimeStamp when, Enum value, String cause) {
        LOG.log(Level.INFO, "Got filter changer state {0} ", value);
        Enum<FilterState> converted = COMCAM_TO_FILTER_STATE.get(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted, cause));
        }
    }

    @Override
    protected void onEvent(StatusMessage msg) {
        Object data = msg.getObject();
        if (data instanceof KeyValueDataList) {
            KeyValueDataList kvdl = (KeyValueDataList) data;
            Object dataObject = kvdl.getListOfKeyValueData().get(0).getValue();
            if (dataObject instanceof EndSetFilterData) {
                EndSetFilterData efd = (EndSetFilterData) dataObject;
                final CCSSetFilterEvent ccsSetFilterEvent = new CCSSetFilterEvent(efd.getFilterName(), efd.getFilterType(), efd.getFilterSlot(), efd.getFilterPosition());
                ccs.fireEvent(ccsSetFilterEvent);
                LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
            } else if (dataObject instanceof SetFilterData) {
                SetFilterData sfd = (SetFilterData) dataObject;
                final CCSSetFilterEvent ccsSetFilterEvent = new CCSSetFilterEvent(sfd.getFilterName(), sfd.getFilterType());
                ccs.fireEvent(ccsSetFilterEvent);
                LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
            }
        }
    }
}
