package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData;
import org.apache.avro.generic.GenericRecord;
import org.influxdb.dto.BatchPoints;
import org.influxdb.dto.Point;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.camera.kafka.avro.KafkaProxyBridge;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.camera.kafka.avro.AvroSchemaMapper;
import org.lsst.ccs.camera.kafka.avro.GenericRecordManager;
import org.lsst.ccs.camera.kafka.avro.KafkaBridge;
import org.lsst.ccs.camera.kafka.avro.RequestResult;
import org.lsst.ccs.camera.kafka.avro.Topic;
import org.lsst.ccs.camera.sal.classes.DataProviderInfoUtils;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.InfluxDbClientService;
import org.lsst.ccs.utilities.scheduler.Scheduler;


/**
 *
 * @author The LSST CCS Team
 */
public class KafkaService extends Subsystem implements HasLifecycle, DataProviderDictionaryService.DataProviderDictionaryListener, StatusMessageListener {

    private static final Logger LOG = Logger.getLogger(KafkaService.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dpdService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private InfluxDbClientService influxDbClientService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private KafkaBridge kafkaBridge;

    @ConfigurationParameter
    private volatile Camera device = Camera.MAIN_CAMERA;

    @ConfigurationParameter(description = "The maximum number of records published for each topic", units = "unitless")
    private volatile int maxNumberOfPublishedRecords = 50;

    private volatile GenericRecordManager recordManager;
    private final Object recordManagerLock = new Object();
    
    private AvroSchemaMapper mapper;

    private final Object topicRegistrationLock = new Object();
    private final Set<AgentInfo> relevantAgentInfo = new CopyOnWriteArraySet<>();
    
    private Scheduler kafkaScheduler;
    private List<String> ignoredPaths = new ArrayList<>();

    private List<String> registeredTopics;

    private final Predicate<BusMessage> isRelevantAgent = (bm) -> {
        AgentInfo agentInfo = bm.getOriginAgentInfo();
        return relevantAgentInfo.contains(agentInfo);
    };

    public KafkaService() {
        super("kafka-service",AgentInfo.AgentType.SERVICE);
    }
    
    @Override
    public void build() {
        
        kafkaScheduler = periodicTaskService.createScheduler("kafka-publication", 100);

        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("kafka-influx-statistics",
                        () -> {
                            processKafkaProxyData();
                        }).withIsFixedRate(true).withPeriod(Duration.ofSeconds(5)));
    }
    
    @Override
    public void postBuild() {
        if ( kafkaBridge.supportsAggregateRecordsPublication() ) {
            periodicTaskService.scheduleAgentPeriodicTask(
                    new AgentPeriodicTask("publish-to-kafka",
                            () -> {
                                resetRecordManagerAndPublishToKafka();
                            }).withIsFixedRate(true).withPeriod(Duration.ofSeconds(5)));
        }
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask("kafka-heartbeat",
                        () -> {
                            publishHeartBeat();
                        }).withIsFixedRate(true).withPeriod(Duration.ofSeconds(1)));
        
    }
    
    @Override
    public void init() {
        LOG.log(Level.INFO, "Initializing KafkaService for Camera: {0}", device);
        mapper = new AvroSchemaMapper(device);        
        dpdService.addDataProviderDictionaryListener(this);
        agent.getMessagingAccess().addStatusMessageListener(this, BusMessageFilterFactory.messageClass(StatusSubsystemData.class).and(isRelevantAgent));
        recordManager = new GenericRecordManager(mapper);
    }
    
    @Override 
    public void postInit() {
        try {
            registeredTopics = kafkaBridge.listTopics();
            LOG.log(Level.INFO, "Established connection to Kafka Bridge.\nExisting topis: {0}", registeredTopics);
        } catch (Exception e) {
            throw new RuntimeException("Could not fetch list of registered topics.",e);
        }        
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData ssd = (StatusSubsystemData) msg;
        String agentName = msg.getOriginAgentInfo().getName();

        KeyValueDataList kvdl = ssd.getEncodedData();
        synchronized(recordManagerLock) {
            for (KeyValueData d : kvdl) {
                String fullPath = agentName + "/" + d.getKey();
                if ( ignoredPaths.contains(fullPath) ) {
                    continue;
                }
                Object value = d.getValue();
                if ( fullPath.endsWith("/state") && ((String)value).equals("OFF_LINE") ) {
                    continue;
                }
                
                long timestamp = d.getCCSTimeStamp().getUTCInstant().toEpochMilli();
                if ( !recordManager.addKeyValueForPath(fullPath, value, timestamp) ) {
                    if ( !ignoredPaths.contains(fullPath) ) {
                        LOG.log(Level.WARNING, "Ignoring path {0}", fullPath);
                        ignoredPaths.add(fullPath);
                    }
                }
            }

            if ( !kafkaBridge.supportsAggregateRecordsPublication() ) {
                resetRecordManagerAndPublishToKafka();
            }
        }
    }

    @Override
    public void dataProviderDictionaryUpdate(DataProviderDictionaryService.DataProviderDictionaryEvent evt) {
        if (evt.getEventType() == DataProviderDictionaryService.DataProviderDictionaryEvent.EventType.ADDED) {
            if (mapper.addSchemaForDictionaryAndAgent(evt.getDictionary(), evt.getAgentInfo())) {
                LOG.log(Level.INFO, "Adding avro topics and schemas for agent {0}", evt.getAgentInfo().getName());
                synchronized (topicRegistrationLock) {
                    for (Entry<String, Topic> e : mapper.getAllTopicsMap().entrySet()) {
                        String topicName = e.getValue().getFullName();
                        if (!registeredTopics.contains(topicName)) {
                            Topic topic = e.getValue();
                            try {
                                if (kafkaBridge.registerKafkaTopic(topic)) {
                                    registeredTopics.add(topicName);
                                } else {
                                    LOG.log(Level.WARNING, "Failed to register topic {0}", topicName);                                    
                                }
                            } catch (Exception ex) {
                                LOG.log(Level.WARNING, "Failed to register topic " + topicName, ex);
                            }
                        }                        
                    }
                    String agentName = evt.getAgentInfo().getName();
                    for ( DataProviderInfo dpi : evt.getDictionary().getDataProviderInfos() ) {
                        if ( ! DataProviderInfoUtils.acceptDataProviderInfo(dpi) ) {
                            String fullPath = agentName + "/" + dpi.getFullPath();
                            ignoredPaths.add(fullPath);
                        }
                    }
                    
                    relevantAgentInfo.add(evt.getAgentInfo());
                }
            }
        } else if (evt.getEventType() == DataProviderDictionaryService.DataProviderDictionaryEvent.EventType.REMOVED) {
            relevantAgentInfo.remove(evt.getAgentInfo());
        }
    }
    
    private void resetRecordManagerAndPublishToKafka() {
        GenericRecordManager localRecordManager = recordManager;
        synchronized(recordManagerLock) {
            recordManager = new GenericRecordManager(mapper);
        }
        for (Topic t : localRecordManager.getTopicsWithData()) {

            if (!registeredTopics.contains(t.getFullName())) {
                continue;
            }
                
            Schema schema = mapper.getSchemaForTopic(t);

            List<GenericRecord> l = null;
            for (GenericRecord gr : localRecordManager.getRecordsForTopic(t)) {
                if (l == null) {
                    l = new ArrayList<>();
                }
                l.add(gr);
                if (l.size() >= maxNumberOfPublishedRecords) {
                    publishListOfRecordsForSchema(schema, l);
                    l = null;
                }
            }
            if (l != null) {
                publishListOfRecordsForSchema(schema, l);
            }
        }

    }
    
    private void publishHeartBeat() {
        Topic hbTopic = mapper.getHeartbeatTopic();
        Schema schema = mapper.getSchemaForTopic(hbTopic);
        GenericRecord record = new GenericData.Record(schema);
        record.put("timestamp", System.currentTimeMillis());
        record.put(mapper.getHeartbeatTopicField().getName(), true);
        publishListOfRecordsForSchema(schema, Collections.singletonList(record));
    }
        
    
    private final Map<String, DataAccumulator> timeAccumMap = new ConcurrentHashMap<>();
    private final Map<String, DataAccumulator> nRecordsAccumMap = new ConcurrentHashMap<>();
    private final Map<String, DataAccumulator> sizeAccumMap = new ConcurrentHashMap<>();
    private static final Object dataProcessingLock = new Object();
    
    private void publishListOfRecordsForSchema(Schema schema, List<GenericRecord> records) {
        final List<GenericRecord> tmpList = new ArrayList<>(records);
        kafkaScheduler.schedule(() -> {
            RequestResult result = kafkaBridge.publishRecordsForSchema(schema, tmpList);
            if( influxDbClientService != null && influxDbClientService.isEnabled() ) {
                synchronized(dataProcessingLock) {        
                    String topicName = schema.getName();
                    DataAccumulator timing = timeAccumMap.computeIfAbsent(topicName, (n) -> {return new DataAccumulator(); });
                    timing.accumulate(result.getTimeInMillis());
                    DataAccumulator nRecords = nRecordsAccumMap.computeIfAbsent(topicName, (n) -> {return new DataAccumulator(); });
                    nRecords.accumulate(result.getNumberOfRecords());
                    DataAccumulator size = sizeAccumMap.computeIfAbsent(topicName, (n) -> {return new DataAccumulator(); });
                    size.accumulate(result.getRequestSize());
                }
            }
        }, 0, TimeUnit.MILLISECONDS);

    }
    
    
    private class DataAccumulator {

        private volatile double accum = 0;
        private volatile double max = -Double.MAX_VALUE;
        private volatile int counter = 0;

        /**
         * Add to the existing value.
         */
        public synchronized void accumulate(double value) {
            accum += value;
            counter++;
            if (value > max) {
                max = value;
            }
        }

        public double getAverageValue() {
            return counter > 0 ? accum / ((double) counter) : Double.NaN;
        }

        public double getMaxValue() {
            return counter > 0 ? max : Double.NaN;
        }

        public double getTotalValue() {
            return accum;
        }

        public int getCounts() {
            return counter;
        }
    }
    
    
    private void processKafkaProxyData() {
        if ( influxDbClientService != null &&  influxDbClientService.isEnabled() ) {
            Map<String, DataAccumulator> oldTimeAccumMap;
            Map<String, DataAccumulator> oldNRecordsAccumMap;
            Map<String, DataAccumulator> oldSizeAccumMap;

            synchronized (dataProcessingLock) {
                oldTimeAccumMap = new HashMap<>(timeAccumMap);
                oldNRecordsAccumMap = new HashMap<>(nRecordsAccumMap);
                oldSizeAccumMap = new HashMap<>(sizeAccumMap);
                timeAccumMap.clear();
                nRecordsAccumMap.clear();
                sizeAccumMap.clear();
            }

            BatchPoints bp = influxDbClientService.createBatchPoints();
            for (String topicName : oldTimeAccumMap.keySet()) {
                long time = System.currentTimeMillis();
                Point.Builder pointBuilder = Point.measurement("kafka_proxy_metrics").time(time, TimeUnit.MILLISECONDS);

                DataAccumulator timing = oldTimeAccumMap.get(topicName);
                pointBuilder = pointBuilder.addField("publish_time_avg", timing.getAverageValue()).
                        addField("publish_time_max", timing.getMaxValue()).
                        addField("publish_count", timing.getCounts());

                DataAccumulator nRecords = oldNRecordsAccumMap.get(topicName);
                pointBuilder = pointBuilder.addField("publish_records_avg", nRecords.getAverageValue()).
                        addField("publish_records_max", nRecords.getMaxValue());
                    
                DataAccumulator size = oldSizeAccumMap.get(topicName);
                pointBuilder = pointBuilder.addField("publish_size_avg", size.getAverageValue()).
                        addField("publish_size_max", size.getMaxValue());

                if (pointBuilder.hasFields()) {
                    Point point = pointBuilder.tag("topic", topicName).tag(influxDbClientService.getGlobalTags()).build();
                    bp = bp.point(point);
                }

            }
            
            if ( !bp.getPoints().isEmpty() ) {
                influxDbClientService.write(bp);                
            }            
        }
    }
    
}
