package org.lsst.ccs.subsystem.ocsbridge;

import io.minio.BucketExistsArgs;
import io.minio.MakeBucketArgs;
import io.minio.PutObjectArgs;
import io.minio.MinioClient;
import io.minio.RemoveObjectArgs;
import io.minio.StatObjectArgs;
import io.minio.errors.ErrorResponseException;
import io.minio.errors.MinioException;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Properties;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;

/**
 * Utility for uploading files to the LFA (s3)
 *
 * @author Farrukh, Max
 */
public class S3FileUploaderAndDeleter {
    
    private final String lfaId;
    private final String lfaSecret;
    private final String lfaBucket;
    private final String lfaEndpoint;
    private final String cscName;
    private boolean enforceNoOverwite;
    
    public S3FileUploaderAndDeleter(OCSBridgeConfig config) {
        //Read LFA Cedentials file
        this(BootstrapResourceUtils.getBootstrapProperties(config.getLfaCredentialsFile()), config.getDevice().getCscName());
    }
    
    public S3FileUploaderAndDeleter(Properties lfaProperties, String cscName) {
        this.cscName = cscName;
        lfaId = lfaProperties.getProperty("lfa.id", "");
        lfaSecret = lfaProperties.getProperty("lfa.secret", "");
        lfaBucket = lfaProperties.getProperty("lfa.bucket", "");
        String endPoint = lfaProperties.getProperty("lfa.endpoint", "");
        if (lfaId.isEmpty() || lfaSecret.isEmpty() || lfaBucket.isEmpty() || endPoint.isEmpty()) {
            throw new IllegalArgumentException("The following quantities must be specified for the LFA credentials: id, secret, bucket, endpoint.");
        }
        lfaEndpoint = endPoint.endsWith("/") ? endPoint : endPoint + "/";
    }

    /**
     * Write an AdditionalFile to S3/LFA
     *
     * @param additionalFile The file to upload
     * @return The URL the uploaded file can be read from
     * @throws java.io.IOException If the file could not be written
     */
    public String writeToS3(AdditionalFile additionalFile) throws IOException {
        // object name is the path name and file name to be appended to the bucket 
        String objectName = computePathForFile(additionalFile);
        
        try {
            // Create a minioClient with the MinIO server, its access key and secret key.
            MinioClient minioClient
                    = MinioClient.builder()
                            .endpoint(lfaEndpoint)
                            .credentials(lfaId, lfaSecret)
                            .build();
            boolean bucketFound = minioClient.bucketExists(BucketExistsArgs.builder().bucket(lfaBucket).build());
            if (!bucketFound) {
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(lfaBucket).build());
            }
            /// Normally we allow ovwrite only when writing to play.min.io, except if we are explicitly testing the no-overwrite policy.
            boolean allowOverwrite = lfaEndpoint.contains("play.min.io") && !enforceNoOverwite;
            // check to make sure that file does not exist already 
            boolean alreadyThere = objectExists(minioClient, lfaBucket, objectName);
            if (alreadyThere && !allowOverwrite) {
                throw new IOException("S3 file already exists " + lfaEndpoint + lfaBucket + "/" + objectName);
            }
            
            try (ByteArrayOutputStream boutstream = new ByteArrayOutputStream()) {
                additionalFile.writeFile(boutstream);
                try (ByteArrayInputStream binstream = new ByteArrayInputStream(boutstream.toByteArray())) {
                    minioClient.putObject(PutObjectArgs.builder().bucket(lfaBucket).object(objectName).stream(binstream, binstream.available(), -1).build());
                }
            }            
            
        } catch (MinioException | InvalidKeyException | NoSuchAlgorithmException ex) {
            System.out.println(" Exception coming next " + ex.toString());
            throw new IOException("Unable to write s3 file", ex);
        }
        return lfaEndpoint + lfaBucket + "/" + objectName;
    }
    
    private String computePathForFile(AdditionalFile additionalFile) {
        return cscName + "/" + additionalFile.getFileType() + "/" + additionalFile.getObsId().getDateString() + "/" + additionalFile.getFileName();
    }
    
    void deleteFromS3(AdditionalFile additionalFile)
            throws IOException {
        
        try {
            MinioClient minioClient
                    = MinioClient.builder()
                            .endpoint(lfaEndpoint)
                            .credentials(lfaId, lfaSecret)
                            .build();
            
            boolean found = minioClient.bucketExists(BucketExistsArgs.builder().bucket(lfaBucket).build());
            if (!found) {
                // Make a new bucket called 'farrukh'.
                minioClient.makeBucket(MakeBucketArgs.builder().bucket(lfaBucket).build());
            }
            
            minioClient.removeObject(RemoveObjectArgs.builder().bucket(lfaBucket).object(computePathForFile(additionalFile)).build());
            
        } catch (MinioException | InvalidKeyException | NoSuchAlgorithmException ex) {
            throw new IOException("Unable to delete s3 file", ex);
        }
    }
    
    private static boolean objectExists(MinioClient minioClient, String bucketName, String objectName) throws IOException {
        try {
            minioClient.statObject(
                    StatObjectArgs.builder().bucket(bucketName).object(objectName).build());
            return true;
        } catch (ErrorResponseException e) {
            if ("NoSuchKey".equals(e.errorResponse().code())) {
                return false;
            } else {
                throw new IOException("Error occurred while checking object existence", e);
            }
        } catch (MinioException | InvalidKeyException | NoSuchAlgorithmException e) {
            throw new IOException("Error occurred while checking object existence", e);
        }
    }

    /**
     * Only necessary for testing
     * @param enforce 
     */
    void enforceNoOverwrite(boolean enforce) {
        enforceNoOverwite = enforce;
    }
}
