package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.states.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutochangerTrucksState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to the filter changer subsystem.
 *
 * @authors tonyj aubourg and aboucaud
 */
public class MainCameraFilterChangerSubsystemLayer extends ControlledSubsystem implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(ComCamFilterChangerSubsystemLayer.class.getName());
    private final static Map<FcsEnumerations.McmFilterState, FilterState> FCS_TO_FILTER_STATE = new HashMap<>();
    private volatile CCSSetFilterEvent ccsSetFilterEvent;
    private String previousFilter = "";

    static {
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmFilterState.LOADING, FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmFilterState.UNLOADING, FilterState.UNLOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmFilterState.LOADING, FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmFilterState.NOFILTER, FilterState.NOFILTER);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmFilterState.ROTATING, FilterState.ROTATING);
    }

    private static class FilterStateConverter {

        private FilterState convertStateInstantaneous(FcsEnumerations.McmFilterState state) {
            FilterState s = FCS_TO_FILTER_STATE.get(state);
            if (s == null) {
                s = FilterState.LOADED;
            }
            return s;
        }

        FilterState convertState(FcsEnumerations.McmFilterState state) {
            FilterState s = convertStateInstantaneous(state);
            return s;
        }
    }

    private final static FilterStateConverter converter = new FilterStateConverter();

    private FilterState convertState(FcsEnumerations.McmFilterState state) {
        return converter.convertState(state);
    }

    public MainCameraFilterChangerSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFilterChangerSubsystemName(), ccs, config);
    }

    @Override
    public void setFilter(String filter) throws ExecutionException {
        // This value is used by onEvent to filter events published by the FCS
        previousFilter = getCurrentFilter();
        // The following test avoids creating setFilter events if the requested filter is already set 
        // This cannot be handled by the CCSCommandPreconditions since the filter currently set is not known by the MCM
        if ( previousFilter.equals(filter) ) {
            return;
        }
        /* Trigger a StartSetFilterEvent */
        ccsSetFilterEvent = new CCSSetFilterEvent(filter, getFilterType(filter));
        ccs.fireEvent(ccsSetFilterEvent);
        LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
        
        Duration timeout = getDurationForSlowFilterChange(filter);

        if ("NONE".equals(filter)) {
            commandSender.sendCommand(Void.TYPE, timeout, "setNoFilter");
            waitUntilMoveComplete(FilterState.NOFILTER, timeout);
        } else {
            // Crude fix for LCOBM-198
            if (filter.contains("_")) {
                filter = filter.split("_")[0];
            }
            commandSender.sendCommand(Void.TYPE, timeout, "setFilterByName", filter);
            waitUntilMoveComplete(FilterState.LOADED, timeout);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<String> getAvailableFilters() throws ExecutionException {
        return commandSender.sendCommand(List.class, "getAvailableFilters");
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<String> getInstalledFilters() throws ExecutionException {
        return commandSender.sendCommand(List.class, "getInstalledFilters");
    }

    @Override
    public String getCurrentFilter() throws ExecutionException {
        return commandSender.sendCommand(String.class, "getOnlineFilterName");
    }

    @Override
    public double getMaxAngleForFilterChange() throws ExecutionException {
        return commandSender.sendCommand(Double.class, "getMaxAngleForFilterChange");
    }

    @Override
    public double getMaxAngleForFastFilterChange() throws ExecutionException {
        return commandSender.sendCommand(Double.class, "getMaxAngleForFastFilterChange");
    }

    @Override
    public Duration getDurationForSlowFilterChange(String filterName) throws ExecutionException {
        return commandSender.sendCommand(Duration.class, "getDurationForSlowFilterChange", filterName);
    }

    @Override
    public Duration getDurationForFastFilterChange(String filterName) throws ExecutionException {
        return commandSender.sendCommand(Duration.class, "getDurationForFastFilterChange", filterName);
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Filter changer connected");
        FcsEnumerations.FilterState state = initialState.getState(FcsEnumerations.FilterState.class);
        LOG.log(Level.INFO, "Got initial FCS FilterState {0}", state);
        FcsEnumerations.McmFilterState mcmState = initialState.getState(FcsEnumerations.McmFilterState.class);
        LOG.log(Level.INFO, "Got initial FCS McmFilterState {0}", mcmState);
        FcsEnumerations.AutochangerInclinationState inclinationState = initialState.getState(FcsEnumerations.AutochangerInclinationState.class);
        LOG.log(Level.INFO, "Got initial FCS AutochangerInclinationState {0}", inclinationState);
        translateFcsStateToFilterState(CCSTimeStamp.currentTime(), mcmState, null);
    }

    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Filter changer disconnected");
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle changedStates = newStates.diffState(oldStates);
        String cause = statusChange.getCause();
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue())
                .forEachOrdered((value) -> {
                    if (value instanceof FcsEnumerations.FilterState) {
                        //translateFcsStateToFilterState(when, (FcsEnumerations.FilterState) value);
                    } else if (value instanceof FcsEnumerations.McmFilterState) {
                        LOG.log(Level.INFO, "Got McmFilterState {0}", value);
                        translateFcsStateToFilterState(when, (FcsEnumerations.McmFilterState) value, cause);
                    } else if (value instanceof FcsEnumerations.AutochangerInclinationState) {
                        LOG.log(Level.INFO, "Got AutochangerInclinationState {0}", value);
                    }
                });
    }

    private void waitUntilMoveComplete(FilterState finalState, Duration estimatedDurationForFilterChange)
            throws ExecutionException {
        Future<Void> waitForStatus = ccs.waitForStatus(finalState);
        try {
            waitForStatus.get(estimatedDurationForFilterChange.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException | TimeoutException ex) {
            throw new ExecutionException("Timeout waiting for filter change to complete", ex);
        }
    }

    private void translateFcsStateToFilterState(CCSTimeStamp when, FcsEnumerations.McmFilterState value, String cause) {
        if (value == null) {
            value = FcsEnumerations.McmFilterState.LOADED;
        }
        FilterState converted = convertState(value);
        LOG.log(Level.INFO, "FilterState was converted to {0} ", converted);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted, cause));
        }
    }

    @Override
    protected void onEvent(StatusMessage msg) {
        Object data = msg.getObject();
        
        if ( data instanceof KeyValueData ) {
            final KeyValueData sentData = (KeyValueData) data;
            String dataKey = sentData.getKey();
            LOG.log(Level.FINE, "Got kvd {0} {1}", new Object[]{dataKey, sentData.getValue()});
            if ( !"fcs/mcm".equals(dataKey) ) {
                return;
            }
            KeyValueDataList kvdl = (KeyValueDataList) sentData.getValue();
            String filterName = null;
            String filterType = null;
            int slot = 0;
            int position = 0;
            int percentage = 0;
            double proximity = 0;
            AutochangerTrucksState location = AutochangerTrucksState.ONLINE;

            for ( KeyValueData kvd : kvdl ) {
                Object value = kvd.getValue();
                switch ( kvd.getKey() ) {
                    case "filter_on_autochanger":
                        filterName = value.toString();
                        filterType = getFilterType(filterName);
                        break;
                    case "filter_previous_socketID":
                        slot = ((Number) value).intValue();
                        break;
                    case "autochanger_trucks_position":
                        position = ((Number) value).intValue();
                        break;
                    case "autochanger_trucks_state":
                        location = AutochangerTrucksState.valueOf(value.toString());
                        break;
                    case "setFilter_percentage":
                        /* Could be used by a LOVE interface but currently unused */
                        percentage = ((Number) value).intValue();
                        break;
                    case "proximity":
                        /* More precise than the autochanger trucks position close to the ONLINE position */
                        proximity = ((Number) value).doubleValue();
                        break;
                }
            }
            
            /* Only publish info on the buses if this corresponds to the new filter at the end of the setFilter */
            boolean endOfExchange = location == AutochangerTrucksState.ONLINE;
            boolean isSameFilter = this.previousFilter.equals(filterName);

            if ( endOfExchange && !isSameFilter ) {
                ccsSetFilterEvent = new CCSSetFilterEvent(filterName, filterType, slot, proximity == 0 ? position : proximity);
                ccs.fireEvent(ccsSetFilterEvent);
                LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
            }
        }
    }

    @Override
    protected void onRepublishServiceData() {
        if (ccsSetFilterEvent != null) {
            ccs.fireEvent(ccsSetFilterEvent);
            LOG.log(Level.INFO, "Resent: {0}", ccsSetFilterEvent);
        }
    }

}
