package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystem.ocsbridge.util.State.StateChangeListener;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.sal.camera.states.CCSCommandStateEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.SummaryStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent.SummaryState;

/**
 * Top level "module" for running the OCSBridge as a CCS Subsystem
 *
 * @author tonyj
 */
public class OCSSubsystem extends Subsystem implements HasLifecycle {

    private OCSBridge ocsBridge;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private OCSBridgeConfig config;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private HeaderServiceEventHandler eventHandler;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentLockService agentLockService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentLoginService loginService;

    @SuppressWarnings("OverridableMethodCallInConstructor")
    OCSSubsystem() {
        super("ocs-bridge", AgentType.OCS_BRIDGE);
    }

    @Override
    public void init() {
        agentStateService.registerState(SummaryState.class, "SAL Summary State", this);
        agentStateService.registerState(OfflineState.class, "SAL Offline State", this);
        agentStateService.registerState(CCSCommandStateEvent.CCSCommandState.class, "SAL Command State", this);

    }

    @Override
    public void postInit() {
        //TO-DO: how does the login service work? Does it need the buses? If so it needs
        //to be moved to postStart.
        loginService.login(config.getUserID(), "noCredentialsNeeded");

        CCS ccs = new CCS();
        MCMLayer mcmLayer = new MCMCCSLayer(this, ccs, config, loginService, agentLockService);
        ocsBridge = new OCSBridge(config, ccs, mcmLayer);        
    }
    
    @Override
    public void postStart() {
        // Make the LSE209 state available as a CCS state
        final State lse209State = ocsBridge.getLse209State();
        final State offlineState = ocsBridge.getOfflineState();
        final State commandState = ocsBridge.getCommandState();
        SoftwareVersionsGenerator softwareVersionsGenerator = new SoftwareVersionsGenerator(ocsBridge, this);
        agentStateService.updateAgentState(lse209State.getState(), offlineState.getState(), commandState.getState());
        StateChangeListener l = (StateChangeListener) (CCSTimeStamp when, Enum state, Enum oldState, String cause) -> {
            agentStateService.updateAgentState(when, state);
            if (state == SummaryStateEvent.SummaryState.STANDBY) {
                softwareVersionsGenerator.run(true);
            }
        };
        lse209State.addStateChangeListener(l);
        offlineState.addStateChangeListener(l);
        commandState.addStateChangeListener(l);
        ocsBridge.setCCSCommandExecutor(new CCSCommandExecutor() {
            @Override
            void sendAck(CCSCommand.CCSAckOrNack can) {
                // Note: We add 100mS to the estimate to keep ShellCommandConsole from timing
                // out, especially on zero length commands.
                OCSSubsystem.this.sendAck(can.getDuration().plus(Duration.ofMillis(100)));
            }

            @Override
            void sendNack(CCSCommand.CCSAckOrNack can) {
                OCSSubsystem.this.sendNack(can.getReason());
            }
        });

        OCSBridgeSALLayer ocsInterface = new OCSBridgeSALLayer(ocsBridge);
        ocsInterface.addSALEventListener(eventHandler);
        ocsBridge.setOCSCommandExecutor(ocsInterface);
        ocsInterface.start();
        
        softwareVersionsGenerator.run(false);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setAvailable() throws Exception {
        CCSCommand.CCSSetAvailableCommand setAvailable = new CCSCommand.CCSSetAvailableCommand();
        ocsBridge.execute(setAvailable);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void revokeAvailable() throws Exception {
        CCSCommand.CCSRevokeAvailableCommand revokeAvailable = new CCSCommand.CCSRevokeAvailableCommand();
        ocsBridge.execute(revokeAvailable);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void simulateFault() throws Exception {
        CCSCommand.CCSSimulateFaultCommand simulateFault = new CCSCommand.CCSSimulateFaultCommand();
        ocsBridge.execute(simulateFault);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clearFault() throws Exception {
        CCSCommand.CCSClearFaultCommand clearFault = new CCSCommand.CCSClearFaultCommand();
        ocsBridge.execute(clearFault);
    }
    
    void addSubsystem(AgentInfo ai, DataProviderDictionary dict) {
        ocsBridge.addSubsystem(ai, dict);
    }
    void removeSubsystem(String subsystemName) {
        ocsBridge.removeSubsystem(subsystemName);
    }
    
}
