package org.lsst.ccs.subsystem.ocsbridge;

import java.util.Properties;
import java.util.UUID;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.sal.SAL;
import org.lsst.sal.SALCommand;
import org.lsst.sal.SALEvent;
import org.lsst.sal.SALTelemetry;
import org.lsst.sal.atcamera.ATCamera;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.sal.camera.SALCamera;
import org.lsst.sal.cccamera.CCCamera;

/**
 *
 * @author Farrukh Azfar. Configuration class to hold parameters for Camera,
 * AUXTel, ComCam -
 */
@SuppressWarnings("FieldMayBeFinal")
public class OCSBridgeConfig {

    @ConfigurationParameter
    private volatile Camera device = Camera.MAIN_CAMERA;
    
    @ConfigurationParameter
    private final boolean hasFilterChanger = true;
    
    @ConfigurationParameter
    private volatile String mcmName = "mcm";
    
    @ConfigurationParameter(description = "Userid to be used to MCM subsystem", isFinal = true)
    private volatile String userId = "ocs";

    @ConfigurationParameter(description = "Properties file with lfa credentials", isFinal = true)
    private volatile String lfaCredentialsFile = "lfa-test.properties";
    
    private final Properties salProperties;
    
    public OCSBridgeConfig() {
        salProperties = new Properties();
        salProperties.put("AGENT_NAME", "ocs-bridge");
        salProperties.put("AGENT_TYPE", "CCS");
        salProperties.put("AGENT_UUID", UUID.randomUUID());
    }
  
    /**
     * Subsystems we should listen to for events
     */
    @ConfigurationParameter(maxLength = 50)
    @SuppressWarnings("VolatileArrayField")
    private volatile String[] listenSubsystems = new String[0];

    public boolean hasFilterChanger() {
        return hasFilterChanger;
    }

    public String getMCMName() {
        return mcmName;
    }

    public Camera getDevice() {
        return device;
    }
    
    public String getLfaCredentialsFile() {
        return lfaCredentialsFile;
    }

    public String[] getListenSubsystems() {
        return listenSubsystems;
    }

    String getUserID() {
        return userId;
    }
    
    SAL<CameraCommand, CameraEvent, CameraTelemetry> getSALManager() {
        return createSALManager(device, salProperties);
    }

    static SAL<CameraCommand, CameraEvent, CameraTelemetry> createSALManager(Camera device, Properties properties) {
        switch (device) {
            case MAIN_CAMERA:
                return SALCamera.create(properties);
            case AUXTEL:
                return ATCamera.create(properties);
            case COMCAM:
                return CCCamera.create(properties);
            default:
                throw new UnsupportedOperationException("Device not yet supported: " + device);
        }
    }
    
    String getCSCRoot() {
        switch (device) {
            case MAIN_CAMERA:
                return "MTCamera";
            case AUXTEL:
                return "ATCamera";
            case COMCAM:
                return "CCCamera";
            default:
                throw new UnsupportedOperationException("Device not yet supported: " + device);
        }
    }
    
    SAL<SALCommand, SALEvent, SALTelemetry> getHeaderServiceManager() {
        switch (device) {
            case MAIN_CAMERA:
                return org.lsst.sal.mtheader.SALMain.create(salProperties);
            case AUXTEL:
                return org.lsst.sal.atheader.SALMain.create(salProperties);
            case COMCAM:
                return org.lsst.sal.ccheader.SALMain.create(salProperties);
            default:
                throw new UnsupportedOperationException("Device not yet supported: " + device);
        }
    }

    static OCSBridgeConfig createDefaultConfig(Camera camera) {
        OCSBridgeConfig config = new OCSBridgeConfig();
        config.device = camera;
        return config;
    }

    
    @ConfigurationParameterChanger(propertyName = "listenSubsystems")
    public void setListenSubsystems(String[] listenSubsystems) {
        for ( String sub : listenSubsystems ) {
            if ( sub == null ) {
                throw new RuntimeException("Illegal value for listened subsystem. It cannot be null.");
            } else if ( sub.isEmpty() ) {
                throw new RuntimeException("Illegal value for listened subsystem. It cannot be empty.");                
            } else if ( sub.contains(" ") ) {
                throw new RuntimeException("Illegal value for listened subsystem. It cannot contain blanks: *"+sub+"*" );                
            }
        }
        this.listenSubsystems = listenSubsystems;
    }
    
    
}
