package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to a real bonn-shutter subsystem.
 *
 * @author tonyj
 */
class BonnShutterSubsystemLayer extends ControlledSubsystem implements ShutterInterface {

    private static final Logger LOG = Logger.getLogger(BonnShutterSubsystemLayer.class.getName());
    private final static Map<Enum<org.lsst.ccs.subsystem.bonnshutter.states.ShutterState>, Enum<ShutterState>> BONNSHUTTER_TO_SHUTTER_STATE = new HashMap<>();

    static {
        BONNSHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.bonnshutter.states.ShutterState.OPEN, ShutterState.OPEN);
        BONNSHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.bonnshutter.states.ShutterState.OPENING, ShutterState.OPENING);
        BONNSHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.bonnshutter.states.ShutterState.CLOSED, ShutterState.CLOSED);
        BONNSHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.bonnshutter.states.ShutterState.CLOSING, ShutterState.CLOSING);
    }

    BonnShutterSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getShutterSubsystemName(), ccs, config);
    }

    @Override
    public void expose(ImageName imageName, Duration exposeTime) throws ExecutionException {
        commandSender.sendCommand("takeExposure", exposeTime.toMillis() / 1000.);
        commandSender.sendCommand("waitForExposure");
    }

    @Override
    public void open(ImageName imageName) throws ExecutionException {
        commandSender.sendCommand("openShutter");
    }

    @Override
    public void close() throws ExecutionException {
        commandSender.sendCommand("closeShutter");
    }

    @Override
    public void prepare() {
        // This is a NOOP for the bonn-shutter, it is always ready
    }
    
    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        String cause = statusChange.getCause();
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateBonnShutterStateToShutterState(when, value, cause);
        });
    }

    private void translateBonnShutterStateToShutterState(CCSTimeStamp when, Enum value, String cause) {
        LOG.log(Level.INFO, "Got bonn-shutter state {0} ", value);
        Enum<ShutterState> converted = BONNSHUTTER_TO_SHUTTER_STATE.get(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted, cause));
        }
    }

    @Override
    public void setImageSequence(boolean imageSequence) {
        // We don't care
    }

}
