package org.lsst.ccs.subsystem.ocsbridge;

import java.io.IOException;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMDirectLayer;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSCommandConverter;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSRevokeAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSimulateFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.OCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.PreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSStateEventConverter;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSStateChangeToEventConverter;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.command.DisableCommand;
import org.lsst.sal.camera.command.EnableCommand;
import org.lsst.sal.camera.command.EnterControlCommand;
import org.lsst.sal.camera.command.ExitControlCommand;
import org.lsst.sal.camera.command.StandbyCommand;
import org.lsst.sal.camera.command.StartCommand;
import org.lsst.sal.camera.command.StartImageCommand;
import org.lsst.sal.camera.command.TakeImagesCommand;
import org.lsst.sal.camera.event.AvailableFiltersEvent;
import org.lsst.sal.camera.event.EndSetFilterEvent;
import org.lsst.sal.camera.event.ErrorCodeEvent;
import org.lsst.sal.camera.event.ConfigurationsAvailableEvent;
import org.lsst.sal.camera.event.ConfigurationAppliedEvent;
import org.lsst.sal.camera.event.StartSetFilterEvent;
import org.lsst.sal.camera.event.LargeFileObjectAvailableEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.CCSCommandStateEvent.CCSCommandState;
import org.lsst.sal.camera.states.SummaryStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent.SummaryState;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraStateChangeEvent;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSEvent;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.subsystem.focalplane.data.ImageMetaDataEvent;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStandbyCommand;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSimulationEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSConfigurationAppliedEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSConfigurationsAvailableEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSImageNameEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.states.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.DelimitedStringSplitJoin;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSCommandConverter.CommandConversionException;
import org.lsst.ccs.subsystem.ocsbridge.util.SummaryInfoConverter;
import org.lsst.sal.camera.command.InitGuidersCommand;
import org.lsst.sal.camera.event.SimulationModeEvent;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSAdditionalFileEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;

/**
 *
 * The Command Layer of the OCS Bridge. Contains executors for all commands
 * supported by the OCSBridge.
 *
 * @author The LSST CCS Team
 */
public class OCSBridge {

    private static final Logger LOG = Logger.getLogger(OCSBridge.class.getName());
    final static DelimitedStringSplitJoin DELIMITED_STRING_SPLIT_JOIN = new DelimitedStringSplitJoin();

    private final CCS ccs;
    private final State lse209State;
    private final State offlineState;
    private OCSCommandExecutor ocsCommandExecutor;
    private CCSCommandExecutor ccsCommandExecutor;
    private final MCMLayer mcm;
    private final OCSBridgeConfig config;

    private final State commandState;

    private final OCSCommandConverter ocsCcsCommandConverter = new OCSCommandConverter();
    private final OCSStateEventConverter ocsCcsStateEventConverter = new OCSStateEventConverter();
    private final OCSStateChangeToEventConverter ocsStateChangeToEventConverter = new OCSStateChangeToEventConverter();
    private final TelemetrySender telemetrySender;
    private final ConfigurationSender configurationSender;
    // Temporary hack for dealing with initGuider commands with limited length (LCOBM-183)
    private StringBuilder roiBuilder = new StringBuilder();
    private final AsynchronousEventHandler eventHandler;
    private final S3FileUploaderAndDeleter s3FileUploader;

    OCSBridge(OCSBridgeConfig config, CCS ccs, MCMLayer mcm) {

        this.config = config;
        this.ccs = ccs;
        this.mcm = mcm;
        this.eventHandler = new AsynchronousEventHandler(config, ccs);
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        lse209State = new State(SummaryState.OFFLINE);
        ccs.getAggregateStatus().add(now, lse209State);
        offlineState = new State(OfflineState.OFFLINE_PUBLISH_ONLY);
        ccs.getAggregateStatus().add(now, offlineState);
        commandState = new State(CCSCommandState.IDLE);
        ccs.getAggregateStatus().add(now, commandState);
        ocsCommandExecutor = new OCSCommandExecutor(this);
        ccsCommandExecutor = new CCSCommandExecutor();
        telemetrySender = TelemetrySender.create(config.getDevice(), ocsCommandExecutor, ccs.getScheduler());
        configurationSender = ConfigurationSender.create(config.getDevice(), ocsCommandExecutor);
        s3FileUploader = new S3FileUploaderAndDeleter(config);

        SALHeartbeatGenerator shg = new SALHeartbeatGenerator(this);
        shg.start();

        mcm.addStateChangeListener((CCSTimeStamp when, Enum state, Enum oldState, String cause) -> {
            LOG.log(Level.INFO, "MCM {0} State changed from {1} to {2}", new Object[]{state.getClass().getName(), oldState, state});
            try {
                if (state instanceof OperationalState) {
                    if (state == OperationalState.ENGINEERING_FAULT && ccs.getAggregateStatus().anyState(SummaryState.ENABLED, SummaryState.DISABLED)) {
                        LOG.log(Level.WARNING, "Switching to FAULT state");
                        lse209State.setState(SummaryState.FAULT, cause);
                        StringBuilder sb = new StringBuilder("MCM has entered fault state");
                        if (cause == null) {
                            sb.append(" for unspecified reason");
                        } else {
                            sb.append(". Cause: ").append(cause);
                        }
                        ErrorCodeEvent ece = new ErrorCodeEvent(1099, sb.toString(), "");
                        ocsCommandExecutor.sendEvent(ece);
                        try {
                            mcm.unlock();
                        } catch (ExecutionException x) {
                            LOG.log(Level.WARNING, "Unable to release MCM lock", x);
                        }
                    }
                }

                // There has to be a better way to do this but this is what I can implement for now - FA
                // Rafts related events
                if ((oldState == RaftsState.CLEARING || oldState == RaftsState.QUIESCENT) && state == RaftsState.INTEGRATING) {
                    LOG.log(Level.INFO, "Detected state transition {0} -> {1}. Getting new image handler.", new Object[]{oldState, state});
                    eventHandler.getNewImageHandler(Duration.ofSeconds(2)).thenAccept(imageHandler -> imageHandler.startIntegration());
                }

                if (oldState == RaftsState.INTEGRATING && state == RaftsState.READING_OUT) {
                    eventHandler.getCurrent().startReadout(when);
                }

                if (oldState == RaftsState.READING_OUT) {
                    eventHandler.getCurrent().endReadout(when);
                }
                CameraEvent converted = ocsStateChangeToEventConverter.convert(when, oldState, state);
                if (converted != null) {
                    ocsCommandExecutor.sendEvent(converted);
                }
            } catch (Exception x) {
                LOG.log(Level.SEVERE, "Error during state change",x);
            }
        });

        // This happens for both the real running and tests. 
        // to do receive CCSJsonFileEvent (from ShutterSimulation)
        mcm.addEventListener((CCSEvent event) -> {
            if (event instanceof CCSAvailableFiltersEvent) {
                List<String> filters = ((CCSAvailableFiltersEvent) event).getAvailableFilters();
                // hard coded a 6 - this is to avert potential problems with a zero length double arrays - I have changed this 
                // to 6 to avert problems - this should not be a permanent solution 
                ocsCommandExecutor.sendEvent(new AvailableFiltersEvent(DELIMITED_STRING_SPLIT_JOIN.join(filters), "", 0., 0., new double[6], new double[6]));
            } else if (event instanceof CCSImageNameEvent) {
                LOG.log(Level.INFO, "CCSImageNameEvent event {0}", event);
                eventHandler.create((CCSImageNameEvent) event, ocsCommandExecutor);
//            } else if (event instanceof CCSMeasuredShutterTime) {
//                CCSMeasuredShutterTime measuredShutterTime = (CCSMeasuredShutterTime) event;
//                LOG.log(Level.INFO, "CCSMeasuredShutterTime event {0}", measuredShutterTime);
//                eventHandler.getHandlerForImage(measuredShutterTime.getImageName()).shutterTimeArrived(measuredShutterTime);
            } else if (event instanceof CCSSetFilterEvent) {
                CCSSetFilterEvent setFilter = (CCSSetFilterEvent) event;
                if (setFilter.isStart()) {
                    ocsCommandExecutor.sendEvent(new StartSetFilterEvent(setFilter.getFilterName(), setFilter.getFilterType()));
                } else {
                    ocsCommandExecutor.sendEvent(new EndSetFilterEvent(setFilter.getFilterName(), setFilter.getFilterType(), setFilter.getSlot(), setFilter.getPosition()));
                }
            } else if (event instanceof CCSConfigurationAppliedEvent) {
                CCSConfigurationAppliedEvent ccsEvent = (CCSConfigurationAppliedEvent) event;
                String otherEvents = configurationSender.getConfigurationEvents();
                ocsCommandExecutor.sendEvent(new ConfigurationAppliedEvent(ccsEvent.getSettings(), makeSemverCompliant(ccsEvent.getVersion()), ccsEvent.getUrl(), ccsEvent.getSchema(), otherEvents));
            } else if (event instanceof CCSConfigurationsAvailableEvent) {
                CCSConfigurationsAvailableEvent ccsEvent = (CCSConfigurationsAvailableEvent) event;
                ocsCommandExecutor.sendEvent(new ConfigurationsAvailableEvent(ccsEvent.getConfigurationsAvailable(), makeSemverCompliant(ccsEvent.getVersion()), ccsEvent.getUrl(), ccsEvent.getSchema()));
            } else if (event instanceof CCSSimulationEvent) {
                CCSSimulationEvent ccsEvent = (CCSSimulationEvent) event;
                ocsCommandExecutor.sendEvent(new SimulationModeEvent(ccsEvent.getSimulationMode()));
            } else if (event instanceof CCSAdditionalFileEvent) {
                AdditionalFile additionalFile = ((CCSAdditionalFileEvent) event).getAdditionalFile();
                LOG.log(Level.INFO, "Received additional file {0}", additionalFile.getFileName());
                try {
                    String minioArchiveURL = s3FileUploader.writeToS3(additionalFile);
                    LOG.log(Level.INFO, "Wrote LFA file to {0}", minioArchiveURL);
                    AdditionalFileToLFAEvent jflfa = new AdditionalFileToLFAEvent(additionalFile, minioArchiveURL);
                    LargeFileObjectAvailableEvent lfa = jflfa.getlfaEvent();
                    // send large file object available event
                    ocsCommandExecutor.sendEvent(lfa);
                } catch (IOException x) {
                    LOG.log(Level.SEVERE, "Unable to send s3 file", x);
                }
            }

        });

        // This only happens when we are not in test mode hence no checks for test mode here. Farrukh 
        mcm.addStatusMessageListener(new EventListener<StatusMessage>() {
            @Override
            public void eventFired(StatusMessage msg) {
                if (msg instanceof StatusConfigurationInfo) {
                    StatusConfigurationInfo sci = (StatusConfigurationInfo) msg;
                    final AgentInfo originAgentInfo = msg.getOriginAgentInfo();
                    ConfigurationInfo info = sci.getConfigurationInfo();
                    if (originAgentInfo != null && "FOCAL_PLANE".equals(originAgentInfo.getAgentProperty("agentCategory"))) {
                        SummaryInfoConverter converter = new SummaryInfoConverter(config.getDevice());
                        ocsCommandExecutor.sendEvent(converter.convert(info));
                    }
                    configurationSender.send(sci);
                } else if (msg instanceof StatusSubsystemData) {
                    StatusSubsystemData ssd = (StatusSubsystemData) msg;
                    final AgentInfo originAgentInfo = msg.getOriginAgentInfo();
                    final String subsystemName = originAgentInfo == null ? "unknown" : originAgentInfo.getName();
                    LOG.log(Level.FINE, "Received from {0} {1}", new Object[]{subsystemName, ssd.getDataKey()});
                    LOG.log(Level.FINE, "Data {0}", new Object[]{ssd.getSubsystemData().getValue().toString()});
                    if (ssd.getSubsystemData().getValue() instanceof org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent) {
                        org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent data = (org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent) ssd.getSubsystemData().getValue();
                        List<String> ccdNames = new ArrayList<>();
                        List<org.lsst.sal.camera.event.ImageReadoutParametersEvent.CcdType> ccdTypes = new ArrayList<>();
                        List<Integer> overCols = new ArrayList<>();
                        List<Integer> overRows = new ArrayList<>();
                        List<Integer> postCols = new ArrayList<>();
                        List<Integer> postRows = new ArrayList<>();
                        List<Integer> preCols = new ArrayList<>();
                        List<Integer> preRows = new ArrayList<>();
                        List<Integer> readCols = new ArrayList<>();
                        List<Integer> readCols2 = new ArrayList<>();
                        List<Integer> readRows = new ArrayList<>();
                        List<Integer> underCols = new ArrayList<>();
                        int index = 0;
                        for (Location location : data.getLocations()) {
                            int nSensors = location.type().getCCDCount();
                            for (int i = 0; i < nSensors; i++) {
                                ccdNames.add(location.getRaftName() + (config.getDevice() == Camera.AUXTEL ? "S00" : location.getSensorName(i)));
                                ccdTypes.add(org.lsst.sal.camera.event.ImageReadoutParametersEvent.CcdType.valueOf(data.getCcdType()[index].toUpperCase()));
                                overCols.add(data.getOverCols()[index]);
                                overRows.add(data.getOverRows()[index]);
                                postCols.add(data.getPostCols()[index]);
                                postRows.add(data.getPostRows()[index]);
                                preCols.add(data.getPreCols()[index]);
                                preRows.add(data.getPreRows()[index]);
                                readCols.add(data.getReadCols()[index]);
                                readCols2.add(data.getReadCols2()[index]);
                                readRows.add(data.getReadRows()[index]);
                                underCols.add(data.getUnderCols()[index]);
                            }
                            index++;
                        }
                        CameraEvent irpe;
                        if (config.getDevice() == Camera.AUXTEL) {
                            irpe = org.lsst.sal.atcamera.event.ImageReadoutParametersEvent.builder()
                                    .imageName(data.getImageName().toString())
                                    .ccdLocation(DELIMITED_STRING_SPLIT_JOIN.join(ccdNames))
                                    .raftBay(ccdNames.stream().map(name -> name.substring(0, 3)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                    .ccdSlot(ccdNames.stream().map(name -> name.substring(3, 6)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                    .ccdType(org.lsst.sal.atcamera.event.ImageReadoutParametersEvent.CcdType.ITL)
                                    .overRows(overRows.get(0))
                                    .overCols(overCols.get(0))
                                    .readRows(readRows.get(0))
                                    .readCols(readCols.get(0))
                                    .readCols2(readCols2.get(0))
                                    .preCols(preCols.get(0))
                                    .preRows(preRows.get(0))
                                    .postCols(postCols.get(0))
                                    .underCols(underCols.get(0))
                                    .daqFolder(data.getDaqFolder())
                                    .daqAnnotation(data.getAnnotation())
                                    .build();
                        } else {
                            irpe = org.lsst.sal.camera.event.ImageReadoutParametersEvent.builder()
                                    .imageName(data.getImageName().toString())
                                    .ccdLocation(DELIMITED_STRING_SPLIT_JOIN.join(ccdNames))
                                    .raftBay(ccdNames.stream().map(name -> name.substring(0, 3)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                    .ccdSlot(ccdNames.stream().map(name -> name.substring(3, 6)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                    .ccdType(ccdTypes.stream().toArray(org.lsst.sal.camera.event.ImageReadoutParametersEvent.CcdType[]::new))
                                    .overRows(overRows.stream().mapToInt(i -> i).toArray())
                                    .overCols(overCols.stream().mapToInt(i -> i).toArray())
                                    .readRows(readRows.stream().mapToInt(i -> i).toArray())
                                    .readCols(readCols.stream().mapToInt(i -> i).toArray())
                                    .readCols2(readCols2.stream().mapToInt(i -> i).toArray())
                                    .preCols(preCols.stream().mapToInt(i -> i).toArray())
                                    .preRows(preRows.stream().mapToInt(i -> i).toArray())
                                    .postCols(postCols.stream().mapToInt(i -> i).toArray())
                                    .underCols(underCols.stream().mapToInt(i -> i).toArray())
                                    .daqFolder(data.getDaqFolder())
                                    .daqAnnotation(data.getAnnotation())
                                    .build();
                        }

                        ocsCommandExecutor.sendEvent(irpe);
                    } else if (ssd.getSubsystemData().getValue() instanceof ImageMetaDataEvent) {
                        ImageMetaDataEvent data = (ImageMetaDataEvent) ssd.getSubsystemData().getValue();
                        eventHandler.getHandlerForImage(data.getImageName()).imageMetaDataArrived(data);
                    } else if (ssd.getSubsystemData().getValue() instanceof AdditionalFile) {
                        AdditionalFile additionalFile = (AdditionalFile) ssd.getSubsystemData().getValue();
                        LOG.log(Level.INFO, "Received additional file {0}", additionalFile.getFileName());
                        try {
                            String writtenMinioURL = s3FileUploader.writeToS3(additionalFile);
                            LOG.log(Level.INFO, "Wrote LFA file to {0}", writtenMinioURL);
                            AdditionalFileToLFAEvent jflfa = new AdditionalFileToLFAEvent(additionalFile, writtenMinioURL);
                            LargeFileObjectAvailableEvent lfa = jflfa.getlfaEvent();
                            // send large file object available event
                            ocsCommandExecutor.sendEvent(lfa);
                        } catch (IOException x) {
                            LOG.log(Level.SEVERE, "Unable to send s3 file", x);
                        }
                    } else if (ssd.getSubsystemData().getValue() instanceof MeasuredShutterTime) {
                        MeasuredShutterTime measuredShutterTime = (MeasuredShutterTime) ssd.getSubsystemData().getValue();
                        LOG.log(Level.INFO, "MeasuredShutterTime event {0}", measuredShutterTime);
                        eventHandler.getHandlerForImage(measuredShutterTime.getImageName()).shutterTimeArrived(measuredShutterTime);
                    } else if (ssd.getSubsystemData().getValue() instanceof List) {
                        telemetrySender.send(ssd);
                    } else {
                        LOG.log(Level.FINE, "Got unhandled telemetry from {0} of type ", new Object[]{subsystemName, ssd.getClass().getSimpleName()});
                    }
                }
            }
        });

        ccs.addStateChangeListener((when, currentState, oldState, cause) -> {

            CameraStateChangeEvent converted = ocsCcsStateEventConverter.convert(when, currentState);
            if (converted == null) {
                LOG.log(Level.WARNING, "Enum {0} of class {1} could not be converted to equivalent OCS state", new Object[]{currentState, currentState.getClass()});
            } else {
                ocsCommandExecutor.sendEvent(converted);
            }
        });
    }

    State<CCSCommandState> getCommandState() {
        return commandState;
    }

    public OCSBridgeConfig getConfig() {
        return config;
    }

    void addSubsystem(AgentInfo ai, DataProviderDictionary dict) {
        telemetrySender.getConverter().addSubsystem(ai, dict);
        configurationSender.getConverter().addSubsystem(ai, dict);
    }

    void removeSubsystem(String subsystemName) {
        telemetrySender.getConverter().removeSubsystem(subsystemName);
        configurationSender.getConverter().removeSubsystem(subsystemName);
    }

    /**
     * Allow a user to provide an alternative implementation of the
     * OCSCommandExecutor. Used to override the default OCSCommandExecutor with
     * one that actually sends acknowledgments back to OCS.
     *
     * @param ocs
     */
    void setOCSCommandExecutor(OCSCommandExecutor ocs) {
        this.ocsCommandExecutor = ocs;
        this.telemetrySender.setSender(ocs);
        this.configurationSender.setSender(ocs);
        // Publish the current state
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        ccs.getAggregateStatus().getStates().stream()
                .map((s) -> ocsCcsStateEventConverter.convert(now, s.getState()))
                .filter(Objects::nonNull)
                .forEach((converted) -> ocsCommandExecutor.sendEvent(converted));
    }

    public OCSCommandExecutor getOcsCommandExecutor() {
        return ocsCommandExecutor;
    }

    void setCCSCommandExecutor(CCSCommandExecutor ccs) {
        this.ccsCommandExecutor = ccs;
    }

    static OCSBridge createOCSBridge(MCMConfig mcmConfig) {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig(mcmConfig.getCameraType());
        ImageNameService ins = ImageNameService.testInstance("jdbc:h2:mem:test;MODE=MYSQL", Controller.OCS, Source.MainCamera, "UTC", Duration.ofHours(12), Instant.parse("2022-01-01T00:00:00.00Z"));
        return createOCSBridge(ocsConfig, mcmConfig, ins);
    }

    static OCSBridge createOCSBridge(Camera camera) {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig(camera);
        MCMConfig mcmConfig = new MCMConfig(camera);
        ImageNameService ins = ImageNameService.testInstance("jdbc:h2:mem:test;MODE=MYSQL", Controller.OCS, camera == Camera.AUXTEL ? Source.AuxTel : Source.MainCamera, "UTC", Duration.ofHours(12), Instant.parse("2022-01-01T00:00:00.00Z"));
        return createOCSBridge(ocsConfig, mcmConfig, ins);
    }

    static OCSBridge createOCSBridge(OCSBridgeConfig config, MCMConfig mcmConfig, ImageNameService imageNamingService) {
        CCS ccs = new CCS();
        MCMLayer mcm = new MCMDirectLayer(new MCM(ccs, mcmConfig, imageNamingService));
        return new OCSBridge(config, ccs, mcm);
    }

    State getLse209State() {
        return lse209State;
    }

    State getOfflineState() {
        return offlineState;
    }

    private String makeSemverCompliant(int version) {
        return version + ".0.0"; // CAP-929
    }

    /**
     * Super class for all OCSExecutors which forward commands to MCM
     */
    class ForwardToMCMExecutor extends OCSCommandExecutor.OCSExecutor {

        private CCSCommandResponse response;
        private final SummaryState initialState;
        private final CameraCommand command;
        private CCSCommand ccsCommand;

        ForwardToMCMExecutor(CameraCommand command) {
            this(command, SummaryStateEvent.SummaryState.ENABLED);
        }

        ForwardToMCMExecutor(CameraCommand command, SummaryState initialState) {
            super(command);
            this.command = command;
            this.initialState = initialState;
        }

        @Override
        Duration testPreconditions() throws OCSCommandExecutor.PreconditionsNotMet {
            if (!lse209State.isInState(initialState)) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command not accepted in: " + lse209State);
            }
            try {
                this.ccsCommand = ocsCcsCommandConverter.convert(command);

                CCSAckOrNack can = forwardCommand();
                if (can.isNack()) {
                    throw new OCSCommandExecutor.PreconditionsNotMet("Command rejected: " + can.getReason());
                } else {
                    return can.getDuration();
                }
            } catch (CommandConversionException x) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command rejected: " + x.getMessage());
            }
        }

        /**
         * This is somewhat illogically executed from testPreconditions to give
         * the MCM the chance to reject the command. Override this if there is
         * anything that needs to be done before the command executes.
         *
         * @return
         */
        protected CCSAckOrNack forwardCommand() {
            response = mcm.execute(ccsCommand);
            return response.waitForAckOrNack();
        }

        @Override
        void execute() throws Exception {
            response.waitForCompletion();
        }
    }

    // 
    void execute(CameraCommand cmd) {
        if (cmd instanceof DisableCommand) {
            execute((DisableCommand) cmd);
        } else if (cmd instanceof EnableCommand) {
            execute((EnableCommand) cmd);
        } else if (cmd instanceof ExitControlCommand) {
            execute((ExitControlCommand) cmd);
        } else if (cmd instanceof EnterControlCommand) {
            execute((EnterControlCommand) cmd);
            // The next three commands override the execute method of the parent class 
            // and have to be treated as special cases and cannot be generalised to
            // be dealt with via ForwardToMCMExecutor followed by execute - we 
            // need to generalise these differently - coming soon - Farrukh 25/6/2018
        } else if (cmd instanceof TakeImagesCommand) {
            execute((TakeImagesCommand) cmd);
        } else if (cmd instanceof StartCommand) {
            execute((StartCommand) cmd);
        } else if (cmd instanceof StartImageCommand) {
            execute((StartImageCommand) cmd);
        } else if (cmd instanceof InitGuidersCommand) {
            execute((InitGuidersCommand) cmd);
        } else if (cmd instanceof StandbyCommand) {
            execute((StandbyCommand) cmd);
        } // rest of camera commands are generalizeable in the following manner - Farrukh 25/6/2018
        else {
            ForwardToMCMExecutor fcm = new ForwardToMCMExecutor(cmd);
            ocsCommandExecutor.executeCommand(fcm);
        }
    }

    void execute(InitGuidersCommand command) {
        String roiSpec = command.getRoiSpec();
        LOG.log(Level.INFO,"Got ROISpec {0}", roiSpec);
        if (roiSpec.endsWith("-")) {
            roiBuilder.append(roiSpec.substring(0, roiSpec.length() - 1));
            OCSExecutor noop = new NoOpExecutor();
            ocsCommandExecutor.executeCommand(noop);
        } else if (roiSpec.length() > 0) {
            roiBuilder.append(roiSpec);
            command = new InitGuidersCommand(roiBuilder.toString());
            OCSExecutor initGuiders = new InitGuidersExecutor(command);
            roiBuilder.setLength(0);
            ocsCommandExecutor.executeCommand(initGuiders);
        } else {
            OCSExecutor initGuiders = new InitGuidersExecutor(command);
            ocsCommandExecutor.executeCommand(initGuiders);
        }
    }

    void execute(TakeImagesCommand command) {
        OCSExecutor takeImages = new TakeImagesExecutor(command);
        ocsCommandExecutor.executeCommand(takeImages);
    }

    void execute(StartImageCommand command) {
        OCSExecutor startImage = new StartImageExecutor(command);
        ocsCommandExecutor.executeCommand(startImage);
    }

    void execute(EnterControlCommand command) {
        OCSExecutor takeControl = new EnterControlExecutor(command);
        ocsCommandExecutor.executeCommand(takeControl);
    }

    void execute(ExitControlCommand command) {
        OCSExecutor exit = new ExitExecutor(command);
        ocsCommandExecutor.executeCommand(exit);
    }

    void execute(StartCommand command) {
        OCSExecutor start = new StartExecutor(command);
        ocsCommandExecutor.executeCommand(start);
    }

    void execute(StandbyCommand command) {
        OCSExecutor standby = new StandbyExecutor(command);
        ocsCommandExecutor.executeCommand(standby);
    }

    void execute(EnableCommand command) {
        OCSExecutor enable = new EnableExecutor(command);
        ocsCommandExecutor.executeCommand(enable);
    }

    void execute(DisableCommand command) {
        OCSExecutor disable = new DisableExecutor(command);
        ocsCommandExecutor.executeCommand(disable);
    }

    void execute(CCSCommand command) {
        if (command instanceof CCSSetAvailableCommand) {
            execute((CCSSetAvailableCommand) command);
        } else if (command instanceof CCSRevokeAvailableCommand) {
            execute((CCSRevokeAvailableCommand) command);
        } else if (command instanceof CCSSimulateFaultCommand) {
            execute((CCSSimulateFaultCommand) command);
        } else if (command instanceof CCSClearFaultCommand) {
            execute((CCSClearFaultCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    void execute(CCSSetAvailableCommand command) {
        CCSExecutor setAvailable = new SetAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(setAvailable));
    }

    void execute(CCSRevokeAvailableCommand command) {
        CCSExecutor revokeAvailable = new RevokeAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(revokeAvailable));
    }

    void execute(CCSSimulateFaultCommand command) {
        CCSExecutor simulateFault = new SimulateFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(simulateFault));
    }

    void execute(CCSClearFaultCommand command) {
        CCSExecutor clearFault = new ClearFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(clearFault));
    }

    // Utility only used for testing   
    public Future<Void> waitForState(Enum state) {
        return ccs.waitForStatus(state);
    }

    CCS getCCS() {
        return ccs;
    }

    class InitGuidersExecutor extends ForwardToMCMExecutor {

        public InitGuidersExecutor(InitGuidersCommand command) {
            super(command);
        }
    }

    class TakeImagesExecutor extends ForwardToMCMExecutor {

        public TakeImagesExecutor(TakeImagesCommand command) {
            super(command);
        }

        @Override
        protected CCSAckOrNack forwardCommand() {
            return super.forwardCommand();
        }
    }

    class StartImageExecutor extends ForwardToMCMExecutor {

        public StartImageExecutor(StartImageCommand command) {
            super(command);

        }

        @Override
        protected CCSAckOrNack forwardCommand() {
            return super.forwardCommand();
        }
    }

    class NoOpExecutor extends OCSExecutor {

        public NoOpExecutor() {
            super(null);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {

        }
    }

    class EnterControlExecutor extends OCSExecutor {

        public EnterControlExecutor(EnterControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_AVAILABLE)) {
                throw new PreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            // First we need to lock the MCM
            mcm.lock();

            // We send the standby command to the MCM, so that it will publish
            // the configurations available event, and lock subsystems. 
            CCSCommandResponse response = mcm.execute(new CCSStandbyCommand());
            CCSAckOrNack ackOrNack = response.waitForAckOrNack();
            if (ackOrNack.isNack()) {
                throw new RuntimeException("Standby command rejected: " + ackOrNack.getReason());
            }
            response.waitForCompletion();
            lse209State.setState(SummaryState.STANDBY);
        }
    }

    class ExitExecutor extends OCSExecutor {

        public ExitExecutor(ExitControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.STANDBY)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            mcm.unlock();
            lse209State.setState(SummaryState.OFFLINE);
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }
    }

    /**
     * Start command is a special case, since it is a lifecycle command, but
     * must also issue a start command to the MCM (to set configuration).
     */
    class StartExecutor extends ForwardToMCMExecutor {

        public StartExecutor(StartCommand command) {
            super(command, SummaryState.STANDBY);
        }

        @Override
        void execute() throws Exception {
            super.execute();
            lse209State.setState(SummaryState.DISABLED);
        }
    }

    class StandbyExecutor extends ForwardToMCMExecutor {

        public StandbyExecutor(StandbyCommand command) {
            super(command, SummaryState.DISABLED);
        }

        @Override
        void execute() throws Exception {
            super.execute();
            lse209State.setState(SummaryState.STANDBY);
        }
    }

    class EnableExecutor extends OCSExecutor {

        public EnableExecutor(EnableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(SummaryState.ENABLED);
        }
    }

    class DisableExecutor extends OCSExecutor {

        public DisableExecutor(DisableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.ENABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            // Fixme: Can we reject the disable command if we are busy?
            // What about if we are not idle?
            // Note logic here is incorrect according to Paul Lotz, we must always accept
            // the disable command.
//            if (startImageTimeout != null && !startImageTimeout.isDone()) {
//                throw new PreconditionsNotMet("Exposure in progress");
//            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standby command if things are happening?
            //TODO: or wait until things finish and return then?
            lse209State.setState(SummaryState.DISABLED);
        }
    }

    class SetAvailableExecutor extends CCSExecutor {

        private SetAvailableExecutor(CCSSetAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_PUBLISH_ONLY)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.OFFLINE_AVAILABLE);
        }

    }

    class RevokeAvailableExecutor extends CCSExecutor {

        private RevokeAvailableExecutor(CCSRevokeAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_AVAILABLE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }

    }

    class SimulateFaultExecutor extends CCSExecutor {

        private SimulateFaultExecutor(CCSSimulateFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            //TODO: Should we also attempt to stop the subsystems?
            lse209State.setState(SummaryState.FAULT, "Executed CCSSimulateFaultCommand");
        }

    }

    class ClearFaultExecutor extends CCSExecutor {

        private ClearFaultExecutor(CCSClearFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.FAULT)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            lse209State.setState(SummaryState.OFFLINE);
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }
    }
}
