package org.lsst.ccs.subsystem.ocsbridge.util;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.camera.sal.xml.MakeXMLConfiguration;
import org.lsst.ccs.camera.sal.xml.MakeXMLConfiguration.DictionaryConfiguration;
import org.lsst.ccs.camera.sal.xml.Mapping;
import org.lsst.ccs.camera.sal.classes.SALClassDescription;
import org.lsst.ccs.camera.sal.classes.SALClassDescription.BusVariable;
import org.lsst.ccs.camera.sal.classes.SALClassDescription.LocationVariable;
import org.lsst.ccs.camera.sal.classes.SALClassDescription.SALVariable;
import org.lsst.ccs.camera.sal.classes.SALClassDescriptionMaker;
import org.lsst.ccs.camera.sal.xml.XMLMaker2.SALType;

/**
 *
 * @author Farrukh Azfar
 */
public class GenericConverter {

    private static final Logger LOG = Logger.getLogger(GenericConverter.class.getName());

    //keep a record of when data is missing on the bus ...
    private final List<String> mia = new ArrayList<>();
    private final Map<Class, List<String>> alreadyWarned = new HashMap<>();
    private final List<String> alreadyWarnedTelemetrySubsystems = new ArrayList<>();
    private final List<String> alreadyWarnedConfigSubsystems = new ArrayList<>();
    private final List<String> subsystemsToConvert = new ArrayList<>();
    private final List<String> componentNames = new ArrayList<>();
    private final Mapping mapping = Mapping.defaultMapping();
    private final SALType salType;
    private final Camera camera;

    private final Map<String, SALClassDescriptionMaker> salClassDescriptionMakerMap = new ConcurrentHashMap<>();

    // change following to false for debugging 
    private static final boolean logAndContinueOnConversionError = true;
    boolean badBValue = false;
    double badDValue = Double.NaN;
    int badIValue = -111111111;
    float badFValue = Float.NaN;
    String badStringValue = "NOTFOUND";
    long badLValue = -111111111;
    short badSValue = (short) -11111;

    public GenericConverter(Camera camera, SALType salType) {
        this.salType = salType;
        this.camera = camera;
    }

    public boolean addSubsystem(String subsystemName) {
        Optional<DictionaryConfiguration> opt = getDictionaryConfgurationForSubsystemName(subsystemName);
        if (opt.isPresent()) {
            return addSubsystem(subsystemName,opt.get(), opt.get().getLevel());
        }
        return false;
    }

    //For tests only
    public boolean addSubsystem(String subsystemName, int level) {
        Optional<DictionaryConfiguration> opt = getDictionaryConfgurationForSubsystemName(subsystemName);
        if (opt.isPresent()) {
            return addSubsystem(subsystemName, opt.get(), level);
        }
        return false;
    }

    public boolean addSubsystem(AgentInfo ai, DataProviderDictionary dict) {
        Optional<DictionaryConfiguration> opt = getDictionaryConfgurationForSubsystemName(ai, dict);
        if (opt.isPresent()) {
            DictionaryConfiguration dictConfig = opt.get();
            return addSubsystem(ai.getName(), dictConfig, dictConfig.getLevel());
        }
        return false;
    }

    private boolean addSubsystem(String subsystemName, DictionaryConfiguration dictConfig, int level) {
        SALClassDescriptionMaker maker = new SALClassDescriptionMaker(dictConfig, mapping, level);
        salClassDescriptionMakerMap.put(subsystemName, maker);
        subsystemsToConvert.add(subsystemName);
        componentNames.add(dictConfig.getComponentName());
        return true;
    }

    public void removeSubsystem(String subsystemName) {
        SALClassDescriptionMaker maker = salClassDescriptionMakerMap.remove(subsystemName);
        subsystemsToConvert.remove(subsystemName);
        if (maker != null) {
            componentNames.remove(maker.getDictionaryConfiguration().getComponentName());
        }
    }

    private Optional<DictionaryConfiguration> getDictionaryConfgurationForSubsystemName(AgentInfo agentInfo, DataProviderDictionary dict) {
//        DictionaryConfiguration dictConfig = MakeXMLConfiguration.getInstance(camera, salType), agentInfo, dict).getDictionaryConfigurationForAgentInfo(agentInfo);
        DictionaryConfiguration dictConfig = MakeXMLConfiguration.getInstance(camera, salType).getDictionaryConfigurationForAgentInfo(agentInfo);
        if (dictConfig != null) {
            return Optional.of(dictConfig);
        }
        return Optional.empty();
    }

    private Optional<DictionaryConfiguration> getDictionaryConfgurationForSubsystemName(String subsystemName) {
        DictionaryConfiguration dictConfig = MakeXMLConfiguration.getInstance(camera, salType).getDictionaryConfigurationForAgent(subsystemName);
        if (dictConfig != null) {
            return Optional.of(dictConfig);
        }
        return Optional.empty();
    }

    public Map<String, Class> getClassMap() {
        final Map<String, Class> classMap = new HashMap<>();
        for (SALClassDescriptionMaker maker : salClassDescriptionMakerMap.values()) {
            classMap.putAll(maker.getSALClasses());
        }
        return classMap;
    }

    public Map<String, SALClassDescription> getSALClassDescriptionMapForAgent(String agentName) {
        return salClassDescriptionMakerMap.get(agentName).getSALClassDescriptions();
    }

    public synchronized List<CameraTelemetry> telemetryConverter(StatusSubsystemData data) throws ReflectiveOperationException {
        // extract subsystem name  and get data

        String subsystemName = data.getOriginAgentInfo().getName();
        KeyValueDataList subsystemData = (KeyValueDataList) data.getEncodedData();
        if (subsystemData == null) {
            return Collections.<CameraTelemetry>emptyList();
        }
        List<KeyValueData> values = (List<KeyValueData>) subsystemData.getValue();
        // data is contained in the list directly above ...

        Map<String, Object> dataNamesAndValues = new HashMap<>();

        LOG.log(Level.FINEST, " GenericConverter:telemetryConverter: Start Logging");

        for (KeyValueData kvd : values) {
            String pathFromBus = kvd.getKey();

            // For now we ignore the states in the passed in telemetry
            //TO-DO: IS THIS NEEDED?
            if (!pathFromBus.contains("/state")) {

                // fill in bus data ...
                dataNamesAndValues.put(pathFromBus, kvd.getValue());
                LOG.log(Level.FINEST, () -> {
                    StringBuilder sb = new StringBuilder();
                    sb.append("GenericConverter:settingsAppliedConverter: Variable name from Bus = ").append(kvd.getKey()).append(", Variable type from bus  = ").append(kvd.getType()).append(", Variable value from bus = ").append(kvd.getValue()).append("\n");
                    return sb.toString();
                });

            }

        }

        Map<String, Map<String, Object>> trendingData = new HashMap<>();
        //TO-DO We need this try-catch block because some of the serialized files
        //are too old. At some point we might want to remove it.
        try {
            if (subsystemData.getAttribute("taskName") == null) {
                trendingData.put("Trending", dataNamesAndValues);
            } else {
                trendingData.put("", dataNamesAndValues);
            }
        } catch (NullPointerException npe) {
            trendingData.put("", dataNamesAndValues);
        }

        return telemetryConverter(subsystemName, trendingData);

    }

    List<CameraTelemetry> telemetryConverter(String subsystemName, Map<String, Map<String, Object>> trendingData) throws ReflectiveOperationException {

        if (!salType.equals(SALType.TELEMETRY)) {
            throw new RuntimeException("GenericConverter:telemetryConverter: Cannot invoke telemetry conversion for type converter of type " + salType);
        }
        ArrayList<CameraTelemetry> salTelemetry = new ArrayList<>();

        for (Entry<String, Map<String, Object>> e : trendingData.entrySet()) {
            Map<String, Object> dataNamesAndValues = e.getValue();

            if (!subsystemsToConvert.contains(subsystemName)) {
                if (!alreadyWarnedTelemetrySubsystems.contains(subsystemName)) {
                    alreadyWarnedTelemetrySubsystems.add(subsystemName);

                    LOG.log(Level.WARNING, () -> {
                        StringBuilder sb = new StringBuilder();
                        sb.append(" GenericConverter: telemetryConverter: returning Empty List of CameraTelemetry , Current susbsystem : ").append(subsystemName).append(", Contents of subsystemsToConvert ").append(subsystemsToConvert.toString());
                        return sb.toString();
                    });
                }
                return Collections.<CameraTelemetry>emptyList();
            }

            Map<String, SALClassDescription> givenSalClassMapInfo = salClassDescriptionMakerMap.get(subsystemName).getSALClassDescriptions();

            LOG.log(Level.FINEST, "GenericConverter:telemetryConverter: SalClassMap printout Follows : ");
            givenSalClassMapInfo.entrySet().forEach((Entry<String, SALClassDescription> scd) -> {
                LOG.log(Level.FINEST, () -> {
                    StringBuilder sb = new StringBuilder();
                    sb.append("GenericConverter:settingsAppliedConverter: Variable name from SALClassDescription = ").append(scd.getKey()).append(", Complete SALClassDescription = ").append(scd.getValue()).append("\n");
                    return sb.toString();
                });
            });
            
            LOG.log(Level.FINEST, "GenericConverter:telemetryConverter: SalClassMap printout Ends : ");

            for (Map.Entry<String, SALClassDescription> scd : givenSalClassMapInfo.entrySet()) {
                String scdSALClassName = scd.getKey();

                // Comment added by Farrukh : Max should check if this is correct
                // (Change by Max): this ensures that we do not loop twice over each class 
                // the outer loop is over categories the inner one over all classes 
                // this if block checks to see that SALClass is associated with the current
                // category in the outer loop this ensures that we create each class only once
                if (!scd.getValue().getCategory().equals(e.getKey())) {
                    continue;
                }

                LOG.log(Level.FINEST, " GenericConverter:telemetryConverter: Class Name from SALClassDescription {0}", scdSALClassName);

                // actual SAL Class ...
                Class realSALClass = scd.getValue().getSimpleSalClass();
                if (realSALClass == null) {
                    LOG.log(Level.WARNING, " GenericConverter:telemetryConverter: No entry found for {0}", scdSALClassName);
                }

                // if the subsystemName from this batch of telemetry data is one of the subsystems we've been requested and if
                // the real class is not null we proceed. 
                if (realSALClass != null) {

                    // get the parameters of the first constructor - (there could be more so potentially this is something to look out for)    
                    Parameter[] parameters = realSALClass.getConstructors()[0].getParameters();

                    CameraTelemetry ct = (CameraTelemetry) reflectedClass(realSALClass, dataNamesAndValues, parameters, scd.getValue(), null);

                    LOG.log(Level.FINEST, " GenericConverter:telemetryConverter: Class Received {0}", ct.toString());

                    int check[] = checkClassCompleteness(ct);
                    int numMissingValues = check[0];
                    int numGetMethods = check[1];
                    int numberOfBooleanGets = check[2];

                    LOG.log(Level.FINEST, " GenericConverter:telemetryConverter: Class quality check Follows ");
                    LOG.log(Level.FINEST, " GenericConverter:telemetryConverter: class name {0}, Number of missing values {1}", new Object[]{ct.getClass().getName(), check[0]});
                    LOG.log(Level.FINEST, " GenericConverter:telemetryConverter: Number of get functions {0}, Number of boolean ''is'' Functions {1}", new Object[]{check[1], check[2]});

                    //TO-DO for Farrukh: figure out why missingValues == numGetMethods for telemetry data
                    //or decide to use new method on SALClassDescription
                    boolean gets = false;
                    boolean bools = false;
                    gets = numMissingValues < numGetMethods;
                    bools = numberOfBooleanGets > 0;
                    //if (!(numMissingValues == numGetMethods)) 
                    if (gets || (bools && numGetMethods == 0)) {
                        salTelemetry.add(ct);
                    }
                }

            }
        }

        LOG.log(Level.FINEST, " GenericConverter:telemetryConverter : Number of classes in SAL Telemetry {0}", salTelemetry.size());
        LOG.log(Level.FINEST, "GenericConverter:telemetryConverter: End Logging ");

        return salTelemetry;
    }

    public List<CameraEvent> settingsAppliedEventConverter(StatusConfigurationInfo sdata) throws ReflectiveOperationException {
        ConfigurationInfo data = sdata.getConfigurationInfo();

        List<ConfigurationParameterInfo> cpinfo = data.getAllParameterInfo();

        LOG.log(Level.FINEST, " GenericConverter:settingsAppliedConverter Start Logging ");

        LOG.log(Level.FINEST, "GenericConverter:settingsAppliedConverter: Data from Bus Message ****");

        LOG.log(Level.FINEST, () -> {
            StringBuilder sb = new StringBuilder();
            for (ConfigurationParameterInfo dinfo : data.getAllParameterInfo()) {
                sb.append("GenericConverter:settingsAppliedConverter: Category from Bus : ").append(dinfo.getCategoryName()).append(" ,Data from bus: ").append(dinfo.getPathName()).append(" = ").append(dinfo.getCurrentValueObject()).append("\n");
            }
            return sb.toString();
        });

        // subsystemsToConvert doesn't contain the given subsystem 
        String subsystemName = sdata.getOriginAgentInfo().getName();
        if (!subsystemsToConvert.contains(subsystemName)) {

            if (!alreadyWarnedConfigSubsystems.contains(subsystemName)) {
                alreadyWarnedConfigSubsystems.add(subsystemName);
                LOG.log(Level.WARNING, () -> {
                    StringBuilder sb = new StringBuilder();
                    sb.append("GenericConverter:settingsAppliedConverter: Returning Empty List of CameraEvents, Current susbsystem = ").append(subsystemName).append(" Contents of subsystemsToConvert :").append(subsystemsToConvert.toString());
                    return sb.toString();
                });
            }

            return Collections.<CameraEvent>emptyList();
        }

        Map<String, Map<String, Object>> categoryDataMap = new HashMap<>();
        for (ConfigurationParameterInfo cpi : cpinfo) {
            String category = cpi.getCategoryName();
            Map<String, Object> dataMapForCategory = categoryDataMap.computeIfAbsent(category, (c) -> new HashMap<>());
            dataMapForCategory.put(cpi.getPathName(), cpi.getCurrentValueObject());
        }

        Map<String, String> catVersionMap = new HashMap<>();
        for (String category : data.getCategorySet()) {
            catVersionMap.put(category, data.getConfigurationDescriptionObject().getCategoryTag(category).toString());
        }

        LOG.log(Level.FINEST, " GenericConverter:settingsAppliedConverter: End Logging ");
        return settingsAppliedEventConverter(subsystemName, categoryDataMap, catVersionMap);
    }

    List<CameraEvent> settingsAppliedEventConverter(String subsystemName, Map<String, Map<String, Object>> categoryDataMap, Map<String, String> categoryVersionMap) throws ReflectiveOperationException {

        LOG.log(Level.FINEST, " GenericConverter:settingsAppliedConverter: Start Logging ");

        if (!salType.equals(SALType.SETTINGS_APPLIED)) {
            throw new RuntimeException("GenericConverter:settingsAppliedEventConverter: Cannot invoke events conversion for type converter of type " + salType);
        }

        // return a list containing all the classes created here 
        ArrayList<CameraEvent> salSettings = new ArrayList<>();

        // local copy of salclassmapinfo passed in constructor not sure why I'm doing this 
        Map<String, SALClassDescription> salClassMapInfo = salClassDescriptionMakerMap.get(subsystemName).getSALClassDescriptions();

        LOG.log(Level.FINEST, "GenericConverter:settingsAppliedConverter: SALClassDescription information Follows ");
        for (Map.Entry<String, SALClassDescription> scd : salClassMapInfo.entrySet()) {
            String scdSALClassName;
            scdSALClassName = scd.getKey();
            LOG.log(Level.FINEST, () -> {
                StringBuilder sb = new StringBuilder();
                sb.append(" GenericConverter: settingsAppliedConverter: Class Name from SALClassDescription: ").append(scdSALClassName).append(", Printout of SALClassDescription ").append(scd.getValue().toString());
                return sb.toString();
            });
        }

        /*
         * Loop over all of the entries in the salClassMapInfo to see which categories they contain
         */
        for (Map.Entry<String, SALClassDescription> sc : salClassMapInfo.entrySet()) {
            String scCategory = sc.getValue().getCategory();
            Map<String, Object> thisCategoriesDataNew = categoryDataMap.get(scCategory);
            if (thisCategoriesDataNew == null) {
                continue;
            }
            // Check that the subsystem on the buses matches the subsystem from salClassMapInfo
            // This code accepts too much for subsystems like ats
            String componentName = componentNames.get(subsystemsToConvert.indexOf(subsystemName));
            if (!sc.getKey().contains("_" + componentName + "_")) {
                continue;
            }

            // I've apparently filled in scCategory 
            if (scCategory != null && scCategory != "null" && !scCategory.isEmpty()) {

                // now get the sal class                 
                Class realSALClass = sc.getValue().getSimpleSalClass();

                //LCOBM-177 skip conversion of timers category for the SAL class affected by this bug
                //Remove this code once LCOBM-177 is fixed.
                if ( realSALClass == MTCamera.logevent_rebpower_Power_timersConfiguration.class ) {
                    continue;
                }
                
                // the check on realSALClass seems to be sufficient 
                if (realSALClass != null) {

                    if (thisCategoriesDataNew.isEmpty()) {
                        continue;
                    }
                    // thisCategoriesDataNew is the same as local settings data 
                    Parameter[] parameters = realSALClass.getConstructors()[0].getParameters();

                    Method[] premethods = realSALClass.getMethods();
                    LOG.log(Level.FINEST, " GenericConverter:settingsAppliedConverter: Preliminary printout of methods and parameter names ");

                    for (Method premethod : premethods) {
                        LOG.log(Level.FINEST, () -> {
                            StringBuilder sb = new StringBuilder();
                            sb.append(" GenericConverter:settingsAppliedConverter: Method name of reflected class before constructor invocation- compare to whats on the bus: ").append(premethod.getName());
                            return sb.toString();
                        });
                    }
                    for (Parameter parameter : parameters) {
                        LOG.log(Level.FINEST, () -> {
                            StringBuilder sb = new StringBuilder();
                            sb.append(" GenericConverter:settingsAppliedConverter: Parameter name of reflected class before constructor invocation- compare to whats on the bus: ").append(parameter.getName());
                            return sb.toString();
                        });
                    }

                    String categoryVersion = categoryVersionMap.get(scCategory);

                    try {
                        CameraEvent ce = (CameraEvent) reflectedClass(realSALClass, thisCategoriesDataNew, parameters, sc.getValue(), categoryVersion);

                        LOG.log(Level.FINEST, () -> {
                            StringBuilder sb = new StringBuilder();
                            sb.append(" GenericConverter:settingsAppliedConverter: Class Received : ").append(ce.toString());
                            return sb.toString();
                        });

                        int check[] = checkClassCompleteness(ce);
                        int numMissingValues = check[0];
                        int numGetMethods = check[1];

                        LOG.log(Level.FINEST, " GenericConverter:settingsAppliedConverter: Class quality check: Number of missing values {0}, Number of get functions {1}", new Object[]{check[0], check[1]});

                        if (!(numMissingValues == numGetMethods)) {
                            salSettings.add(ce);
                        }

                        if ((numMissingValues == numGetMethods)) {

                            LOG.log(Level.WARNING, " GenericConverter:settingsAppliedConverter: No values have been found on the bus, values extracted from ConfigInfo and inserted into the Map dataNamesAndValues: {0}", ce.toString());

                            for (String tospath : thisCategoriesDataNew.keySet()) {
                                // For now we ignore the states in the passed in telemetry
                                if (!tospath.contains("/state")) {

                                    LOG.log(Level.WARNING, () -> {
                                        StringBuilder sb = new StringBuilder();
                                        sb.append("GenericConverter:settingsAppliedConverter: Path: ").append(tospath).append(", category data: ").append(thisCategoriesDataNew.get(tospath));

                                        return sb.toString();
                                    });
                                }

                            }

                        }
                    } catch (ReflectiveOperationException | RuntimeException x) {
                        if (logAndContinueOnConversionError) {
                            LOG.log(Level.WARNING, "Error converting configuration, continuing", x);
                        } else {
                            throw x;
                        }
                    }

                }
            }

        }

        LOG.log(Level.FINEST, " GenericConverter:settingsAppliedConverter: End Logging ");

        // still needs to be assigned a value - return SalSettings
        return salSettings;
    }

    /**
     * TODO Add description of what this method is doing, and what the input
     * parameters are
     *
     * @param cls
     * @param dataNamesAndValues
     * @param parameters
     * @param salClassDescriptionList
     * @param categoryVersion
     * @return
     * @throws ReflectiveOperationException
     */
    private Object reflectedClass(Class cls, Map<String, Object> dataNamesAndValues, Parameter[] parameters, SALClassDescription salClassDescriptionList, String categoryVersion) throws ReflectiveOperationException {

        // arguments for this class's constructor     
        ArrayList<Object> constructorArgs = new ArrayList<>();
        Map<String, List<SALClassDescription.BusVariable>> locationBusVariableCorrespondence = new HashMap<>();
        Map<String, List<String>> locationAvailableValuesMap = new HashMap<>();

        for (SALVariable sv : salClassDescriptionList.getVariables()) {
            if (sv instanceof SALClassDescription.PatternMatchedSALVariable) {
                SALClassDescription.PatternMatchedSALVariable patternVar = (SALClassDescription.PatternMatchedSALVariable) sv;
                LocationVariable locationVar = patternVar.getLocationVariable();
                String locationName = locationVar.getVariableName();
                locationBusVariableCorrespondence.put(locationName, locationVar.getBusVariables());
                locationAvailableValuesMap.put(locationName, locationVar.getAvailableLocationValues());
            }
        }

        LOG.log(Level.FINEST, " GenericConverter:reflectedClass: Start Logging ");

        LOG.log(Level.FINEST, () -> {
            StringBuilder sb = new StringBuilder();
            sb.append(" GenericConverter:reflectedClass: Data names and values : ").append(dataNamesAndValues.toString());
            return sb.toString();
        });

        LOG.log(Level.FINEST, " GenericConverter:reflectedClass: SAL variables from SAL class description ");
        for (SALVariable psv : salClassDescriptionList.getVariables()) {
            LOG.log(Level.FINEST, () -> {
                StringBuilder sb = new StringBuilder();
                sb.append(" GenericConverter:reflectedClass: SAL Variable ").append(psv.toString());
                return sb.toString();
            });
        }

        boolean found = false;

        // loop over received constructor parameters
        for (Parameter parameter : parameters) {

            found = false;

            LOG.log(Level.FINEST, () -> {
                StringBuilder sb = new StringBuilder();
                sb.append(" GenericConverter:reflectedClass: SAL class parameters: Parameter Name : ").append(parameter.getName()).append(", Parameter Type :").append(parameter.getType());
                return sb.toString();
            });

            if (parameter.getName().trim().equals("priority")) {
                constructorArgs.add(1);

                LOG.log(Level.FINEST, " GenericConverter:reflectedClass:  Priority found and set ");
                found = true;

            }

            if (parameter.getName().trim().equals("version")) {
                constructorArgs.add(categoryVersion);

                LOG.log(Level.FINEST, " GenericConverter:reflectedClass: version found {0}", parameter);

                found = true;

            }
            // if this variable is not priority nor version which are present in all SettingsApplied classes(but not telemetry) then we loop
            // over all SALClassDescriptionList to find a match

            if (!found) {
                for (SALVariable sv : salClassDescriptionList.getVariables()) {

                    // what's the parameter Type ? Here we are most interested in Strings lists and arrays which are treated differently
                    String parameterType = parameter.getType().toString();
                    boolean parameterIsArray = parameter.getType().isArray();

                    // do I need to check any of the following or am I ok  ?
                    boolean isAnArrayList = parameterType.contains("List") || parameterType.contains("List");
                    boolean isAnArray = (!isAnArrayList) && parameterType.contains("[");
                    boolean isAString = parameterType.contains("String") || parameterType.contains("String");
                    boolean isAMap = parameterType.contains("Map");

                    LOG.log(Level.FINEST, () -> {
                        StringBuilder sb = new StringBuilder();
                        sb.append(" GenericConverter:reflectedClass  Parameter Name: ").append(parameter.getName()).append(", SAL Class Variable Name :").append(sv.getVariableName()).append(", Parameter Type: ").append(parameter.getType()).append(", SAL Class Variable Type: ").append(sv.getType());
                        return sb.toString();
                    });

                    // we're checking to see if the SALClassDescription variable has used patternMatching - ie been through the string replacement                    
                    boolean involvesPatternMatching = sv instanceof SALClassDescription.PatternMatchedSALVariable;
                    if (involvesPatternMatching) {

                        LOG.log(Level.FINEST, () -> {
                            StringBuilder sb = new StringBuilder();
                            sb.append(" GenericConverter:reflectedClass: pattern matching, Compare SAL Class parameter name ").append(parameter.getName()).append(", SAL class Variable name ").append(sv.getVariableName()).append(", SAL Class Parameter type ").append(parameter.getType()).append(",SAL Class description parameter type ").append(sv.getType());
                            return sb.toString();
                        });
                    }

                    boolean matchParName = sv.getVariableName().toLowerCase().trim().equals(parameter.getName().toLowerCase().trim());

                    String svVariableName = "";
                    // check to see if the current SALClassDescription variable is a location variable
                    boolean isLocation = false;
                    if (sv instanceof SALClassDescription.LocationVariable) {
                        svVariableName = ((SALClassDescription.LocationVariable) sv).getVariableName();
                        isLocation = true;
                    } else if (sv instanceof SALClassDescription.PatternMatchedSALVariable) {
                        svVariableName = ((SALClassDescription.PatternMatchedSALVariable) sv).getLocationVariable().getVariableName();
                    }

                    // check that the variable name on the SALClassDescription is the same as the parameter name in the class constructor
                    if (matchParName) {

                        // this is less frequent - ie a match has been found so no need for supplier string 
                        LOG.log(Level.FINEST, " GenericConverter:reflectedClass: Match found, SAL Class parameter {0} SAL variable description {1}", new Object[]{parameter.getName(), sv.getVariableName()});
                        LOG.log(Level.FINEST, " GenericConverter: reflectedClass: Match found, SAL Class parameter type {0} SAL variable description type {1}", new Object[]{parameter.getType(), sv.getType()});

                        // the current parameter is an array and was created using pattern matching and it is not a location variable
                        if (parameterIsArray && involvesPatternMatching && !isLocation) {

                            List<String> locationVals = locationAvailableValuesMap.get(svVariableName);
                            // get the size of array
                            Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType(),
                                    locationVals.size());

                            // get the associated busvariables so that the data can be extracted
                            for (BusVariable busVar : ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables()) {

                                String pathName = busVar.getPathAndNameOnBuses();
                                String busVarLocation = busVar.getLocationValue();
                                int i = locationVals.indexOf(busVarLocation);

                                Object returnedBusData = returnBusData(cls, parameter, pathName, dataNamesAndValues.get(pathName));

                                if (!(null == returnedBusData)) {

                                    LOG.log(Level.FINEST, () -> {
                                        StringBuilder sb = new StringBuilder();
                                        sb.append("GenericConverter:reflectedClass: Filling array, Parameter name : ").append(parameter.getName()).append(", SAL Class Description Name ").append(sv.getDescription()).append(", Bus type: ").append(returnedBusData.getClass().getTypeName()).append(", Bus variable name {3}: ").append(busVar.getPathAndNameOnBuses()).append(", Array element index: ").append(i).append(", Array element value from bus ").append(returnedBusData);
                                        return sb.toString();
                                    });

                                    Object returnedData = returnedBusData;
                                    if (returnedData.getClass().isArray()) {
                                        arrayArg = returnedData;
                                    } else {
                                        // Farrukh April 15 2022 - so here I am assuming that
                                        // returnedData is a single element of an array 
                                        java.lang.reflect.Array.set(arrayArg, i, returnedData);
                                    }
                                } else {
                                    // Farrukh/Tony April 15 2022 - set array element to the correct bad value given the type of
                                    // the parameter - 
                                    java.lang.reflect.Array.set(arrayArg, i, AssignValuesForMissingBusData(parameter));
                                    // was   java.lang.reflect.Array.set(arrayArg, i, -11111111)); Now type is taken 
                                    // care of by passing parameter to AssignValuesForMissingBusData ..

                                }
                            }
// this is when we've finally found a variable and its values to go into a constructor ... perhaps not the place for a supplier string 
                            LOG.log(Level.FINEST, " GenericConverter:reflectedClass: adding argument to constructor list : SAL class parameter  name {0}, parameter type {1}, SAL class description parameter name {2}, SAL class description parameter type {3}, Constructor array argument value {4} constructor array argument type {5}", new Object[]{parameter.getName(), parameter.getClass().getTypeName(), sv.getVariableName(), sv.getType(), arrayArg.toString(), arrayArg.getClass().getTypeName()});

                            constructorArgs.add(arrayArg);
                            found = true;

                        }

                        // Farrukh April 15 2022 - should we be assuming that location will always be filled in ? 
                        // fill in the names of locations - these will be a String variable in the class constructor 
                        // deal with locations - we have to be sure that SAL classes NEVER will have a string array in the constructor.
                        // Containing location is not good enough, we have to make sure the variable 
                        if (!parameterIsArray && sv instanceof SALClassDescription.LocationVariable) {
                            // we've already checked that the names of the parameters in the class and the ProcessedInfo are the same 
                            // here we have created the value of the location variable inside the processed Info so no issues.

                            // again not frequent so perhaps no need for supplier string 
                            LOG.log(Level.FINEST, "GenericConverter:reflectedClass: Location variables, SAL Class parameter name  {0}, SAL Description Name {1}, SAL Class Description printout {2}", new Object[]{parameter.getName(), sv.getVariableName(), sv.toString()});

                            if (locationBusVariableCorrespondence.get(sv.getVariableName()) == null) {

                                // should be very infrequent or we're in trouble . so no supplier string here 
                                LOG.log(Level.WARNING, " GenericConverter:reflectedClass: Null from locationBusVariableCorrespondence {0}", sv.toString());

                            }
                            // we should already have the location variables and they should always be present 
                            constructorArgs.add(convertStringList(((LocationVariable) sv).getAvailableLocationValues()));// has to be a string so I am not checking 
                            // if its not a string something is seriously wrong ...
                            found = true;

                        }

                        //
                        // one of the unusual cases where no pattern matching has occurred but the variable is an array
                        if (parameterIsArray && !involvesPatternMatching) {

                            boolean notmatched = (sv instanceof SALClassDescription.SimpleSALVariable);

                            // rare so no supplier string ? 
                            LOG.log(Level.FINEST, () -> {
                                StringBuilder sb = new StringBuilder();
                                sb.append(" GenericConverter:reflectedClass : No pattern matching has occurred but we have found an array investigating :  SAL parameter name : ").append(sv.getVariableName()).append(", SALClassDescription parameter ").append(parameter.getName());
                                sb.append("\n GenericConverter:reflectedClass : Found non pattern matched array more info  SAL parameter type: ").append(parameter.getType()).append(",SALClassDescription parameter type: ").append(sv.getType());
                                sb.append("\n you should find the name directly below if it is the corresponding bus variable in SAL class description ");
                                return sb.toString();
                            });

                            // again rare so no supplier string ..
                            if (notmatched) {
                                LOG.log(Level.FINEST, () -> {
                                    StringBuilder sb = new StringBuilder();
                                    sb.append(" GenericConverter:reflectedClass: Found non pattern matched array from SALClassDescription associated bus variable : ").append(((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses());
                                    return sb.toString();
                                });
                            }

                            // get bus variable name
                            String busVariableName = ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses();
                            Object returnedBusData = returnBusData(cls, parameter, busVariableName, dataNamesAndValues.get(busVariableName));
                            if (!(null == returnedBusData)) {
                                int xmlArraySize = ((SALClassDescription.SimpleSALVariable) sv).getCount();
                                Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType(), xmlArraySize);

                                int dataArraySize = java.lang.reflect.Array.getLength(returnedBusData);
                                if (dataArraySize > xmlArraySize) {
                                    // should be an extremely rare error so no supplier string here . 
                                    LOG.log(Level.WARNING, "Array for variable {0} exceeds the specified xml size (MaxLength = {1}, actual size = {2}). Excess data is lost.", new Object[]{busVariableName, xmlArraySize, dataArraySize});
                                }

                                //Copy the existing data and then, if needed, fill the rest with missing values
                                for (int i = 0; i < dataArraySize; i++) {
                                    java.lang.reflect.Array.set(arrayArg, i, java.lang.reflect.Array.get(returnedBusData, i));
                                }
                                for (int i = dataArraySize; i < xmlArraySize; i++) {
                                    java.lang.reflect.Array.set(arrayArg, i, AssignValuesForMissingBusData(parameter));
                                }
                                constructorArgs.add(arrayArg);
                            } else {
                                // Bus didn't carry a value for this parameter, its size is unknown, we create an array of one
                                // element and set it equal to the bad value for this particular type, Farrukh 16 April 2022. 
                                Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType());
                                java.lang.reflect.Array.set(arrayArg, 0, AssignValuesForMissingBusData(parameter));
                                constructorArgs.add(arrayArg);
                            }
                            found = true;
                        }

                        // pattern matching has occured and the parameter is not an array - 
                        if (involvesPatternMatching && !parameterIsArray && !isLocation) {

                            boolean matched = (sv instanceof SALClassDescription.PatternMatchedSALVariable);

                            // not very frequent so no supplier string used ...
                            LOG.log(Level.FINEST, () -> {
                                StringBuilder sb = new StringBuilder();
                                sb.append(" GenericConverter:reflectedClass: Pattern matched but not array, nor location :  SAL parameter name ").append(parameter.getName()).append(", SALClassDescription parameter ").append(sv.getVariableName()).append(", Parameter type ").append(parameter.getType()).append(" SALClassDescription Parameter Type ").append(sv.getType()).append(" See below for Bus pathname in SalClassDescription ");
                                return sb.toString();
                            });

                            if (matched) {

                                LOG.log(Level.FINEST, () -> {
                                    StringBuilder sb = new StringBuilder();
                                    sb.append(" GenericConverter:reflectedClass: Pattern matched but not array or location, SalClassDescription's bus path name : ").append(((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(0).getPathAndNameOnBuses());
                                    return sb.toString();
                                }
                                );
                            }

                            SALClassDescription.PatternMatchedSALVariable pmv = (SALClassDescription.PatternMatchedSALVariable) sv;
                            if (parameter.getType() != String.class) {
                                throw new RuntimeException("What should we do with type: " + parameter.getType() + " for parameter " + parameter.getName());

                            }
                            String result = "";
                            for (BusVariable bv : pmv.getBusVariables()) {
                                String pathName = bv.getPathAndNameOnBuses();
                                String returnedBusData = (String) returnBusData(cls, parameter, pathName, dataNamesAndValues.get(pathName));
                                if (!result.isEmpty()) {
                                    result += ":";
                                }
                                result += returnedBusData;
                            }
                            //TO-DO
                            //We need to make sure that result is fully populated.
                            constructorArgs.add(result);
                            found = true;
                        }

                        if (!involvesPatternMatching && !parameterIsArray && !isLocation) {

                            boolean notmatched = (sv instanceof SALClassDescription.SimpleSALVariable);

                            LOG.log(Level.FINEST, () -> {
                                StringBuilder sb = new StringBuilder();
                                sb.append(" GenericConverter:reflectedClass : Not array nor pattern matched nor location: parameter name: ").append(parameter.getName()).append(", SALClassDescription parameter: ").append(sv.getVariableName()).append(" parameter type {0} ").append(parameter.getType()).append("  SALClassDescription parameter type {1}").append(sv.getType());
                                return sb.toString();
                            });
                            LOG.log(Level.FINEST, () -> {
                                StringBuilder sb = new StringBuilder();
                                sb.append(" GenericConverter:reflectedClass: you should see the BusVariable from SALClassDescription on the next line ");
                                return sb.toString();
                            });
                            if (notmatched) {
                                LOG.log(Level.FINEST, () -> {
                                    StringBuilder sb = new StringBuilder();
                                    sb.append(" GenericConverter:reflectedClass: Bus Variable Name from SALClassDescription for unmatched variable ").append(((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses());
                                    return sb.toString();
                                });
                            }

                            // proposed change April 9 2022. Also duplicate this in other places - Farrukh  
                            String pathName = ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses();
                            Object returnedBusData = returnBusData(cls, parameter, pathName, dataNamesAndValues.get(pathName));
                            if (!(null == returnedBusData)) {
                                constructorArgs.add(returnedBusData);
                            } else {

                                constructorArgs.add(AssignValuesForMissingBusData(parameter));

                            }

                            found = true;

                        }

                    }
                }

            }

            // If we are here then we have not derived the appropriate parameter name from the dictionary
            // so SALClassDescriptionInfo which extracts info from the dictionary and makes all the parameter
            // names for the actual SAL class constructor was unable to derive the name of this parameter
            // in this case it is difficult to know what to do except to take the parameter from the actual SAL
            // We should not enter this if loop unless something is seriously wrong - so no supplier string is used
            if (!found) {

                // rare and probably a warning we want to stick out - so no supplie string . 
                LOG.log(Level.WARNING, " GenericConverter:reflectedClass: Problem : parameter was not found, for Class Name {0} Parameter Name {1}", new Object[]{cls.getCanonicalName(), parameter.getName()});
                LOG.log(Level.WARNING, " GenericConverter:reflectedClass : Problem : Parameter was not found is it in the list following ? ");

                for (String tospath : dataNamesAndValues.keySet()) {
                    LOG.log(Level.WARNING, "GenericConverter:reflectedClass: Problem, parameter not found - here are all the parameters we have stored, path name {0} value {1} type {2}", new Object[]{tospath, dataNamesAndValues.get(tospath), dataNamesAndValues.get(tospath).getClass().getTypeName()});
                }

                mia.add(parameter.getName());
                alreadyWarned.put(cls, mia);
                int jj = Collections.frequency(alreadyWarned.get(cls), parameter.getName());
                // we should be able to set the number of warnings we want to tolerate
                // if you want to test for arrays and the type use parameter.getParameterizedType().getTypeName()
                // this will give you double[], long[], int[] etc...
                if (parameter.getType().isArray()) {
                    Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType(), 0);
                    constructorArgs.add(arrayArg);
                    // keeping older block around in case we want to revert                    
/*                    if (parameter.getParameterizedType().getTypeName().toLowerCase().contains("double")) {
                        constructorArgs.add(new double[0]);
                    }
                    if (parameter.getParameterizedType().getTypeName().toLowerCase().contains("long")) {
                        constructorArgs.add(new long[0]);
                    }
                    if (parameter.getParameterizedType().getTypeName().toLowerCase().contains("int")) {
                        constructorArgs.add(new int[0]);
                    } */
                } else {
                    constructorArgs.add(AssignValuesForMissingBusData(parameter));
                }

            }

        }

        LOG.log(Level.FINEST, "GenericConverter:reflectedClass: cross check just before reflection, class name {0}, constructor (compare with arguments below) {1}", new Object[]{cls.getName(), cls.getConstructors()[0].toString()});

        for (int i = 0; i < constructorArgs.size(); i++) {
            LOG.log(Level.FINEST, "GenericConverter: reflectedClass - matched arguments being provided to above constructor{0} {1} Value and Type 2 {2}", new Object[]{constructorArgs.get(i), constructorArgs.get(i).getClass().getTypeName(), Arrays.toString(constructorArgs.get(i).getClass().getTypeParameters())});
        }

        // so now constructor args must be full.....or not let's check ....
        try {
            // potentially useful debug statement 
            //  System.out.println("printing the class out " + cls.toString());
            LOG.log(Level.FINEST, " GenericConverter:reflectedClass: End Logging ");

            return cls.getConstructors()[0].newInstance(constructorArgs.toArray());
        } catch (ReflectiveOperationException | RuntimeException x) {
            throw new RuntimeException(String.format("Error creating SAL event for class %s with args %s", cls.getCanonicalName(), betterToString(parameters, constructorArgs)), x);
        }

    }

    // return data 
    private Object returnBusData(Class cls, java.lang.reflect.Parameter parameter, String pathName, Object returnValue) { //List<KeyValueData> values) {

        boolean classTypeIsConfig = false;

        if (cls.getCanonicalName().contains("Event") && cls.getCanonicalName().contains("org.lsst.sal.cccamera.event")) {
            classTypeIsConfig = true;
        }

        // We are currently converting Durations to strings, here is check against that 
        // if however encounter a duration that should not be a string we should not convert it
        // to a string ..rather to a double 
        LOG.log(Level.FINEST, " GenericConverter:returnBusData: Start Logging");
        LOG.log(Level.FINEST, " GenericConverter:returnBusData: Parameter name {0}, Parameter type {1}", new Object[]{parameter.getName(), parameter.getParameterizedType().getTypeName()});
        LOG.log(Level.FINEST, " GenericConverter:returnBusData: Given path name {0}", pathName);
        LOG.log(Level.FINEST, " GenericConverter:returnBusData: Return Value {0}", returnValue);

        if (returnValue == null) {
            return null;
        }

        if (returnValue instanceof Double) {

            LOG.log(Level.FINEST, " GenericConverter:returnBusData: instance of double {0}", returnValue);

        }

//     double interval = java.time.Duration.ofMillis(BUSVALUE);
//     double interval = interval/1000.0; // convert to seconds ...                
        if (returnValue instanceof java.time.Duration && !parameter.getType().toString().toLowerCase().contains("string")) {
            // change to just double once xml is fixed 
            returnValue = ((java.time.Duration) returnValue).getSeconds();

            LOG.log(Level.FINEST, " GenericConverter:returnBusData: java.time.duration in seconds {0}", returnValue);

        }

        if (returnValue instanceof HashMap) {
            returnValue = ((HashMap) returnValue).toString();

            LOG.log(Level.FINEST, " GenericConverter:returnBusData: returnValue is hashmap - printing as string {0}", returnValue.toString());

            if (returnValue.getClass().getTypeName().contains("[")) {
                LOG.log(Level.FINEST, " GenericConverter:returnBusData: a hashmap containing [ in getTypeName {0}", returnValue.toString());
            }

        }

        if (returnValue.getClass().getTypeName().toLowerCase().contains("string") && returnValue instanceof String[]) {

            LOG.log(Level.FINEST, " GenericConverter:returnBusData: return value is a string {0}", returnValue);

            returnValue = convertStringArray((String[]) returnValue);
        }

        // Maybe here check for arrayList and containing a string - though I am not sure this will work. Farrukh April 9 2022
        // TODO: We should use toString on any unsupported Object, not an explicit list
        // old  : if (returnValue instanceof List || returnValue instanceof Map || returnValue instanceof Enum || returnValue instanceof Duration) {
        if (returnValue instanceof List || returnValue instanceof Map || returnValue instanceof Enum) {

            // Can find out what the first element in the array list is with the following 
            //  System.out.println("INSTANCE OF " + ((ArrayList)returnValue).get(0).getClass().getTypeName());
            returnValue = returnValue.toString();
        }

        if (returnValue instanceof java.time.Duration && parameter.getType().toString().toLowerCase().contains("string")) {
            returnValue = returnValue.toString();
        }

        if (returnValue.getClass().getTypeName().toLowerCase().contains("string") && (returnValue instanceof List)) {

            returnValue = convertStringList((List<String>) returnValue);
        }

        // I think Here we should be returning badDValue badIValue etc - depending on whether we find a value or not - Farrukh April 8 2022. 
        if (null == returnValue) {

            LOG.log(Level.WARNING, " GenericConverter:returnBusData: From returnBusData : return value is null ! {0}", returnValue);

            // the following warning is for the case when a configuration event class has a null 
            // value for any pathName - this shouldn't happen thus it is outside the DEBUG_PRINT 
            if (classTypeIsConfig) {
                LOG.log(Level.WARNING, " GenericConverter:returnBusData: Bus data contains null for pathname : {0}, for Config class name  {1}, parameter name {2}", new Object[]{pathName, cls.getTypeName(), parameter.getName()});
            }
            // will pick up parameter as a double or a float and not as java.time.Duration
            // since the java.time.Duration will (should!) only be the data type on the bus
            returnValue = AssignValuesForMissingBusData(parameter);
        }

        LOG.log(Level.FINEST, " GenericConverter:returnBusData: End Logging");

        return returnValue;
    }

    // clean up the class name used for xmls
    /*
    String cleanupXMLClassName(String xmlClassName) {

        xmlClassName = xmlClassName.replace("comcam-", "");
        String[] blahs = xmlClassName.split("_");
        String cleanedName = "";

        for (int j = 1; j < blahs.length; j++) {
            cleanedName += blahs[j] + "_";
        }

        int lastIndex = cleanedName.lastIndexOf("_");
        cleanedName = cleanedName.substring(0, lastIndex);

        return cleanedName.trim();
    }
     */
    // clean up the SAL class Name for comparision
    /*
    String cleanupSALClassName(String salClassName) {

        String cleanedName = salClassName.split("\\.")[salClassName.split("\\.").length - 1];

        cleanedName = cleanedName.replace("ATCamera", "").trim();
        cleanedName = cleanedName.replace("MTCamera", "").trim();
        cleanedName = cleanedName.replace("CCCamera", "").trim();
        cleanedName = cleanedName.replace("Telemetry", "").trim();
        cleanedName = cleanedName.replace("ConfigurationAppliedEvent", "").trim();

        if (cleanedName.startsWith("_")) {
            cleanedName = cleanedName.replaceFirst("[_]", "").trim();
        }

        return cleanedName.trim();
    }
     */
 /* TO-DO Farrukh (following Max's suggestion) change the multiple if blocks to if - else if and an else and through an exception in the else block */
    private boolean badValueOrValues(Object invokedValue, Object badValue, String className, String methodName) {
        boolean bad = true;

        boolean isanArray = invokedValue.getClass().isArray();
        String type = invokedValue.getClass().getTypeName();
        boolean isDouble = false;
        boolean isString = false;
        boolean isInt = false;
        boolean isFloat = false;
        boolean isLong = false;
        boolean isBool = false;
        boolean isShort = false;

        if (type.toLowerCase().contains("double")) {
            isDouble = true;
            badValue = badDValue;
            if (isanArray && isDouble) {
                double[] doubleArray = (double[]) invokedValue;
                for (int i = 0; i < doubleArray.length; i++) {
                    // even if one value is good the array is passable        
                    if (!Double.isNaN(doubleArray[i])) {
                        bad = false;
                    }
                }
            }
        } else if (type.toLowerCase().contains("string")) {
            isString = true;
            badValue = badStringValue;
            if (isanArray) {
                String[] stringArray = (String[]) invokedValue;
                for (int i = 0; i < stringArray.length; i++) {
                    // even if one value is good the array is passable        
                    if (!(stringArray[i] == (String) badStringValue)) {
                        bad = false;
                    }
                }
            }
        } else if (type.toLowerCase().contains("int")) {
            isInt = true;
            badValue = badIValue;
            if (isanArray) {
                int[] intArray = (int[]) invokedValue;
                for (int i = 0; i < intArray.length; i++) {
                    // even if one value is good the array is passable        
                    if (!(intArray[i] == (int) badValue)) {
                        bad = false;
                    }
                }
            }
        } else if (type.toLowerCase().contains("float")) {
            isFloat = true;
            badValue = badFValue;
            if (isanArray) {
                float[] floatArray = (float[]) invokedValue;
                for (int i = 0; i < floatArray.length; i++) {
                    // even if one value is good the array is passable        
                    if (!Float.isNaN(floatArray[i])) {
                        bad = false;
                    }
                }
            }
        } else if (type.toLowerCase().contains("long")) {
            isLong = true;
            badValue = badLValue;
            if (isanArray) {
                long[] longArray = (long[]) invokedValue;
                for (int i = 0; i < longArray.length; i++) {
                    // even if one value is good the array is passable        
                    if (!(longArray[i] == (long) badLValue)) {
                        bad = false;
                    }
                }
            }
        } else if (type.toLowerCase().contains("short")) {
            isShort = true;
            badValue = badSValue;
            if (isanArray) {
                short[] shortArray = (short[]) invokedValue;
                for (int i = 0; i < shortArray.length; i++) {
                    // even if one value is good the array is passable        
                    if (!(shortArray[i] == (long) badSValue)) {
                        bad = false;
                    }
                }
            }
        } /*else if (type.toLowerCase().contains("boolean")){
            isBool = true;
            badValue = badBValue;
            if (isanArray) {
                boolean[] boolArray = (boolean[]) invokedValue;
                for (int i = 0; i < boolArray.length; i++) {
                // even if one value is good the array is passable        
                    if (!(boolArray[i] == (boolean) badBValue)) {
                        bad = false;
                    }
                }
            }
        }*/ else {
            throw new RuntimeException("Value provided is not in the list of expected values : " + type);
        }

        if (!isanArray) {
            if (!invokedValue.equals(badValue)) {
                bad = false;
            }
        }

        LOG.log(Level.FINEST, " GenericConverter:badValueOrValues Start Logging ");
        LOG.log(Level.FINEST, " GenericConverter:badValueOrValues : Class name {0}, method name {1}", new Object[]{className, methodName});
        LOG.log(Level.FINEST, " GenericConverter:badValueOrValues: Array ? {0}, type {1}", new Object[]{isanArray, invokedValue.getClass().getTypeName()});
        LOG.log(Level.FINEST, " GenereicConverter:badValueOrValues: invokedValue {0}, badValue {1}", new Object[]{invokedValue, badValue});
        LOG.log(Level.FINEST, " GenericConverter:badValueOrValues End Logging ");

        return bad;
    }

    /*
    Map<String, Object> copyStringObjectMap(Map<String, Object> inMap) {

        Map<String, Object> copy = new HashMap<>();

        copy.putAll(inMap);

        return copy;
    }
     */
    private static String betterToString(Parameter[] parameters, ArrayList<Object> data) {
        List<String> result = new ArrayList<>();
        for (int i = 0; i < data.size(); i++) {
            boolean classesMatch = ClassUtils.isAssignable(data.get(i).getClass(), parameters[i].getType(), true);
            String argString = parameters[i].getName() + "=" + (data.get(i).getClass().isArray() ? arrayToString(data.get(i)) : data.get(i).toString());
            if (!classesMatch) {
                argString += String.format(" WARNING: classMismatch %s != %s", parameters[i].getType(), data.get(i).getClass());
            }
            result.add(argString);
        }
        return "[" + String.join(", ", result) + "]";
    }

    private static String convertStringSet(Set<String> stringSet) {

        String[] localArray = new String[stringSet.size()];
        stringSet.toArray(localArray);
        return manipulateStringArray(localArray);
    }

    private static String convertStringList(List<String> stringList) {

        String[] localArray = new String[stringList.size()];
        stringList.toArray(localArray);
        return manipulateStringArray(localArray);
    }

    private static String convertStringArray(String[] stringArray) {

        String[] localArray = null;

        return manipulateStringArray(stringArray);

    }

    private static String manipulateStringArray(String[] inputArray) {
        String returnStringI = "";
        String returnStringF = "";
        int lengthOfStringArray = inputArray.length;
        if (!(lengthOfStringArray > 0)) {
            System.out.println("Warning from GenericConverter3:manipulateStringArray : Length of string Array is 0 !");
            returnStringF = "NOTFOUND";
        } else {
            for (int i = 0; i < lengthOfStringArray; i++) {
                returnStringI += inputArray[i] + ":";
                if (i == lengthOfStringArray - 1) {
                    int target = returnStringI.lastIndexOf(":");
                    if (target == -1) {
                        returnStringF = returnStringI;
                    } else {
                        returnStringF = returnStringI.substring(0, target);
                    }
                }
            }
        }
        return returnStringF;
    }

    private static String arrayToString(Object array) {
        List<String> result = new ArrayList<>();
        int l = Array.getLength(array);
        for (int i = 0; i < l; i++) {
            result.add(Array.get(array, i).toString());
        }
        return "[" + String.join(", ", result) + "]";
    }

    // Comment added April 14 2022 - Farrukh Azfar : this program calls badValueOrValues
    // to check if a class has a certain number of bad values set for its constructor 
    // parameters - the purpose here is to determine if a class should be returned if
    // it has greater than a total number of bad arguments ...
    private int[] checkClassCompleteness(Object cls) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, NoSuchMethodException {

        Parameter[] thisClassParameters = cls.getClass().getConstructors()[0].getParameters();
        Method[] methods = cls.getClass().getDeclaredMethods();
        ArrayList<Method> getMethods = new ArrayList<>();
        int numMissingValues = 0;
        int numGetMethods = 0;
        int[] returnValues = new int[3];
        int countBooleanFunctions = 0;

        for (int i = 0; i < methods.length; i++) {
            String methodName = methods[i].getName();

            Method method = null;
            method = cls.getClass().getDeclaredMethod(methodName);
            // check to see if its boolean 
            String methodS = method.toString();
            if (methodS.toLowerCase().contains("boolean")) {
                countBooleanFunctions++;
            }

            // System.out.println(methodName + " Method of myClass: " + method.toString() );
            String[] parts = methodName.split("\\s");
            if (!parts[parts.length - 1].contains("toString") && (parts[parts.length - 1].contains("get") /*|| parts[parts.length - 1].startsWith("is")*/)) {
                getMethods.add(methods[i]);
                numGetMethods++;
            }
        }

        // don't check bad values for booleans -
        boolean checkbadvalue = false;

        for (Method method : getMethods) {

            if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("double")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badDValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("string")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badStringValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("float")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badFValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("int")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badIValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("long")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badLValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("short")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badSValue, cls.getClass().getName(), method.getName());
            }

            if (checkbadvalue) {
                numMissingValues++;
            }

            if (!checkbadvalue && method.getName().toLowerCase().equals("getlocation")) {
                numMissingValues++;
            }

        }

        returnValues[0] = numMissingValues;
        returnValues[1] = numGetMethods;
        returnValues[2] = countBooleanFunctions;
//        System.out.println(" Class Name " + cls.getClass().getName()+ " Number of boolean functions " + countBooleanFunctions);
        return returnValues;
    }

    private Map<String, ConfigurationParameterInfo> alternateCategoryData(ConfigurationInfo data, String category) {
        Map<String, ConfigurationParameterInfo> returnMap = new HashMap<>();
        System.out.println(" AlternateCategoryData requested for category " + category);
        try {
            returnMap = data.getCurrentParameterInfoForCategory(category);
        } catch (NullPointerException e) {
            System.out.println(" Null pointer exception caught " + e.toString() + " for category " + category);

        }

        return returnMap;
    }

    private Object AssignValuesForMissingBusData(java.lang.reflect.Parameter parameter) {

        Object badValue = null;
        String type = parameter.getParameterizedType().getTypeName().toLowerCase();

        if (type.contains("float")) {
            badValue = (float) badFValue;
        }
        if (type.contains("int")) {
            badValue = (int) badIValue;
        }
        if (type.contains("long")) {
            badValue = (long) badLValue;
        }
        if (type.contains("boolean")) {
            badValue = (boolean) badBValue;
        }
        if (type.contains("double")) {
            badValue = (double) badDValue;
        }
        if (type.contains("short")) {
            badValue = (short) badSValue;
        }
        if (type.contains("string")) {
            badValue = (String) badStringValue;
        }
        return badValue;
    }

}
