package org.lsst.ccs.subsystem.ocsbridge;

import java.io.Serializable;
import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.camera.sal.xml.XMLMaker2.SALType;
import org.lsst.sal.camera.CameraTelemetry;

/**
 * Handler receipt of telemetry from CCS, conversion into equivalent SAL
 * messages, and send to SAL.
 *
 * @author tonyj
 */
public abstract class TelemetrySender extends GenericSender {

    private static final Duration PARTIAL_DATA_TIMEOUT = Duration.ofSeconds(10);

    private static final Logger LOG = Logger.getLogger(TelemetrySender.class.getName());
    private final Map<String, PartialData> partialDataMap = new ConcurrentHashMap<>();
    private final ScheduledExecutorService scheduler;
    private OCSTelemetrySender sender;

    static TelemetrySender create(Camera device, OCSTelemetrySender sender, ScheduledExecutorService scheduler) {
        switch (device) {
            case COMCAM:
                return new ComCamTelemetrySender(sender, scheduler);
            case AUXTEL:
                return new AuxTelTelemetrySender(sender, scheduler);
            case MAIN_CAMERA:
                return new CameraTelemetrySender(sender, scheduler);
            default:
                throw new IllegalArgumentException("Unsupported device: " + device);
        }

    }

    protected TelemetrySender(OCSTelemetrySender sender, ScheduledExecutorService scheduler, Camera camera) {
        super(camera, SALType.TELEMETRY);
        this.scheduler = scheduler;
        this.sender = sender;
    }

    void send(StatusSubsystemData data) {
        handlePartialTelemetry(data);
    }

    private void convertAndSend(StatusSubsystemData data) {
        try {
            List<CameraTelemetry> converted = getConverter().telemetryConverter(data);
            if (converted.isEmpty()) {
                LOG.log(Level.FINE, "No converted data {0}", data.getOriginAgentInfo().getName());
            } else {
                for (CameraTelemetry t : converted) {
                    t = applyAfterBurner(t);
                    LOG.log(Level.FINE, "Sending {0} {1}", new Object[]{data.getOriginAgentInfo().getName(), t});
                    sender.sendTelemetry(t);
                }
            }
        } catch (ReflectiveOperationException ex) {
            LOG.log(Level.WARNING, String.format("Problem converting telemetry subsytem: %s key: %s", data.getOriginAgentInfo().getName(), data.getDataKey()));
        }
    }
    
    /**
     * Allow subclasses to modify sent events if necessary
     * @param t The event to be sent.
     * @return The input event or modified event
     */
    protected CameraTelemetry applyAfterBurner(CameraTelemetry t) {
        return t;
    }
    
    
    private void handlePartialTelemetry(StatusSubsystemData data) {
        String name = data.getOriginAgentInfo().getName();
        KeyValueDataList kvdl = (KeyValueDataList) data.getSubsystemData();
        Serializable type = kvdl.getAttribute("publicationType");
        Serializable task = kvdl.getAttribute("taskName");
        Serializable total = kvdl.getAttribute("totalTasks");
        LOG.log(Level.FINE, "Name: {0}, type: {1}, task: {2}, total: {3}", new Object[]{name, type, task, total});
        if (type == null) {
            convertAndSend(data); 
        } else if ("scheduledFull".equals(type)) {
            convertAndSend(data);
        } else if ("scheduledPartial".equals(type)) {
            // We keep collecting partial data until we either have a complete set, or a timer elapses
            PartialData dataSoFar = partialDataMap.computeIfAbsent(name, (s) -> new PartialData(s, (int) total));
            dataSoFar.put(task.toString(), data);
            if (dataSoFar.isComplete()) {
                dataSoFar.send();
            }

        } else {
            // ignored for now
        }
    }

    void setSender(OCSCommandExecutor ocs) {
        this.sender = ocs;
    }

    private class PartialData {

        private final Map<String, StatusSubsystemData> dataSoFar = new ConcurrentHashMap<>();
        private final String subsystemName;
        private final int totalTasks;
        private final ScheduledFuture<?> timer;

        PartialData(String subsystemName, int totalTasks) {
            this.subsystemName = subsystemName;
            this.totalTasks = totalTasks;
            timer = scheduler.schedule(() -> send(), PARTIAL_DATA_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS);
        }

        private void put(String taskName, StatusSubsystemData data) {
            dataSoFar.put(taskName, data);
        }

        private boolean isComplete() {
            return dataSoFar.size() == totalTasks;
        }

        private synchronized void send() {
            if (!timer.isDone()) {
                partialDataMap.remove(subsystemName);
                //timer.cancel(false);
                convertAndSend(combineData(dataSoFar.values()));
            }
        }

        private StatusSubsystemData combineData(Collection<StatusSubsystemData> dataSoFar) {
            StatusSubsystemData result = null;
            for (StatusSubsystemData moreData : dataSoFar) {
                if (result == null) {
                    result = moreData;
                } else {
                    KeyValueDataList kvdl = (org.lsst.ccs.bus.data.KeyValueDataList) result.getEncodedData();
                    KeyValueDataList moreKVDL = (org.lsst.ccs.bus.data.KeyValueDataList) moreData.getEncodedData();
                    for (KeyValueData kvd : moreKVDL.getListOfKeyValueData()) {
                        kvdl.addData(kvd);
                    }
                }
            }
            return result;
        }

    }

}
