package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Phaser;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.data.ImageMetaDataEvent;
import org.lsst.ccs.subsystem.imagehandling.data.MeasuredShutterTime;
import static org.lsst.ccs.subsystem.ocsbridge.OCSBridge.DELIMITED_STRING_SPLIT_JOIN;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSImageNameEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.event.EndOfImageTelemetryEvent;
import org.lsst.sal.camera.event.EndReadoutEvent;
import org.lsst.sal.camera.event.StartIntegrationEvent;
import org.lsst.sal.camera.event.StartReadoutEvent;

/**
 * For each image triggered there are a number of asynchronous events fired
 * which must be rendezvoused to generate appropriate OCS events. One instance
 * of this class is created for each image, and used to track these events.
 *
 * @author tonyj
 */
public class AsynchronousEventHandler {

    private static final Logger LOG = Logger.getLogger(AsynchronousEventHandler.class.getName());

    private final ImageNameComparator imageNameComparator = new ImageNameComparator();
    private final ConcurrentSkipListMap<ImageName, AsynchronousImageHandler> activeHandlers = new ConcurrentSkipListMap<>(imageNameComparator);
    private AsynchronousImageHandler currentImageHandler;
    private ImageName previousImageName;
    private final CCS ccs;
    private final OCSBridgeConfig config;
    
    AsynchronousEventHandler(OCSBridgeConfig config, CCS ccs) {
        this.config = config;
        this.ccs = ccs;
    }
    
    /**
     * Create a new AsynchronousImageHandler. This is done when the
     * ccsImageNameEvent arrives since it has the newly created image name.
     *
     * @param ccsImageNameEvent
     * @return
     */
    AsynchronousImageHandler create(CCSImageNameEvent ccsImageNameEvent, OCSCommandExecutor ocsCommandExecutor) {
        ImageName imageName = ccsImageNameEvent.getImageName();
        AsynchronousImageHandler handler = new AsynchronousImageHandler(ccsImageNameEvent, ocsCommandExecutor);
        // TODO: Add a timeout
        activeHandlers.put(imageName, handler);
        synchronized(activeHandlers) {
            activeHandlers.notifyAll();
        }
        return handler;
    }

    /**
     * Waits for a new AsynchronousImageHandler to be available. New in this context means a new image created since the previous
     * time this method was called.
     * @param waitTime The maximum time to wait for the image
     * @return A Future which will contain the new AsynchronousImageHandler once it is created, or will fail if the waitTime is exceeded.
     */
    CompletableFuture<AsynchronousImageHandler> getNewImageHandler(Duration waitTime) {
        CompletableFuture<AsynchronousImageHandler> result = new CompletableFuture<>();
        ccs.runInBackground(() -> {
            LocalDateTime deadline = LocalDateTime.now().plusNanos(waitTime.toNanos());
            while (!result.isDone() && !result.isCancelled() && LocalDateTime.now().isBefore(deadline)) {
                Map.Entry<ImageName, AsynchronousImageHandler> lastEntry = activeHandlers.lastEntry();
                if (lastEntry != null && (previousImageName == null || imageNameComparator.compare(previousImageName, lastEntry.getKey()) < 0)) {
                    currentImageHandler = lastEntry.getValue();
                    previousImageName = currentImageHandler.getImageNameEvent().getImageName();
                    LOG.log(Level.INFO, "Create image handler for {0}", previousImageName);
                    result.complete(currentImageHandler);
                }
                synchronized (activeHandlers) {
                    try {
                        activeHandlers.wait(100);
                    } catch (InterruptedException ex) {
                        result.completeExceptionally(ex);
                        break;
                    }
                }
            }
            result.completeExceptionally(new RuntimeException("Wait for new event handler timed out"));
        });
        return result;
    }
    /**
     * Gets the current AsynchronousImageHandler. The current image handler exists between
     * when getNewImageHandler is called, and when endReadout is called.
     * @return The AsynchronousImageHandler. 
     * @throws RuntimeException is no current image handler exists
     */
    AsynchronousImageHandler getCurrent() {
        if (currentImageHandler == null) {
            throw new RuntimeException("No currentImageHandler exists");
        }
        return currentImageHandler;
    }

    /**
     * Gets the AsynchronousImageHandler for a named image
     * @param imageName The image name to look for
     * @return The AsynchronousImageHandler for the named image
     * @throws RuntimeException if the requested image handler does not exist
     */
    AsynchronousImageHandler getHandlerForImage(ImageName imageName) {
        AsynchronousImageHandler imageHandler = activeHandlers.get(imageName);
        if (imageHandler == null) {
            throw new RuntimeException("Could not find image "+imageName);
        }
        return imageHandler;
    }
    
    /**
     * Class used to track events on a per-image basis
     */
    class AsynchronousImageHandler {

        private final CCSImageNameEvent imageNameEvent;
        private final Phaser endOfImageTelemetryCountdown;
        private MeasuredShutterTime measuredShutterTime;
        private ImageMetaDataEvent imageMetaDataEvent;
        private OCSCommandExecutor ocsCommandExecutor;

        private AsynchronousImageHandler(CCSImageNameEvent imageName, OCSCommandExecutor ocsCommandExecutor) {
            this.imageNameEvent = imageName;
            this.ocsCommandExecutor = ocsCommandExecutor;
            // We create a count down for when to send the endOfImageTelemetry.
            // We expect 3 things to happen
            // 1. ImageMetaData arrives
            // 2. 300ms after EndOfReadout (somewhat arbitrary but is intended to allow telemetry to arrive)
            // 3. measuredShutterTime arrives. This will never arrive if this is AuxTel or ComCam, or if the shutter is not opened, we compensate in startReadout
            endOfImageTelemetryCountdown = new Phaser(3) {
                @Override
                protected boolean onAdvance(int phase, int registeredParties) {
                    sendEndOfImageTelemetry();
                    LOG.log(Level.INFO, "Removing image handler for {0}", imageNameEvent.getImageName());
                    activeHandlers.remove(imageNameEvent.getImageName());
                    return true;
                }                
            };
        }

        private CCSImageNameEvent getImageNameEvent() {
            return imageNameEvent;
        }

        /**
         * Called at start of integration
         */
        void startIntegration() {
            final ImageName imageName = imageNameEvent.getImageName();            
            StartIntegrationEvent sie = StartIntegrationEvent.builder()
                    .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                    .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                    .imagesInSequence(imageNameEvent.getImagesInSequence())
                    .imageIndex(imageNameEvent.getSequenceNumber() + 1)
                    .imageName(imageName.toString())
                    .imageNumber(imageName.getNumber())
                    .imageSource(imageName.getSource().getCode())
                    .imageController(imageName.getController().getCode())
                    .imageDate(imageName.getDateString())
                    .timestampAcquisitionStart(imageNameEvent.getIntegrationStartTime().getTAIDouble())
                    .exposureTime(imageNameEvent.getExposureTime())
                    .mode(imageNameEvent.getMode())
                    .timeout(imageNameEvent.getTimeout())
                    .build();
            sendEvent(sie);
        }

        private void sendEvent(CameraEvent event) {
            ocsCommandExecutor.sendEvent(event);
        }

        /**
         * Called at startReadout
         * @param when The timestamp for the state transition
         */
        void startReadout(CCSTimeStamp when) {
            final ImageName imageName = imageNameEvent.getImageName();
            // If we are not going to get shutterTimeArrived then we compensate here
            if (!imageNameEvent.isShutterOpen() || config.getDevice() !=  Camera.MAIN_CAMERA) {
                endOfImageTelemetryCountdown.arrive();
            }
            StartReadoutEvent sre = StartReadoutEvent.builder()
                    .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                    .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                    .imagesInSequence(imageNameEvent.getImagesInSequence())
                    .imageIndex(imageNameEvent.getSequenceNumber() + 1)
                    .imageName(imageName.toString())
                    .imageNumber(imageName.getNumber())
                    .imageSource(imageName.getSource().getCode())
                    .imageController(imageName.getController().getCode())
                    .imageDate(imageName.getDateString())
                    .timestampAcquisitionStart(imageNameEvent.getIntegrationStartTime().getTAIDouble())
                    .timestampStartOfReadout(when.getTAIDouble())
                    .exposureTime(imageNameEvent.getExposureTime())
                    .build();

            sendEvent(sre);
        }
        /**
         * Called at endReadout
         * @param when The timestamp for the state transition
         */
        void endReadout(CCSTimeStamp when) {
            final ImageName imageName = imageNameEvent.getImageName();
            // If this is not the main camera, or if the shutter is not opened, schedule endOfImageTelemetry
            //if (!imageNameEvent.isShutterOpen() || config.getDevice() !=  Camera.MAIN_CAMERA) {
                ccs.schedule(Duration.ofMillis(300), () -> endOfImageTelemetryCountdown.arrive());
            //}
            // Once end readout arrives, we are no longer the current image handler
            currentImageHandler = null;
            EndReadoutEvent ere = EndReadoutEvent.builder()
                    .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                    .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                    .imagesInSequence(imageNameEvent.getImagesInSequence())
                    .imageIndex(imageNameEvent.getSequenceNumber() + 1)
                    .imageName(imageName.toString())
                    .imageNumber(imageName.getNumber())
                    .imageSource(imageName.getSource().getCode())
                    .imageController(imageName.getController().getCode())
                    .imageDate(imageName.getDateString())
                    .timestampAcquisitionStart(imageNameEvent.getIntegrationStartTime().getTAIDouble())
                    .timestampEndOfReadout(when.getTAIDouble())
                    .requestedExposureTime(imageNameEvent.getExposureTime())
                    .build();
            sendEvent(ere);
        }

        /**
         * Called when (if) the measured shutter time arrives
         * @param measuredShutterTime The measured shutter time
         */
        void shutterTimeArrived(MeasuredShutterTime measuredShutterTime) {
            this.measuredShutterTime = measuredShutterTime;
            endOfImageTelemetryCountdown.arrive();
        }

        /**
         * Called when the imageMetaDataEvent arrives
         * @param imageMetaDataEvent The image meta data
         */
        void imageMetaDataArrived(ImageMetaDataEvent imageMetaDataEvent) {
            this.imageMetaDataEvent = imageMetaDataEvent;
            endOfImageTelemetryCountdown.arrive();
        }

        private void sendEndOfImageTelemetry() {
            EndOfImageTelemetryEvent event = EndOfImageTelemetryEvent.builder()
                    .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                    .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                    .imagesInSequence(imageNameEvent.getImagesInSequence())
                    .imageIndex(imageNameEvent.getSequenceNumber() + 1)
                    .imageName(imageMetaDataEvent.getImageName().toString())
                    .imageSource(imageMetaDataEvent.getImageName().getSource().getCode())
                    .imageController(imageMetaDataEvent.getImageName().getController().getCode())
                    .imageDate(imageMetaDataEvent.getImageName().getDateString())
                    .imageNumber(imageMetaDataEvent.getImageName().getNumber())
                    .timestampAcquisitionStart(imageMetaDataEvent.getLastClearTime().getTAIDouble())
                    // FIXME: The next two lines are only correct when no shutter motion
                    .timestampDateEnd(imageMetaDataEvent.getIntegrationEndTime().getTAIDouble())
                    .timestampDateObs(imageMetaDataEvent.getIntegrationStartTime().getTAIDouble())
                    .darkTime(imageMetaDataEvent.getDarkTime())
                    .imageTag(imageMetaDataEvent.getDaqTag())
                    .exposureTime(imageNameEvent.getExposureTime())
                    .measuredShutterOpenTime(measuredShutterTime != null ? measuredShutterTime.getMeasuredShutterOpenTime() : imageNameEvent.getRequestedShutterOpenTime())
                    .emulatedImage(imageMetaDataEvent.getEmulatedImageName() == null ? "" : imageMetaDataEvent.getEmulatedImageName())
                    .build();
            sendEvent(event);
        }
    }
    
    private static class ImageNameComparator implements Comparator<ImageName> {

        @Override
        public int compare(ImageName o1, ImageName o2) {
            // TODO: Something better
            return o1.toString().compareTo(o2.toString());
        }

    }
}
