package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.CommandInvocationException;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListenerList;
import org.lsst.sal.SAL;
import org.lsst.sal.SALCommand;
import org.lsst.sal.SALEvent;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.sal.SALException;
import org.lsst.sal.SALHasErrorCode;
import org.lsst.sal.SALReceivedCommand;
import org.lsst.sal.SALTelemetry;

/**
 * An implementation of OCSCommandExecutor which uses SAL itself.
 *
 * @author tonyj
 */
public class OCSBridgeSALLayer extends OCSCommandExecutor {

    private final OCSBridge bridge;
    private final SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr;
    private boolean shutdown = false;
    private static final Logger logger = Logger.getLogger(OCSBridgeSALLayer.class.getName());
    private Thread commandThread;
    private Thread eventThread;
    private SALReceivedCommand<CameraCommand> currentCommand;
    private final EventListenerList<SALEvent> eventListenerList = new EventListenerList<>();

    OCSBridgeSALLayer(OCSBridge bridge) {
        super(bridge);
        this.bridge = bridge;
        mgr = bridge.getConfig().getSALManager();
    }

    public void start() {

        Thread t = new Thread("SALCommandReceiver") {

            @Override
            public void run() {
                try {
                    runCommandLoop();
                } catch (SALException x) {
                    logger.log(Level.WARNING, "Failed to initialize SAL communication layer, "
                            + "check that SAL has been setup correctly. Reverting to standalone mode.", x);
                }
            }
        };
        t.start();

        SAL<SALCommand, SALEvent, SALTelemetry> headerServiceMgr = bridge.getConfig().getHeaderServiceManager();
        if (headerServiceMgr != null) {
            Thread t2 = new Thread("SALEventReceiver") {

                @Override
                public void run() {
                    try {
                        runEventLoop(headerServiceMgr);
                    } catch (SALException x) {
                        logger.log(Level.WARNING, "Failed to initialize SAL communication layer, "
                                + "check that SAL has been setup correctly. Reverting to standalone mode.", x);
                    }
                }
            };
            t2.start();
        }
    }

    private void runCommandLoop() throws SALException {
        try {
            commandThread = Thread.currentThread();

            while (!shutdown) {
                currentCommand = mgr.getNextCommand(Duration.ofMinutes(1));
                if (currentCommand == null) {
                    logger.fine("Still waiting for a command");
                } else {
                    bridge.execute(currentCommand.getCommand());
                }
            }
            //mgr.salShutdown();
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while waiting for commands", ex);
            throw ex;
        } finally {
            commandThread = null;
        }
    }

    private void runEventLoop(SAL<SALCommand, SALEvent, SALTelemetry> headerServiceMgr) throws SALException {
        try {
            eventThread = Thread.currentThread();

            while (!shutdown) {
                SALEvent currentEvent = headerServiceMgr.getNextEvent(Duration.ofMinutes(1));
                if (currentEvent == null) {
                    logger.fine("Still waiting for an event");
                } else {
                    eventListenerList.fireEvent(currentEvent);
                }
            }
            //mgr.salShutdown();
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while waiting for event", ex);
            throw ex;
        } finally {
            eventThread = null;
        }
    }

    void addSALEventListener(EventListener<SALEvent> eventHandler) {
        eventListenerList.addEventListener(eventHandler);
    }

    void removeSALEventListener(EventListener<SALEvent> eventHandler) {
        eventListenerList.removeEventListener(eventHandler);
    }

    private void shutdown() throws InterruptedException {
        shutdown = true;
        if (commandThread != null) {
            commandThread.join();
        }
        if (eventThread != null) {
            eventThread.join();
        }
    }

    @Override
    protected void reportComplete(OCSCommandExecutor.OCSExecutor command) {
        super.reportComplete(command);
        try {
            currentCommand.reportComplete();
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", ex);
        }
    }

    @Override
    protected void reportError(OCSCommandExecutor.OCSExecutor command, Exception ex) {
        super.reportError(command, ex);
        try {
            // The useful information in a Java exception is often buried deep within nested exceptions.
            // Here we attempt to extract it so we can send a meaningfull message back to SAL.
            // Messages send back with commands are limited to 256 characters, so we need to keep it short
            int errorCode = 0;
            StringBuilder meaningfullMessage = new StringBuilder();
            for (Throwable current = ex; current != null; current = current.getCause()) {
                if (current instanceof ExecutionException) {
                    continue; // Typicailly not useful
                }
                if (current instanceof CommandInvocationException) {
                    continue; // Typicailly not useful
                }
                if (meaningfullMessage.length() > 0) {
                    meaningfullMessage.append("\n\t");
                }
                if (current.getMessage() == null) {
                    meaningfullMessage.append(current.toString());
                } else {
                    meaningfullMessage.append(current.getMessage());
                }
                if (current instanceof SALHasErrorCode && errorCode == 0) {
                    errorCode = ((SALHasErrorCode) current).getErrorCode();
                }
            }
            currentCommand.reportError(errorCode, meaningfullMessage.toString());
        } catch (SALException x) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", x);
        }
    }

    @Override
    protected void acknowledgeCommand(OCSCommandExecutor.OCSExecutor command, Duration timeout) {
        super.acknowledgeCommand(command, timeout);
        try {
            currentCommand.acknowledgeCommand(timeout);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", ex);
        }
    }

    @Override
    protected void rejectCommand(OCSCommandExecutor.OCSExecutor command, String reason) {
        super.rejectCommand(command, reason);
        try {
            currentCommand.rejectCommand(reason, 0);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", ex);
        }
    }

    @Override
    public void sendEvent(CameraEvent event) {
        super.sendEvent(event);
        try {
            mgr.logEvent(event);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error sending event to SAL "+event.getClass(), ex);
        }
    }

    @Override
    public void sendTelemetry(CameraTelemetry telemetry) {
        super.sendTelemetry(telemetry);
        try {
            mgr.sendTelemetry(telemetry);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error sending telemetry to SAL "+telemetry.getClass(), ex);
        }
    }

    @Override
    String getSALVersion() {
        return mgr.getSALVersion();
    }

    @Override
    String getXMLVersion() {
        return mgr.getXMLVersion();
    }

    @Override
    String getOSPLVersion() {
        return mgr.getOSPLVersion();
    }
}
