package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import org.lsst.ccs.imagenaming.ImageName;
import static org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.MOVE_TIME;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterReadinessState;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSMeasuredShutterTime;

/**
 * Shutter simulation for use with MCM
 *
 * @author tonyj
 */
public class ShutterSimulation implements ShutterInterface {

    private final CCS ccs;
    private final State shutterState;
    private final MCMConfig config;
    private ScheduledFuture<?> notReadyFuture;
    private boolean imageSequence = false;
    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME = Duration.ofMillis(4000);

    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME_FOR_SEVERAL_IMAGES = Duration.ofMillis(10000);
    private ImageName currentImageName;
    private long currentOpenTime;

    ShutterSimulation(CCS ccs, State shutterState, MCMConfig config) {
        this.ccs = ccs;
        this.shutterState = shutterState;
        this.config = config;
        // When the shutter is closed, we only keep the motors powered up for a limited time
        if (config.getShutterPrepTime() > 0) {
            shutterState.addStateChangeListener((when, state, oldState, cause) -> {
                if (state == ShutterState.CLOSED) {
                    scheduleNotReady();
                } else {
                    cancelNotReady();
                }
            });
        }

    }

    @Override
    public void start(String configName) throws ExecutionException {
        // NOOP
    }

    @Override
    public void expose(ImageName imageName, Duration exposureTime) throws ExecutionException {
        shutterState.checkState(ShutterState.CLOSED);
        shutterState.setState(ShutterState.OPENING);
        // TODO: This does not correctly handle the case when both blades move at once
        ccs.schedule(MOVE_TIME, () -> {
            shutterState.setState(ShutterState.OPEN);
        });
        ccs.schedule(exposureTime, () -> {
            shutterState.setState(ShutterState.CLOSING);
        });
        ccs.schedule(exposureTime.plus(MOVE_TIME), () -> {
            shutterState.setState(ShutterState.CLOSED);
        });
        // TODO: We should only send CCSMeasuredShutterTime for MAIN_CAMERA
        if (config.getCameraType() == config.getCameraType().MAIN_CAMERA && imageName != null) {
//            double measuredShutterOpenTime = (System.currentTimeMillis() - currentOpenTime) / 1000.0;
            double measuredShutterOpenTime = exposureTime.getSeconds();
            ccs.schedule(exposureTime.plus(MOVE_TIME).plus(Duration.ofMillis(300)), () -> {
                // send a dummy CCSMeasuredShutterTime with a fake time, and an image Name
                CCSMeasuredShutterTime cst = new CCSMeasuredShutterTime(imageName, measuredShutterOpenTime + 0.1);
                ccs.fireEvent(cst);
            });
        }
    }

    @Override
    public void open(ImageName imageName) throws ExecutionException {
        this.currentImageName = imageName;
        this.currentOpenTime = System.currentTimeMillis();
        shutterState.setState(ShutterState.OPENING);
        ccs.schedule(MOVE_TIME, () -> {
            shutterState.setState(ShutterState.OPEN);
        });
    }

    @Override
    public void close() throws ExecutionException {
        shutterState.setState(ShutterState.CLOSING);
        ccs.schedule(MOVE_TIME, () -> {
            shutterState.setState(ShutterState.CLOSED);
        });
        if (currentImageName != null) {
            if (config.getCameraType() == config.getCameraType().MAIN_CAMERA) {
                double measuredShutterOpenTime = (System.currentTimeMillis() - currentOpenTime) / 1000.0;
                CCSMeasuredShutterTime cst = new CCSMeasuredShutterTime(currentImageName, measuredShutterOpenTime);
                ccs.schedule(Duration.ofMillis(300), () -> {
                    ccs.fireEvent(cst);
                });
            }
            currentImageName = null;
        }
    }

    @Override
    public void prepare() {
        if (config.getShutterPrepTime() > 0) {
            cancelNotReady();
            ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), new State(ShutterReadinessState.GETTING_READY));
            ccs.schedule(Duration.ofMillis((long) (config.getShutterPrepTime() * 1000)), () -> {
                ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), new State(ShutterReadinessState.READY));
                scheduleNotReady();
            });
        }
    }

    private void scheduleNotReady() {

        Duration ready_time;
        if (imageSequence == true) {
            ready_time = READY_TIME_FOR_SEVERAL_IMAGES;
        } else {
            ready_time = READY_TIME;

        }

        notReadyFuture = ccs.schedule(ready_time, () -> {
            ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), new State(ShutterReadinessState.NOT_READY));
        });
    }

    private void cancelNotReady() {
        if (notReadyFuture != null) {
            notReadyFuture.cancel(false);
        }
    }

    @Override
    public void setImageSequence(boolean imageSequence) {
        this.imageSequence = imageSequence;
    }

}
