package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.JsonFile;
import org.lsst.ccs.subsystem.ocsbridge.events.ShutterMotionProfileFitResult;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to the main camera shutter subsystem.
 *
 * @author tonyj
 */
class MainCameraShutterSubsystemLayer extends ControlledSubsystem implements ShutterInterface {

    private static final Logger LOG = Logger.getLogger(MainCameraShutterSubsystemLayer.class.getName());
    private final static Map<Enum<org.lsst.ccs.subsystem.shutter.common.PhysicalState>, Enum<ShutterState>> CAMERA_SHUTTER_TO_SHUTTER_STATE = new HashMap<>();
    private volatile CountDownLatch motionDoneCountDown;
    private volatile ImageName currentImageName;
    private final ShutterMotionProfileSender motionProfileSender = new ShutterMotionProfileSender();
    
    static final String MOTOR_ENCODER_PATH = "shutter/motorEncoder/";
    static final String HALL_SENSOR_PATH = "shutter/hallSensor/";
    static final String MOTION_PROFILE_FIT_RESULT = "/motionProfileFitResult";

    static {
        // TODO: Camera has other states. We need to decide how to map them.
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.OPENED, ShutterState.OPEN);
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.OPENING, ShutterState.OPENING);
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.CLOSED, ShutterState.CLOSED);
        CAMERA_SHUTTER_TO_SHUTTER_STATE.put(org.lsst.ccs.subsystem.shutter.common.PhysicalState.CLOSING, ShutterState.CLOSING);
    }

    MainCameraShutterSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getShutterSubsystemName(), ccs, config);
    }

    @Override
    public void expose(ImageName imageName, Duration exposeTime) throws ExecutionException {
        // We should get two MotionDone publications as a result of the takeExposure command, one for each blade.
        try {
            motionDoneCountDown = new CountDownLatch(2);
            currentImageName = imageName;
            commandSender.sendCommand("takeExposure", exposeTime.toMillis() / 1000.);
            boolean success = motionDoneCountDown.await(exposeTime.toMillis() + 2000, TimeUnit.MILLISECONDS);
            if (!success) {
                throw new TimeoutException("Timed out waiting for shutter exposure to complete");
            }
        } catch (InterruptedException | TimeoutException x) {
            throw new ExecutionException("Error during expose", x);
        }
    }

    @Override
    public void open() throws ExecutionException {
        commandSender.sendCommand("openShutter");
    }

    @Override
    public void close() throws ExecutionException {
        commandSender.sendCommand("closeShutter");
    }

    @Override
    public void prepare() {
        // This is a NOOP for the main camera shutter, it is always ready
    }
    
    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        changedStates.getComponentStateBundle("statemachine").getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateCameraShutterStateToShutterState(when, value);
        });
    }

    @Override
    protected void onEvent(StatusMessage msg) {
        if (msg.getObject() instanceof KeyValueData &&  "MotionDone".equals(((KeyValueData) msg.getObject()).getKey()) && motionDoneCountDown != null) {
            LOG.log(Level.INFO, "Got Motion Done {0} ", msg);
            int currentCount = (int) motionDoneCountDown.getCount();
            if (currentCount > 0) {
                motionDoneCountDown.countDown();
                MotionDone motionDone = (MotionDone) ((KeyValueData) msg.getObject()).getValue();
                processMotionDone(currentImageName, currentCount, motionDone);
            }
        }
    }

    private void translateCameraShutterStateToShutterState(CCSTimeStamp when, Enum value) {
        LOG.log(Level.INFO, "Got shutter state {0} ", value);
        Enum<ShutterState> converted = CAMERA_SHUTTER_TO_SHUTTER_STATE.get(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted));
        }
    }

    @Override
    public void setImageSequence(boolean imageSequence) {
        // We don't care
    }

    private void processMotionDone(ImageName imageName, int currentCount, MotionDone motionDone) {
        final boolean isOpen = currentCount==2;
        // TODO: We also need to send this info to trending (LCOBM-106) and send it to the LFA
        motionProfileSender.processMotionDone(imageName, motionDone, isOpen);
        JsonFile jsonProfile = motionProfileSender.getJsonFile();
        sendEvent(AdditionalFile.EVENT_KEY, jsonProfile);
        
        ShutterMotionProfileFitResult encoderFitResult = motionProfileSender.getEncoderFitResult();
        String encoderPath = MOTOR_ENCODER_PATH + motionDone.side().toString()+"/"+(isOpen ? "open" : "close")+MOTION_PROFILE_FIT_RESULT;
        sendEvent(encoderPath, encoderFitResult);

        ShutterMotionProfileFitResult hallSensorFitResult = motionProfileSender.getHallSensorFitResult();
        String hallSensorPath = HALL_SENSOR_PATH + motionDone.side().toString()+"/"+(isOpen ? "open" : "close")+MOTION_PROFILE_FIT_RESULT;
        sendEvent(hallSensorPath, hallSensorFitResult);
    }
}
