package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.Serializable;
import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.imagenaming.ImageName;
import static org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.CLEAR_TIME;
import static org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.DISCARD_TIME;
import static org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.READOUT_TIME;
import org.lsst.ccs.subsystem.ocsbridge.states.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class FocalPlaneSimulation implements FocalPlaneInterface {

    private final State raftsState;
    private final CCS ccs;

    /**
     * Idle time before a clear is required
     */
    static final Duration QUIESCENT_BEFORE_CLEAR = Duration.ofMillis(4000);

    public FocalPlaneSimulation(CCS ccs, State raftsState) {
        this.raftsState = raftsState;
        this.ccs = ccs;
        // Whenever we enter quiescent state, we start a timer to indicate when a clear is needed
        raftsState.addStateChangeListener(new State.StateChangeListener<RaftsState>() {
            private ScheduledFuture<?> clearFuture;

            @Override
            public void stateChanged(CCSTimeStamp when, RaftsState currentState, RaftsState oldState, String cause) {
                if (currentState == RaftsState.QUIESCENT) {
                    clearFuture = ccs.schedule(QUIESCENT_BEFORE_CLEAR, () -> {
                        raftsState.setState(when, RaftsState.NEEDS_CLEAR, "Spent more than "+ QUIESCENT_BEFORE_CLEAR +" in quiescent state.");
                    });
                } else {
                    if (clearFuture != null) {
                        clearFuture.cancel(false);
                        clearFuture = null;
                    }
                }
            }

        });

    }

    @Override
    public void clear(int nClears) {
        raftsState.checkState(RaftsState.QUIESCENT, RaftsState.NEEDS_CLEAR);
        raftsState.setState(RaftsState.CLEARING);
        ccs.schedule(CLEAR_TIME.multipliedBy(nClears), () -> {
            raftsState.setState(RaftsState.QUIESCENT);
        });
    }

    @Override
    public void discardRows(int nRows) throws ExecutionException {
        raftsState.checkState(RaftsState.INTEGRATING);
        raftsState.setState(RaftsState.DISCARDING);
        ccs.schedule(DISCARD_TIME.multipliedBy(nRows), () -> {
            raftsState.setState(RaftsState.INTEGRATING);
        });
    }

    @Override
    public void endIntegration(boolean readout) throws ExecutionException {
        raftsState.checkState(RaftsState.INTEGRATING);
        if (readout) {
            raftsState.setState(RaftsState.READING_OUT);
            ccs.schedule(READOUT_TIME, () -> {
                raftsState.setState(RaftsState.QUIESCENT);
            });
        } else {
            raftsState.setState(RaftsState.NEEDS_CLEAR);
        }
    }

    @Override
    public void start(String configName) throws ExecutionException {
        // NOOP
    }

    @Override
    public void startIntegration(ImageName imageName, Map<String, ? extends Serializable> parsedKeyValueData, Set locations, String annotation, Duration stopGuiding) throws ExecutionException {
        raftsState.checkState(RaftsState.QUIESCENT);
        raftsState.setState(RaftsState.INTEGRATING);
    }

    @Override
    public void play(String playlist, boolean repeat) {
        // Nothing to do at the moment
    }

    @Override
    public void definePlaylist(String playlist, String folder, String... images) {
        // Nothing to do at the moment
    }

    @Override
    public void clearAndStartIntegration(ImageName imageName, int nClears, Map<String, ? extends Serializable> parsedKeyValueData, Set locations, String annotation, Duration stopGuiding) throws ExecutionException {
        try {
            clear(nClears);
            ccs.waitForStatus(RaftsState.QUIESCENT).get(CLEAR_TIME.toMillis()+1000, TimeUnit.MILLISECONDS);
            startIntegration(imageName, parsedKeyValueData, locations, annotation, stopGuiding);
        } catch (InterruptedException ex) {
            throw new ExecutionException("Interrupt waiting for clear", ex);
        } catch (TimeoutException ex) {
            throw new ExecutionException("Timeout waiting for clear", ex);
        }
    }

    @Override
    public void initGuiders(String roiSpec) {
        // Nothing to do at the moment
    }

    @Override
    public void clearROI() throws ExecutionException {
        // Nothing to do at the moment
    }

    @Override
    public void setHeaderKeywords(Map<String, Serializable> headersMap) {
       // Nothing to do at the moment
    }

}
