package org.lsst.ccs.subsystem.ocsbridge;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPathExpressionException;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.camera.sal.xml.util.ChecksumExtractorUtils;
import org.lsst.ccs.camera.sal.xml.util.SerializationUtils;
import org.w3c.dom.Document;
import org.lsst.ccs.camera.sal.xml.MakeXMLConfiguration;
import org.lsst.ccs.camera.sal.xml.XMLMaker2;
import org.lsst.ccs.camera.sal.xml.XMLMaker2.SALType;
import org.xml.sax.SAXException;

/**
 *
 * A CCS Module that generates serialized command, data, configuration, monitoring and
 * trending files for the generation and testing of XML files.
 *
 * @author The LSST CCS Team
 */
public class DataAndDictionarySerializer implements HasLifecycle, AgentPresenceListener {

    private static final Logger LOG = Logger.getLogger(DataAndDictionarySerializer.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Agent agent;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dpdService;

    //Checksum variables
    private ChecksumExtractorUtils extractor;
    private final Map<String,Long> xmlChecksums = new HashMap<>();

    @ConfigurationParameter(description = "The Camera name", isFinal = true, units = "unitless")
    private Camera camera = Camera.COMCAM;
    
    @ConfigurationParameter(description = "The output directory", units = "unitless")
    private String outputDir = "/tmp/";
    private File outputLocation;
    
    private final DictionaryListener dataDictionaryListener = new DictionaryListener();
    private final TrendingAndMonitoringDataListener trendingAndMonitoringDataListener = new TrendingAndMonitoringDataListener();
    private final ConfigurationListener configurationListener = new ConfigurationListener();

    private final Predicate<BusMessage> workersAndServicesOnly = (bm) -> {
        return ((StatusMessage) bm).getOriginAgentInfo().isAgentWorkerOrService();
    };

    @Override
    public void init() {
        dpdService.addDataProviderDictionaryListener(dataDictionaryListener);
        agent.getMessagingAccess().addStatusMessageListener(trendingAndMonitoringDataListener, BusMessageFilterFactory.messageClass(StatusSubsystemData.class).and(workersAndServicesOnly));
        agent.getMessagingAccess().addStatusMessageListener(configurationListener, BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class).and(workersAndServicesOnly));
        
        InputStream is = DataAndDictionarySerializer.class.getResourceAsStream("/xml/"+camera.getCscName()+SALType.TELEMETRY.getXMLFileSuffix());
        try {
            extractor = new ChecksumExtractorUtils();
            xmlChecksums.putAll(extractor.extractChecksumsFromSALXMLInputStream(is));
        } catch (XPathExpressionException | SAXException | IOException | ParserConfigurationException xe) {
            throw new RuntimeException("Failed to extract checksums : ",xe);
        }
        is = DataAndDictionarySerializer.class.getResourceAsStream("/xml/"+camera.getCscName()+SALType.SETTINGS_APPLIED.getXMLFileSuffix());
        try {
            xmlChecksums.putAll(extractor.extractChecksumsFromSALXMLInputStream(is));
        } catch (XPathExpressionException | SAXException | IOException | ParserConfigurationException xe) {
            throw new RuntimeException("Failed to extract checksums : ",xe);
        }

    }

    @Override
    public void shutdown() {
        dpdService.removeDataProviderDictionaryListener(dataDictionaryListener);
        agent.getMessagingAccess().removeStatusMessageListener(trendingAndMonitoringDataListener);
        agent.getMessagingAccess().removeStatusMessageListener(configurationListener);
    }
    
    @ConfigurationParameterChanger(propertyName = "outputDir")
    public void setOutputDir(String outputDir) {
        try {
            outputLocation = new File(outputDir);
            outputLocation.mkdirs();
        } catch (Exception e) {
            throw new RuntimeException("Failed to set configuration parameter xmlOutputDir to "+outputDir,e);
        }
        this.outputDir = outputDir;
    }
    

    @Command(type = Command.CommandType.QUERY, level = 0)
    public void writeSerializedFilesForAllAgents() {
        for ( String agentName: dataDictionaryListener.getAllAgents() ) {
            writeSerializedFilesForAgent(agentName);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = 0)
    public void writeSerializedFilesForAgent(String agentName) {

        //Data Dictionary
        writeDataDictionaryForAgent(agentName);

        //ConfigurationInfo
        Object configInfo = configurationListener.getConfigurationInfoForAgent(agentName);
        writeObject(configInfo, agentName + "-config-info.ser");

        //Monitoring data
        Map<String, StatusSubsystemData> monitoringMap = trendingAndMonitoringDataListener.getMonitoringDataForAgent(agentName);
        for (Entry<String, StatusSubsystemData> e : monitoringMap.entrySet()) {
            String taskName = e.getKey();
            StatusSubsystemData ssd = e.getValue();
            String outputName = taskName.isEmpty() ? agentName + "-monitoring.ser" : agentName + "-" + taskName.replace("/", "_") + "-monitoring.ser";
            writeObject(ssd, outputName);
        }

        //Trending data
        Map<String, StatusSubsystemData> trendMap = trendingAndMonitoringDataListener.getTrendingDataForAgent(agentName);
        for (Entry<String, StatusSubsystemData> e : trendMap.entrySet()) {
            String taskName = e.getKey();
            StatusSubsystemData ssd = e.getValue();
            String outputName = taskName.isEmpty() ? agentName + "-trending.ser" : agentName + "-" + taskName.replace("/", "_") + "-trending.ser";
            writeObject(ssd, outputName);
        }
    }

    private void writeDataDictionaryForAgent(String agentName) {
        Object dataDict = dataDictionaryListener.getDataDictionaryForAgent(agentName);
        writeObject(dataDict, agentName + "-status-dictionary.ser");        
    }
    
    @Command(type = Command.CommandType.QUERY, level = 0)
    public void writeXMLFilesForAgent(String agentName) throws ParserConfigurationException, IOException, ClassNotFoundException, TransformerException, SAXException, XPathExpressionException {
        writeXMLFilesForAgent(agentName, true, true);
        
    }
    public void writeXMLFilesForAgent(String agentName, boolean writeEvents, boolean writeTelemetry) throws ParserConfigurationException, IOException, ClassNotFoundException, TransformerException, SAXException, XPathExpressionException {
        DataProviderDictionary dictionary = dataDictionaryListener.getDataDictionaryForAgent(agentName);
        writeXMLFilesFromSerializedDictionary(agentName, dictionary, writeEvents, writeTelemetry);
    }

    private void writeXMLFilesFromSerializedDictionary(String agentName, DataProviderDictionary dict, boolean writeEvents, boolean writeTelemetry) throws ParserConfigurationException, IOException, ClassNotFoundException, TransformerException, SAXException, XPathExpressionException {

        XMLMaker2 maker = new XMLMaker2(false);

        if (writeTelemetry) {
            MakeXMLConfiguration config = MakeXMLConfiguration.getInstance(camera, XMLMaker2.SALType.TELEMETRY, agentName, dict);
            writeXMLForConfig(config, maker, agentName);
        }

        if (writeEvents) {
            // Now do the same for settings applied        
            MakeXMLConfiguration config = MakeXMLConfiguration.getInstance(camera, XMLMaker2.SALType.SETTINGS_APPLIED, agentName, dict);
            writeXMLForConfig(config, maker, agentName);
        }

    }

    
    @Command(type = Command.CommandType.QUERY, level = 0)
    public String printTrendingQuantities() {
        String result = printTrendingQuantitiesFor(dataDictionaryListener.agentDataDictionary.entrySet());
        result += "\n";
        result += printTrendingQuantitiesFor(dataDictionaryListener.ignoredAgentDataDictionary.entrySet());
        return result;
    }
    private String printTrendingQuantitiesFor(Set<Entry<String,DataProviderDictionary>> set) {
        StringBuilder sb = new StringBuilder("Trending quantities by agent:");        
        for( Entry<String,DataProviderDictionary> e : set ) {
            String agentName = e.getKey();
            sb.append("\n--- ").append(agentName);
            DataProviderDictionary dict = e.getValue();
            int count = 0;
            for ( DataProviderInfo dpi : dict.getDataProviderInfos() ) {
                if ( DataProviderInfo.Type.TRENDING.name().equals(dpi.getAttributeValue(DataProviderInfo.Attribute.DATA_TYPE)) ) {
                    if ( !dpi.getFullPath().startsWith("runtime") ) {
                        count++;
                        sb.append("\n    -").append(count).append("- ").append(dpi.toString());                        
                    }
                }
            }
            if ( count == 0 ) {
                sb.append(" no trending quantities in dictionary");
            }           
        }
        return sb.toString();        
    }
    
    @Command(type = Command.CommandType.QUERY, level = 0)
    public void writeXMLFilesFromSerializedDictionary(String agentName, String dictionaryFileName) throws ParserConfigurationException, IOException, ClassNotFoundException, TransformerException, SAXException, XPathExpressionException {
        DataProviderDictionary dict = SerializationUtils.readDictionaryFromFile(dictionaryFileName);        
        writeXMLFilesFromSerializedDictionary(agentName, dict, true, true);
    }
    

    private void writeXMLForConfig(MakeXMLConfiguration config, XMLMaker2 maker, String agentName) throws ParserConfigurationException, TransformerException {
        File outputFileName = new File(outputLocation, config.getXMLFileNameForAgent(agentName));
        Document document = maker.createXML(config);
        maker.writeXML(new StreamResult(outputFileName), document);
        LOG.log(Level.INFO, "Wrote {0}", outputFileName);
    }

    private void writeObject(Object obj, String name) {
        File outputFileName = new File(outputLocation, name);        
        LOG.log(Level.INFO, "Writing out file {0}", outputFileName.getAbsolutePath());
        try ( ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(outputFileName))) {
            oos.writeObject(obj);
        } catch (IOException ex) {
            LOG.log(Level.WARNING, "Failed to write " + outputFileName.getAbsolutePath(), ex);
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo ai : agents) {
            String agentName = ai.getName();
            dataDictionaryListener.removeAgent(agentName);
            trendingAndMonitoringDataListener.removeAgent(agentName);
            configurationListener.removeAgent(agentName);
        }
    }

    
    private class DictionaryListener implements DataProviderDictionaryService.DataProviderDictionaryListener {

        private final Map<String, DataProviderDictionary> agentDataDictionary = new ConcurrentHashMap<>();
        private final Map<String, DataProviderDictionary> ignoredAgentDataDictionary = new ConcurrentHashMap<>();
        private MakeXMLConfiguration xmlConfig = null;
        
        
        private MakeXMLConfiguration getXmlConfiguration() {
            if (xmlConfig == null) {
                xmlConfig = MakeXMLConfiguration.getInstance(camera, SALType.TELEMETRY);
            }
            return xmlConfig;
        }
        
        
        @Override
        public void dataProviderDictionaryUpdate(DataProviderDictionaryService.DataProviderDictionaryEvent evt) {
            try {
                if (evt.getEventType() == DataProviderDictionaryService.DataProviderDictionaryEvent.EventType.ADDED) {
                    AgentInfo.AgentType type = evt.getAgentInfo().getType();
                    if (type == AgentInfo.AgentType.WORKER || type == AgentInfo.AgentType.SERVICE) {

                        AgentInfo agentInfo = evt.getAgentInfo();
                        String agentName = agentInfo.getName();
                        
                        if ( getXmlConfiguration().getDictionaryConfigurationForAgentInfo(agentInfo) == null ) {
                             if ( ! ignoredAgentDataDictionary.containsKey(agentName) ) {
                                ignoredAgentDataDictionary.put(agentName, evt.getDictionary());
                            } 
                            LOG.log(Level.INFO,"Ignoring dictionary for agent {0}; it's not an agent worth listening to.", agentName);                            
                            return;
                        }
                        if ( !agentDataDictionary.containsKey(agentName) ) {
                            DataProviderDictionary dict = evt.getDictionary();
                            agentDataDictionary.put(agentName, dict);
                            
                            LOG.log(Level.INFO,"Processing checksums for agent {0}", agentName);
                            Map<String,Long> dictionaryChecksums = extractor.extractChecksumsFromDataProviderDictionaryForAgent(camera, agentName,  dict);
                                    
                            boolean writeEventsXml = false;
                            boolean writeTelemetryXml = false;
                            boolean writeDataDictionary = false;

                            for (Map.Entry<String, Long> entry : dictionaryChecksums.entrySet()) {
                                String topic = entry.getKey();
                                
                                if ( topic.contains("InfluxDb") ) 
                                    continue;            
                                
                                Long checksum = entry.getValue();
                                String topicName = topic;
                                if ( topic.contains("logevent") ) {
                                    topicName += "Configuration";
                                }
                                Long xmlChecksum = xmlChecksums.get(topicName);
                                boolean equals = checksum.equals(xmlChecksum);
                                if (!equals) {
                                    if (xmlChecksum == null) {
                                        LOG.log(Level.WARNING, "Missing topic in XML: {0}", topic);
                                    } else {
                                        LOG.log(Level.WARNING, "Checksum mismatch for {0} ({1} != {2})", new Object[]{topic, checksum, xmlChecksum});
                                    }
                                    if (topic.contains("logevent")) {
                                        writeEventsXml = true;
                                    } else {
                                        writeTelemetryXml = true;
                                    }
                                }
                                writeDataDictionary = true;
                            }

                            if (writeEventsXml || writeTelemetryXml) {
                                LOG.log(Level.INFO, "Writing xml files for agent {0} (events:{1}, telemetry:{2})", new Object[]{agentName, writeEventsXml, writeTelemetryXml});
                                try {
                                    writeXMLFilesForAgent(agentName, writeEventsXml, writeTelemetryXml);
                                } catch (IOException | ClassNotFoundException | ParserConfigurationException | TransformerException | XPathExpressionException | SAXException e) {
                                    LOG.log(Level.WARNING, "Failed to write xml for agent " + agentName, e);
                                }
                            }

                            if (writeDataDictionary) {
                                writeDataDictionaryForAgent(agentName);
                            }
                        }    
                    }
                }
            } catch (Throwable t) {
                LOG.log(Level.SEVERE, "Exception adding dictionary", t);
            }
        }

        public DataProviderDictionary getDataDictionaryForAgent(String agentName) {
            return agentDataDictionary.get(agentName);
        }

        public void removeAgent(String agentName) {
            agentDataDictionary.remove(agentName);
        }

        public Set<String> getAllAgents() {
            return agentDataDictionary.keySet();
        }
        
    }
        
    private class TrendingAndMonitoringDataListener implements StatusMessageListener {

        private final Map<String, Map<String, StatusSubsystemData>> agentMonitoringData = new ConcurrentHashMap<>();
        private final Map<String, Map<String, StatusSubsystemData>> agentTrendingData = new ConcurrentHashMap<>();

        @Override
        public void onStatusMessage(StatusMessage msg) {
            StatusSubsystemData ssd = (StatusSubsystemData) msg;
            if (!ssd.getDataKey().endsWith("State")) {
                String agentName = msg.getOriginAgentInfo().getName();
                KeyValueData kvd = ssd.getSubsystemData();
                if (kvd instanceof KeyValueDataList) {
                    KeyValueDataList kvdl = (KeyValueDataList) kvd;
                    String type = (String) kvdl.getAttribute("publicationType");
                    String taskName;
                    if (type == null) {
                        //This is Trending data
                        taskName = ((KeyValueDataList) kvd).getListOfKeyValueData().get(0).getKey()+"-"+dpdService.calculateHashCodeForData(ssd);
                        agentTrendingData.computeIfAbsent(agentName, (k) -> new ConcurrentHashMap<>()).put(taskName, ssd);
                    } else {
                        //This is Monitoring data
                        taskName = (String) kvdl.getAttribute("taskName");
                        if ("scheduledFull".equals(type)) {
                            taskName = "";
                        }
                        agentMonitoringData.computeIfAbsent(agentName, (k) -> new ConcurrentHashMap<>()).put(taskName, ssd);
                    }
                }
            }
        }

        public Map<String, StatusSubsystemData> getMonitoringDataForAgent(String agentName) {
            Map<String,StatusSubsystemData> result = agentMonitoringData.get(agentName);
            if ( result != null )
                return new HashMap(result);
            return new HashMap();
        }
        public Map<String, StatusSubsystemData> getTrendingDataForAgent(String agentName) {
            Map<String,StatusSubsystemData> result = agentTrendingData.get(agentName);
            if ( result != null )
                return new HashMap(result);
            return new HashMap();
        }

        public void removeAgent(String agentName) {
            agentMonitoringData.remove(agentName);
            agentTrendingData.remove(agentName);
        }
    }

    private class ConfigurationListener implements StatusMessageListener {

        private final Map<String, StatusConfigurationInfo> agentConfigurationData = new ConcurrentHashMap<>();

        @Override
        public void onStatusMessage(StatusMessage msg) {
            String agentName = msg.getOriginAgentInfo().getName();
            agentConfigurationData.put(agentName, (StatusConfigurationInfo) msg);
        }

        public StatusConfigurationInfo getConfigurationInfoForAgent(String agentName) {
            return agentConfigurationData.get(agentName);
        }

        public void removeAgent(String agentName) {
            agentConfigurationData.remove(agentName);
        }
    }

}
