package org.lsst.ccs.subsystem.ocsbridge.util;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Encapsulates a state, and generates state change events.
 * <p>
 * This class is thread-safe, its methods can be called on any thread. Listeners are
 * guaranteed to be notified sequentially. For every listener, the order of notifications
 * corresponds to the order of state changes. However, the internal state of this {@code State}
 * instance is not guaranteed to remain unchanged by the the time listeners are notified.
 * <p>
 * The internal state of this {@code State} instance is protected by its monitor lock.
 * The way to extract multiple state parameters (value and cause) is to call corresponding
 * getters while holding this lock.
 * 
 * @author tonyj
 * @param <T> The enumeration representing the states.
 */
public class State<T extends Enum> {

    private T currentState;           // protected by "this" monitor
    private String currentCause;      // protected by "this" monitor
    private boolean callingListeners; // protected by "this" monitor
    
    private final Class<T> enumClass;
    private final List<StateChangeListener<T>> listeners = new CopyOnWriteArrayList<>();
    private static final Logger logger = Logger.getLogger(State.class.getName());
    private final ConcurrentLinkedDeque<Runnable> workQueue = new ConcurrentLinkedDeque<>();

    /**
     * Create a new state
     * @param initialState Initial state
     */
    public State(T initialState) {
        this.enumClass = (Class<T>) initialState.getClass();
        currentState = initialState;
    }

    /**
     * Changes the current state. Generates a status change notification when
     * the status is changed.
     *
     * @param state The new state
     */
    public void setState(T state) {
        setState(CCSTimeStamp.currentTime(), state, null);
    }

    public void setState(CCSTimeStamp when, T state) {
        setState(when, state, null);
    }
    
    public void setState(CCSTimeStamp when, State<T> state) {
        T stateValue;
        String stateCause;
        synchronized (state) {
            stateValue = state.currentState;
            stateCause = state.currentCause;
        }
        setState(when, stateValue, stateCause);
    }
    
    public void setState(CCSTimeStamp when, T state, String cause) {
        
        T oldState;
        boolean willCallListeners;
        synchronized (this) {
            if (currentState == state) return;
            oldState = currentState;
            currentState = state;
            currentCause = cause;

            // Instead of calling the listeners immediately we queue them up. This ensures that if
            // a listener triggers a state change the order of executution will be more determinate.
            listeners.forEach((l) -> {
                workQueue.add(() -> l.stateChanged(when, state, oldState, cause));
            });
            if (callingListeners) { // some other thread is already calling listeners from the queue
                willCallListeners = false;
            } else {
                willCallListeners = true;
                callingListeners = true;
            }
        }
        
        StringBuilder sb = new StringBuilder();
        sb.append("State Changed ").append(state.getClass().getSimpleName()).append(": ").append(oldState);
        sb.append("->").append(state).append(".");
        if (cause != null) {
            sb.append(" Cause: ").append(cause).append(".");
        }
        logger.log(Level.INFO, sb.toString());
        
        while (willCallListeners) {
            Runnable runnable;
            while ((runnable = workQueue.pollFirst()) != null) {
                runnable.run();
            }
            synchronized (this) {
                if (workQueue.isEmpty()) {
                    willCallListeners = false;
                    callingListeners = false;
                }
            }
        }
    }

    synchronized public T getState() {
        return currentState;
    }
    
    synchronized public String getCause() {
        return currentCause;
    }

    public Class<T> getEnumClass() {
        return enumClass;
    }

    synchronized public boolean isInState(T state) {
        return currentState == state;
    }
    
    public void addStateChangeListener(StateChangeListener<T> listener) {
        listeners.add(listener);
    }
    
    public void removeStateChangeListener(StateChangeListener<T> listener) {
        listeners.remove(listener);
    }

    /**
     * Check the state and generate an exception if the current state does not
     * match.
     *
     * @param states The expected state, may be a list of possible states
     * @throws InvalidStateException If the state does not match.
     */
    synchronized public void checkState(T... states) throws InvalidStateException {
        for (T state : states) {
            if (state == currentState) {
                return;
            }
        }
        throw new InvalidStateException(String.format("State: %s expected %s was %s", enumClass.getSimpleName(), Arrays.toString(states), currentState));
    }

    public static class InvalidStateException extends RuntimeException {

        private static final long serialVersionUID = -8878766853816068565L;

        public InvalidStateException(String reason) {
            super(reason);
        }
    }

    public static interface StateChangeListener<T extends Enum> {
        void stateChanged(CCSTimeStamp when, T state, T oldState, String cause);
    }

    @Override
    synchronized public String toString() {
        return "State{" + enumClass.getSimpleName() +" = "+ currentState +'}';
    }
}
