package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;

/**
 * Configuration class to hold parameters for the MCM
 *
 * @author Farrukh Azfar
 */
@SuppressWarnings("FieldMayBeFinal")
public class MCMConfig {

    @ConfigurationParameter
    private volatile boolean hasFilterChanger = true;
    
    @ConfigurationParameter
    private volatile int maxImagesPerSequence = 10;
    
    @ConfigurationParameter(units = "seconds")
    private volatile double minExposeTime = 1.0;
    
    @ConfigurationParameter(units = "seconds")
    private volatile double maxExposeTime = 30.0;
    
    @ConfigurationParameter
    private volatile String focalPlaneSubsystemName;

    @ConfigurationParameter
    private volatile String shutterSubsystemName;

    @ConfigurationParameter
    private volatile String filterChangerSubsystemName;

    @ConfigurationParameter
    private volatile Camera cameraType = Camera.MAIN_CAMERA;

    @ConfigurationParameter(maxLength = 100)
    private volatile Set<String> requiredKeywords = new HashSet<>(Arrays.asList("imageType", "groupId"));

    @ConfigurationParameter(maxLength = 100)
    private volatile Set<String> allowedKeywords = new HashSet<>(Arrays.asList("testType"));

    @ConfigurationParameter(description = "List of additional subsystems which need to have their configuration broadcast", maxLength = 100)
    private volatile Set<String> otherConfiguredSubsystems;
    
    @ConfigurationParameter(description = "List of additional subsystems to be locked by MCM")
    private volatile Set<String> otherLockedSubsystems;
    
    @ConfigurationParameter(description = "Time for shutter to prepare to move. If zero then no preparation time is needed", units="seconds")
    private volatile double shutterPrepTime = 0.15;

    @ConfigurationParameter(description = "Controls whether we always command focal-plane to do a clear on all images")
    private volatile boolean alwaysClear = false;

    MCMConfig() {

    }

    public MCMConfig(Camera cameraType) {
        this.cameraType = cameraType;

    }

    public String getFocalPlaneSubsystemName() {
        return focalPlaneSubsystemName;
    }

    boolean hasFocalPlaneSubsystem() {
        return getFocalPlaneSubsystemName() != null && !getFocalPlaneSubsystemName().trim().isEmpty();
    }
    
    public String getShutterSubsystemName() {
        return shutterSubsystemName;
    }
    
    boolean hasShutterSubsystem() {
        return getShutterSubsystemName() != null && !getShutterSubsystemName().trim().isEmpty();
    }
    
    String getFilterChangerSubsystemName() {
        return filterChangerSubsystemName;
    }

    boolean hasFilterChanger() {
        return hasFilterChanger;
    }
    
    public boolean hasFilterChangerSubsystem() {
        return hasFilterChanger && getFilterChangerSubsystemName() != null && !getFilterChangerSubsystemName().trim().isEmpty();
    }


    public int getMaxImagesPerSequence() {
        return maxImagesPerSequence;
    }

    public double getMinExposeTime() {
        return minExposeTime;
    }

    public double getMaxExposeTime() {
        return maxExposeTime;
    }

    public Camera getCameraType() {
        return cameraType;
    }

    Set<String> getRequiredKeys() {
        return requiredKeywords;
    }

    Set<String> getAllowedKeys() {
        HashSet<String> result =  new HashSet<>();
        result.addAll(requiredKeywords);
        result.addAll(allowedKeywords);
        return result;
    }
    
    Set<String> getOtherConfiguredSubsystems() {
        return otherConfiguredSubsystems;
    }
    
    Set<String> getOtherLockedSubsystems() {
        return otherLockedSubsystems;
    }

    public static MCMConfig createDefaultConfig() {
        return new MCMConfig();
    }

    public static MCMConfig createNewDefaultConfig() {
        final MCMConfig mcmConfig = new MCMConfig();
        mcmConfig.alwaysClear = true;
        mcmConfig.shutterPrepTime = 0;
        return mcmConfig;
    }
    
    public double getShutterPrepTime() {
        return shutterPrepTime;
    }

    boolean isAlwaysClear() {
        return alwaysClear;
    }
}
