package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.ocsbridge.states.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to a real focal-plane subsystem.
 *
 * @author tonyj
 */
class FocalPlaneSubsystemLayer extends ControlledSubsystem implements FocalPlaneInterface {

    private static final Logger LOG = Logger.getLogger(FocalPlaneSubsystemLayer.class.getName());
    private final static Map<Enum<FocalPlaneState>, Enum<RaftsState>> FOCALPLANE_TO_RAFTS_STATE = new HashMap<>();

    static {
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.NEEDS_CLEAR, RaftsState.NEEDS_CLEAR);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.CLEARING, RaftsState.CLEARING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.INTEGRATING, RaftsState.INTEGRATING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.READING_OUT, RaftsState.READING_OUT);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.QUIESCENT, RaftsState.QUIESCENT);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.ROW_SHIFT, RaftsState.DISCARDING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.IMAGE_WAIT, RaftsState.READING_OUT);
    }
    private volatile boolean initGuidersSucceeded;
    private volatile Duration estimatedGuidingTimePeriod;
    private volatile CompletableFuture<Integer> guidingFinished;

    FocalPlaneSubsystemLayer(MCMSubsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFocalPlaneSubsystemName(), ccs, config);
    }

    @Override
    public void endIntegration(boolean readout) throws ExecutionException {
        try {
            waitForGuidingToFinish(guidingFinished);
        } finally {
            commandSender.sendCommand("endIntegration");
        }
    }

    @Override
    public void startIntegration(ImageName imageName, Map<String, ? extends Serializable> parsedKeyValueData, Set locations, String annotation, Duration stopGuiding) throws ExecutionException {
        commandSender.sendCommand("setHeaderKeywords", parsedKeyValueData);
        guidingFinished = possiblyStartGuiding(imageName, stopGuiding, estimatedGuidingTimePeriod);
        commandSender.sendCommand("startNamedIntegration", imageName, annotation, locations);
    }

    private CompletableFuture<Integer> possiblyStartGuiding(ImageName imageName, Duration stopGuidingTime, Duration estimatedGuidingTimePeriod) throws ExecutionException {

        if (initGuidersSucceeded && stopGuidingTime != null) {
            CompletableFuture<Integer> guidersStopped = new CompletableFuture<>();
            commandSender.sendCommand("startGuiding", imageName);
            // Schedule the end of guiding to happen before the shutter closes
            // Scheduling stopGuiding in PT0.159S PT0.159S PT0S
            LOG.log(Level.INFO, "Scheduling stopGuiding in {0} {1} {2}", new Object[]{stopGuidingTime, estimatedGuidingTimePeriod, stopGuidingTime.minus(estimatedGuidingTimePeriod)});
            ccs.schedule(stopGuidingTime.minus(estimatedGuidingTimePeriod), () -> {
                try {
                    int nStamps = commandSender.sendCommand(Integer.class, "stopGuiding");
                    guidersStopped.complete(nStamps);
                } catch (ExecutionException x) {
                    guidersStopped.completeExceptionally(x.getCause());
                }
            });
            return guidersStopped;
        } else {
            return CompletableFuture.completedFuture(0);
        }
    }
    
    private void waitForGuidingToFinish(CompletableFuture<Integer> guidingFinished) throws ExecutionException {
        if (guidingFinished != null) {
            try {
                long start = System.currentTimeMillis();
                int nStamps = guidingFinished.get(10, TimeUnit.SECONDS);
                long stop = System.currentTimeMillis();
                LOG.log(Level.INFO, "Guiding finished after {0}ms and {1} stamps", new Object[]{stop-start, nStamps});
            } catch (ExecutionException x) {
                throw x;
            } catch (InterruptedException | TimeoutException x) {
                throw new ExecutionException("waitForGuiderToFinish did not complete", x);
            }
        }
    }

    @Override
    public void clear(int nClears) throws ExecutionException {
        commandSender.sendCommand("clear", nClears);
    }
    
    @Override
    public void clearAndStartIntegration(ImageName imageName, int nClears, Map<String, ? extends Serializable> parsedKeyValueData, Set locations, String annotation, Duration stopGuiding) throws ExecutionException {
        commandSender.sendCommand("setHeaderKeywords", parsedKeyValueData);
        guidingFinished = possiblyStartGuiding(imageName, stopGuiding, estimatedGuidingTimePeriod);    
        commandSender.sendCommand("clearAndStartNamedIntegration",imageName, nClears, annotation, locations);
    } 

    @Override
    public void discardRows(int nRows) throws ExecutionException {
        commandSender.sendCommand("shiftNRows", nRows);
    }

    @Override
    public void initGuiders(String roiSpec) throws ExecutionException {
        estimatedGuidingTimePeriod = Duration.ofMillis(commandSender.sendCommand(Integer.class, "initGuiders", roiSpec));
        initGuidersSucceeded = true;
    }
    
    @Override
    public void clearROI() throws ExecutionException {
        if (initGuidersSucceeded) {
            commandSender.sendCommand("clearROI");
            initGuidersSucceeded = false;
        }
    }
    
    @Override
    public void setHeaderKeywords(Map<String, Serializable> headersMap) throws ExecutionException {
        commandSender.sendCommand("setHeaderKeywords", headersMap);
    }
    
    @Override
    public void play(String playlist, boolean repeat) throws ExecutionException {
        commandSender.sendCommand("play", playlist, repeat);
    }

    @Override
    public void definePlaylist(String playlist, String folder, String... images) throws ExecutionException {
        commandSender.sendCommand("definePlaylist", playlist, folder, images);
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateFocalPlaneStateToRaftsState(when, value);
        });
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Focal plane connected");
        FocalPlaneState state = initialState.getState(FocalPlaneState.class);
        translateFocalPlaneStateToRaftsState(CCSTimeStamp.currentTime(), state);
    }

    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Focal plane disconnected");
        // TOOD: Deal with initial state
//        StateBundle result = commandSender.sendCommand("getState", StateBundle.class);
//        FocalPlaneState state = result.getState(FocalPlaneState.class);
//        translateFocalPlaneStateToRaftsState(state);
    }
    /**
     * Translate focal-plane states to the RaftsState messages the MCM is expecting
     * @param when 
     * @param value
     */
    private void translateFocalPlaneStateToRaftsState(CCSTimeStamp when, Enum value) {
        LOG.log(Level.INFO, "Got focal-plane state {0} ", value);
        Enum<RaftsState> converted = FOCALPLANE_TO_RAFTS_STATE.get(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted));
        }
    }
}
