package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.Serializable;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.ocsbridge.states.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * <ol>
 * <li>If focalPlaneSubsystem is null (the default) it simulates all focalPlane
 * operations within this class</li>
 * <li>If focalPlaneSubsystem is specified, it delegates the operations to the
 * specified subsystem</li>
 * </ol>
 *
 * @author tonyj
 */
public class FocalPlane {

    private final CCS ccs;
    private FocalPlaneInterface focalPlane;
    private final State raftsState;
    private final MCMConfig config;
    
    /**
     * Time to clear the sensors. Used when pre-scheduling a clear from initImage
     */
    static final Duration CLEAR_TIME = Duration.ofMillis(70);
    /**
     * Time to discard one row. Used for computing time estimates
     */
    static final Duration DISCARD_TIME = Duration.ofMillis(1);
    /**
     * Time to readout the science rafts. Used for computing time estimates
     */
    static final Duration READOUT_TIME = Duration.ofMillis(2000);

    FocalPlane(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        raftsState = new State(RaftsState.NEEDS_CLEAR);
        ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), raftsState);
        // Assumes simulation by default
        focalPlane = new FocalPlaneSimulation(ccs, raftsState);
    }

    void clear(int nClears) throws ExecutionException {
        if (!config.isAlwaysClear()) focalPlane.clear(nClears);
    }

    void discardRows(int nRows) throws ExecutionException {
        focalPlane.discardRows(nRows);
    }
    
    void startIntegration(ImageName imageName, Map<String, ? extends Serializable> parsedKeyValueData, Set locations, String annotation, Duration stopGuiding) throws ExecutionException {
        if (config.isAlwaysClear()) {
            focalPlane.clearAndStartIntegration(imageName, 1, parsedKeyValueData, locations, annotation, stopGuiding);
        } else {
            focalPlane.startIntegration(imageName, parsedKeyValueData, locations, annotation, stopGuiding);
        }
    }

    void endIntegration(boolean readout) throws ExecutionException {
        focalPlane.endIntegration(readout);
    }
    
    void initGuiders(String roiSpec) throws ExecutionException {
        focalPlane.initGuiders(roiSpec);
    }
    
    void clearROI() throws ExecutionException {
        focalPlane.clearROI();
    }
    
    void setHeaderKeywords(Map<String, Serializable> headersMap) throws ExecutionException {
        focalPlane.setHeaderKeywords(headersMap);
    }

    void play(String playlist, boolean repeat) throws ExecutionException {
        focalPlane.play(playlist, repeat);
    }

    void definePlaylist(String playlist, String folder, String[] images) throws ExecutionException {
        focalPlane.definePlaylist(playlist, folder, images);
    }

    /**
     * Called when the MCM receives a start command.
     *
     * @param configName
     */
    void start(String configName) throws ExecutionException {
        focalPlane.start(configName);
    }

    /**
     * Called to install a real focal-plane subsystem, replacing the default simulation.
     * @param focalPlaneSubsystem 
     */
    ControlledSubsystem registerMCMSubsystem(MCMSubsystem mcm) {
        final FocalPlaneSubsystemLayer controlledSubsystem = new FocalPlaneSubsystemLayer(mcm, ccs, config);
        focalPlane = controlledSubsystem;
        return controlledSubsystem;
    }
}
