package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.Serializable;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAllocateImageNameCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearAndStartNamedIntegrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCloseShutterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDefinePlaylistCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndIntegrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSOpenShutterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPlayCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetHeaderKeywordsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStandbyCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSWaitForImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.events.ShutterMotionProfileFitResult;
import org.lsst.ccs.subsystem.ocsbridge.states.CalibrationState;
import org.lsst.ccs.subsystem.ocsbridge.states.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.states.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterReadinessState;
import org.lsst.ccs.subsystem.ocsbridge.states.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.states.StandbyState;
import org.lsst.ccs.subsystem.ocsbridge.states.TakeImageReadinessState;

/**
 * Run the simulated MCM as a CCS subsystem, able to send/receive commands using
 * CCS buses.
 *
 * @author tonyj
 */
public class MCMSubsystem extends Subsystem implements HasLifecycle, HasDataProviderInfos {

    private static final Logger LOG = Logger.getLogger(MCMSubsystem.class.getName());

    private final CCS ccs = new CCS();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private MCMConfig config;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private MCMLockHandler lockHandler;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dataProviderDictionaryService;
    private MCM mcm;
    private Set<ControlledSubsystem> controlledSubsystems;

    @SuppressWarnings("OverridableMethodCallInConstructor")
    public MCMSubsystem() {
        super("mcm", AgentType.MCM);
    }

    @Override
    public void init() {
        agentStateService.registerState(ShutterState.class, "Shutter state", this);
        agentStateService.registerState(TakeImageReadinessState.class, "Image readiness State", this);
        agentStateService.registerState(FilterState.class, "Filter State", this);
        agentStateService.registerState(CalibrationState.class, "Calibration State", this);
        agentStateService.registerState(ShutterReadinessState.class, "Shutter readiness State", this);
        agentStateService.registerState(RaftsState.class, "Focal plane state", this);
        agentStateService.registerState(StandbyState.class, "Standby state", this);
        // 
        if (config.hasShutterSubsystem() && config.getCameraType() == Camera.MAIN_CAMERA) {
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.MOTOR_ENCODER_PATH+"MINUSX/open"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.MOTOR_ENCODER_PATH+"MINUSX/close"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.MOTOR_ENCODER_PATH+"PLUSX/open"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.MOTOR_ENCODER_PATH+"PLUSX/close"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            //
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.HALL_SENSOR_PATH+"MINUSX/open"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.HALL_SENSOR_PATH+"MINUSX/close"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.HALL_SENSOR_PATH+"PLUSX/open"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
            dataProviderDictionaryService.registerClass(ShutterMotionProfileFitResult.class, MainCameraShutterSubsystemLayer.HALL_SENSOR_PATH+"PLUSX/close"+MainCameraShutterSubsystemLayer.MOTION_PROFILE_FIT_RESULT);
        }
    }
    
    @Override
    public void postInit() {
        mcm = new MCM(ccs, config, imageNameService);
        mcm.registerMCMSubsystem(this);
    }
    
    @Override
    public void postStart() {

        controlledSubsystems = new HashSet<>();

        if (config.hasFocalPlaneSubsystem()) {
            controlledSubsystems.add(mcm.getFocalPlane().registerMCMSubsystem(this));
        } 
        
        if (config.hasShutterSubsystem()) {
            controlledSubsystems.add(mcm.getShutter().registerMCMSubsystem(this));
        }
        
        if (config.hasFilterChangerSubsystem()) {
            controlledSubsystems.add(mcm.getFilterChanger().registerMCMSubsystem(this));
        }      

        for (String otherAgent : config.getOtherConfiguredSubsystems()) {
            controlledSubsystems.add(new ControlledSubsystem(this, otherAgent, ccs, config));
        }

        ccs.addStateChangeListener((when, state, oldState, cause) -> {
            // publish whenever state change occurs
            agentStateService.updateAgentState(when, state);
        });

        // publish initial states
        List<Enum> initialStates = new ArrayList<>();
        ccs.getAggregateStatus().getStates().forEach((state) -> {
            initialStates.add(state.getState());
        });
        agentStateService.updateAgentState(initialStates.toArray(new Enum[initialStates.size()]));

        ccs.addStatusMessageListener((StatusMessage msg) -> {
            // This should be called when a simulated subsystem generates trending/config
            // data.
            this.getMessagingAccess().sendStatusMessage(msg);
        });
        
        ccs.addEventListener((CCSEvent event) -> {
            this.publishSubsystemDataOnStatusBus(new KeyValueData("CCSEvent",event));
        });
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void initImage(@Argument(description="Time in seconds") double delta) throws Exception {
        CCSInitImageCommand initImage = new CCSInitImageCommand(delta);
        executeAndHandleResponse(initImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void startImage(boolean shutter, String sensors, String keyValueData, String annotation, double timeout) throws Exception {
        CCSStartImageCommand startImage = new CCSStartImageCommand(shutter, sensors, keyValueData, annotation, timeout);
        executeAndHandleResponse(startImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void takeImages(double expTime, int numImages, boolean shutter, String sensors, String keyValueData, String annotation) throws Exception {
        CCSTakeImagesCommand takeImages = new CCSTakeImagesCommand(expTime, numImages, shutter, sensors, keyValueData, annotation);
        executeAndHandleResponse(takeImages);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setFilter(String name) throws Exception {
        CCSSetFilterCommand setFilter = new CCSSetFilterCommand(name);
        executeAndHandleResponse(setFilter);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void initGuiders(String roiSpec) throws Exception {
        CCSInitGuidersCommand initGuiders = new CCSInitGuidersCommand(roiSpec);
        executeAndHandleResponse(initGuiders);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void endImage() throws Exception {
        CCSEndImageCommand endImage = new CCSEndImageCommand();
        executeAndHandleResponse(endImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clear(int nClears) throws Exception {
        CCSClearCommand clear = new CCSClearCommand(nClears);
        executeAndHandleResponse(clear);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void play(String playlist, @Argument(defaultValue = "false") boolean repeat) throws Exception {
        CCSPlayCommand play = new CCSPlayCommand(playlist, repeat);
        executeAndHandleResponse(play);
    }    

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void definePlaylist(String playlist, String daqFolder, String... images) throws Exception {
        CCSDefinePlaylistCommand definePlaylist = new CCSDefinePlaylistCommand(playlist, daqFolder, images);
        executeAndHandleResponse(definePlaylist);
    }  
    
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void discardRows(int nRows) throws Exception {
        CCSDiscardRowsCommand discardRows = new CCSDiscardRowsCommand(nRows);
        executeAndHandleResponse(discardRows);
    }
    
    // Commands added to make EO testing easier, not currently used by OCS
    
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setHeaderKeywords(Map<String, Serializable> headersMap) throws Exception {
       CCSSetHeaderKeywordsCommand setHeaderKeywords = new CCSSetHeaderKeywordsCommand(headersMap);
       executeAndHandleResponse(setHeaderKeywords);
    }
    
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public ImageName allocateImageName() throws Exception {
        CCSAllocateImageNameCommand allocateImage = new CCSAllocateImageNameCommand();
        return executeAndHandleResponse(allocateImage);
    }
    
    /**
     * Starts an integration, and returns immediately.The exposure is to be finished
     * by explicitly calling endIntegration.
     * @param imageName The image name to use
     * @param shutter Whether to open/close the shutter
     * @param clears The number of clears to perform
     * @param annotation The image annotation
     * @param sensors The set of sensors to readout
     * @param headersMap The set of headers to pass to focal-plane
     * @throws java.lang.Exception
     */
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clearAndStartNamedIntegration(ImageName imageName, boolean shutter, int clears, String annotation, Set sensors, Map<String, Serializable> headersMap) throws Exception {
        CCSClearAndStartNamedIntegrationCommand clearAndStart = new CCSClearAndStartNamedIntegrationCommand(imageName, shutter, clears, annotation, sensors, headersMap);
        executeAndHandleResponse(clearAndStart);     
    }

    /**
     * Ends a previous clearAndStartNamedIntegration. This method may return before
     * the integration is complete. Use waitForImageCompletion to wait for completion.
     * @throws java.lang.Exception
     */
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void endIntegration() throws Exception {
        CCSEndIntegrationCommand endIntegration = new CCSEndIntegrationCommand();
        executeAndHandleResponse(endIntegration);           
    }
    
    /**
     * Unlike takeImages, this method will return immediately.Use waitForImageCompletion to block
     * until image is complete.
     * @param imageName The image name to use (previously allocated?)
     * @param shutter Whether or not to open the shutter
     * @param expTime The exposure time
     * @param clears The number of clears
     * @param annotation The image annotation
     * @param sensors The set of sensors to readout
     * @param headersMap
     * @throws java.lang.Exception
     */
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void takeImage(ImageName imageName, boolean shutter, double expTime, int clears, String annotation, Set sensors, Map<String, Serializable> headersMap) throws Exception {
        CCSTakeImageCommand takeImage = new CCSTakeImageCommand(imageName, shutter, expTime, clears, annotation, sensors, headersMap);
        executeAndHandleResponse(takeImage);         
    }
    
    /**
     * Block until the image is complete
     * @throws java.lang.Exception
     */
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void waitForImage() throws Exception {
        CCSWaitForImageCommand wait = new CCSWaitForImageCommand();
        executeAndHandleResponse(wait);              
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void openShutter() throws Exception {
        CCSOpenShutterCommand openShutter = new CCSOpenShutterCommand();
        executeAndHandleResponse(openShutter);              
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void closeShutter() throws Exception {
        CCSCloseShutterCommand closeShutter = new CCSCloseShutterCommand();
        executeAndHandleResponse(closeShutter);              
    }
        
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void start(String configuration) throws Exception {
        if (lockHandler != null) {
            // This throws a RuntimeException if it fails (maybe it should be a checked exception?)
            lockHandler.lockAll();
        }
        CCSStartCommand start = new CCSStartCommand(configuration);
        executeAndHandleResponse(start);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void standby() throws Exception {
        if (lockHandler != null) {
            // This throws a RuntimeException if it fails (maybe it should be a checked exception?)
            lockHandler.unlockAll();
        }
        CCSStandbyCommand standby = new CCSStandbyCommand();
        executeAndHandleResponse(standby);
    }
    
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void enableCalibration() throws Exception {
        CCSEnableCalibrationCommand enableCalibration = new CCSEnableCalibrationCommand();
        executeAndHandleResponse(enableCalibration);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void disableCalibration() throws Exception {
        CCSDisableCalibrationCommand disableCalibration = new CCSDisableCalibrationCommand();
        executeAndHandleResponse(disableCalibration);
    }

    private <T extends Object> T executeAndHandleResponse(CCSCommand command) throws Exception {
        CCSCommandResponse<T> response = mcm.execute(command);
        CCSAckOrNack can = response.waitForAckOrNack();
        if (can.isNack()) {
            sendNack(can.getReason());
            return null;
        } else {
            // Note: We add 1000 mS to the estimate to keep ShellCommandConsole from timing
            // out, especially on zero length commands.
            sendAck(can.getDuration().plus(Duration.ofMillis(1000)));
            return response.waitForCompletion();
        }
    }

    //Callback method to publish current data when services or consoles connect
    @Override
    public void publishDataProviderCurrentData(AgentInfo... agents) {
        LOG.log(Level.INFO, "Spontaneous publication of current FCState for connecting agents: ",Arrays.asList(agents));
        for (ControlledSubsystem cs : controlledSubsystems) {
            cs.onRepublishServiceData();
        }
        mcm.publishDataProviderCurrentData();
    }

    Set<ControlledSubsystem> getControlledSubsystems() {
        return controlledSubsystems;
    }
}
