package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import org.lsst.ccs.imagenaming.ImageName;
import static org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.MOVE_TIME;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Shutter simulation for use with MCM
 * @author tonyj
 */
public class ShutterSimulation implements ShutterInterface {

    private final CCS ccs;
    private final State shutterState;
    private final MCMConfig config;
    private ScheduledFuture<?> notReadyFuture;
    private boolean imageSequence = false;

    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME = Duration.ofMillis(4000);

    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME_FOR_SEVERAL_IMAGES = Duration.ofMillis(10000);

    ShutterSimulation(CCS ccs, State shutterState, MCMConfig config) {
        this.ccs = ccs;
        this.shutterState = shutterState;
        this.config = config;
        // When the shutter is closed, we only keep the motors powered up for a limited time
        if (config.getShutterPrepTime() > 0) {
            shutterState.addStateChangeListener((when, state, oldState, cause) -> {
                if (state == ShutterState.CLOSED) {
                    scheduleNotReady();
                } else {
                    cancelNotReady();
                }
            });
        }

    }

    @Override
    public void start(String configName) throws ExecutionException {
        // NOOP
    }

    @Override
    public void expose(ImageName imageName, Duration exposureTime) throws ExecutionException {
        shutterState.checkState(ShutterState.CLOSED);
        shutterState.setState(Shutter.ShutterState.OPENING);
        Duration time = MOVE_TIME;
        // TODO: This does not correctly handle the case when both blades move at once
        ccs.schedule(time, () -> {
            shutterState.setState(Shutter.ShutterState.OPEN);
        });
        time = time.plus(exposureTime);
        ccs.schedule(time, () -> {
            shutterState.setState(Shutter.ShutterState.CLOSING);
        });
        time = time.plus(MOVE_TIME);
        ccs.schedule(time, () -> {
            shutterState.setState(Shutter.ShutterState.CLOSED);
        });
    }

    @Override
    public void open() throws ExecutionException {
        shutterState.setState(Shutter.ShutterState.OPENING);
        ccs.schedule(MOVE_TIME, () -> {
            shutterState.setState(Shutter.ShutterState.OPEN);
        });
    }

    @Override
    public void close() throws ExecutionException {
        shutterState.setState(Shutter.ShutterState.CLOSING);
        ccs.schedule(MOVE_TIME, () -> {
            shutterState.setState(Shutter.ShutterState.CLOSED);
        });
    }

    @Override
    public void prepare() {
        if (config.getShutterPrepTime() > 0) {
            cancelNotReady();
            ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), new State(Shutter.ShutterReadinessState.GETTING_READY));
            ccs.schedule(Duration.ofMillis((long) (config.getShutterPrepTime()*1000)), () -> {
                ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), new State(Shutter.ShutterReadinessState.READY));
                scheduleNotReady();
            });
        }
    }
    private void scheduleNotReady() {

        Duration ready_time;
        if (imageSequence == true) {
            ready_time = READY_TIME_FOR_SEVERAL_IMAGES;
        } else {
            ready_time = READY_TIME;

        }

        notReadyFuture = ccs.schedule(ready_time, () -> {
            ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), new State(Shutter.ShutterReadinessState.NOT_READY));
        });
    }

    private void cancelNotReady() {
        if (notReadyFuture != null) {
            notReadyFuture.cancel(false);
        }
    }

    @Override
    public void setImageSequence(boolean imageSequence) {
        this.imageSequence = imageSequence;
    }

}
