package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.ccs.camera.Camera;

/**
 * Interface to shutter used by MCM. 
 *
 * @author tonyj
 */
public class Shutter {

    /**
     * Time needed to move a shutter blade
     */
    static final Duration MOVE_TIME = Duration.ofMillis(980);
    private final MCMConfig config;

    public enum ShutterReadinessState {

        NOT_READY, READY, GETTING_READY
    };

    public enum ShutterState {

        CLOSED, OPENING, OPEN, CLOSING
    };

    private final State shutterReadinessState;
    private final State shutterState;
    private ShutterInterface shutter;

    private final CCS ccs;

    Shutter(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        // If no PREP time, then shutter is always ready
        shutterReadinessState = new State(config.getShutterPrepTime() > 0 ? ShutterReadinessState.NOT_READY : ShutterReadinessState.READY);
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        ccs.getAggregateStatus().add(now, shutterReadinessState);
        shutterState = new State(ShutterState.CLOSED);
        ccs.getAggregateStatus().add(now, shutterState);
        // By default assumes simulation
        shutter = new ShutterSimulation(ccs, shutterState, config);
    }

    public void startImageSequence() {
        shutter.setImageSequence(true);
    }

    public void endImageSequence() {
        shutter.setImageSequence(false);
    }


    void prepare() {
        shutter.prepare();
    }

    void expose(ImageName imageName, Duration exposureTime) throws ExecutionException {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutter.expose(imageName, exposureTime);
    }

    void start(String configName) throws ExecutionException {
        shutter.start(configName);
    }

    void open() throws ExecutionException {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutter.open();
    }

    void close() throws ExecutionException {
        if (!shutterState.isInState(ShutterState.CLOSED)) {
            shutter.close();
        }
    }

    void ensureOpen() throws ExecutionException, InterruptedException, TimeoutException {
        if (shutterState.getState() != ShutterState.OPEN) {
            shutter.prepare();
            shutter.open(); 
            CompletableFuture<Void> futureOpen = ccs.waitForStatus(ShutterState.OPEN);
            futureOpen.get(2000, TimeUnit.MILLISECONDS);
        }
    }

    void ensureClosed() throws ExecutionException, InterruptedException, TimeoutException {
        if (shutterState.getState() != ShutterState.CLOSED) {
            shutter.close(); 
            CompletableFuture<Void> futureClose = ccs.waitForStatus(ShutterState.CLOSED);
            futureClose.get(2000, TimeUnit.MILLISECONDS);
        }        
    }
    
    
    ControlledSubsystem registerMCMSubsystem(MCMSubsystem mcm) {
        shutter = config.getCameraType() == Camera.MAIN_CAMERA ? 
                new MainCameraShutterSubsystemLayer(mcm, ccs, config) : new BonnShutterSubsystemLayer(mcm, ccs, config);
        return (ControlledSubsystem) shutter;
    }
}
