package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to a real comcam filter changer subsystem.
 *
 * @author tonyj
 */
public class MainCameraFilterChangerSubsystemLayer extends ControlledSubsystem implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(ComCamFilterChangerSubsystemLayer.class.getName());
    private final static Map<FcsEnumerations.McmState, FilterChanger.FilterState> FCS_TO_FILTER_STATE = new HashMap<>();
    private volatile CCSSetFilterEvent ccsSetFilterEvent;

    static {
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.LOADING, FilterChanger.FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.UNLOADING, FilterChanger.FilterState.UNLOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.LOADING, FilterChanger.FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.NO_FILTER, FilterChanger.FilterState.NOFILTER);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.ROTATING, FilterChanger.FilterState.ROTATING);
    }

    private static class FilterStateConverter {

        private FilterChanger.FilterState convertStateInstantaneous(FcsEnumerations.McmState state) {
            FilterChanger.FilterState s = FCS_TO_FILTER_STATE.get(state);
            if (s == null) {
                s = FilterChanger.FilterState.LOADED;
            }
            return s;
        }

        FilterChanger.FilterState convertState(FcsEnumerations.McmState state) {
            FilterChanger.FilterState s = convertStateInstantaneous(state);
            return s;
        }
    }

    private final static FilterStateConverter converter = new FilterStateConverter();

    private FilterChanger.FilterState convertState(FcsEnumerations.McmState state) {
        return converter.convertState(state);
    }

    public MainCameraFilterChangerSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFilterChangerSubsystemName(), ccs, config);
    }

    // those commands were defined mostly in accordance with the ComCam filter
    // changer system
    // adapting here to final FCS logics
    @Override
    public void setFilter(String filter) throws ExecutionException {
        if ("NONE".equals(filter)) {
            commandSender.sendCommand(Void.TYPE, getEstimatedDurationForFilterChange(filter), "setNoFilter");
            waitUntilMoveComplete(FilterChanger.FilterState.NOFILTER, getEstimatedDurationForFilterChange(filter));
        } else {
            commandSender.sendCommand(Void.TYPE, getEstimatedDurationForFilterChange(filter), "setFilterByName", filter);
            waitUntilMoveComplete(FilterChanger.FilterState.LOADED, getEstimatedDurationForFilterChange(filter));
        }
    }

    @Override
    public Map<String, String> getAvailableFilters() throws ExecutionException {
        // map ? Why ? string keys and names, this is very specific of comcam implementation
        // let's retune name, name for FCS
        @SuppressWarnings("unchecked")
        List<String> l = commandSender.sendCommand(List.class, "listAllFilters");
        Map<String, String> m = new HashMap<>();
        l.forEach(s -> m.put(s, s));
        return m;
    }

    @Override
    public List<String> getInstalledFilters() throws ExecutionException {
        @SuppressWarnings("unchecked")
        Map<Integer, String> m = commandSender.sendCommand(Map.class, "listFiltersOnChanger");
        return new ArrayList<>(m.values());
    }

    @Override
    public String getCurrentFilter() throws ExecutionException {
        // Temporarily change to using printFilterONLINEName. Note this returns "NONE" rather than null.
        return commandSender.sendCommand(String.class, "printFilterONLINEName");
    }

    @Override
    public Duration getEstimatedDurationForFilterChange(String filterName) {
        // TODO: Would be good if coult ask the filter changer how long it expects
        // the move to take. Currently we can only ask the estimated time to a
        // known position.
        return Duration.ofSeconds(240);
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Filter changer connected");
        FcsEnumerations.FilterState state = initialState.getState(FcsEnumerations.FilterState.class);
        FcsEnumerations.McmState mcmState = initialState.getState(FcsEnumerations.McmState.class);
        LOG.log(Level.INFO, "Got initial McmState {0}", mcmState);
        FcsEnumerations.AutochangerInclination inclinationState = initialState.getState(FcsEnumerations.AutochangerInclination.class);
        LOG.log(Level.INFO, "Got initial AutochangerInclination {0}", inclinationState);
        translateFcsStateToFilterState(CCSTimeStamp.currentTime(), mcmState);
    }

    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Filter changer disconnected");
        // TOOD: Deal with initial state
        // StateBundle result = commandSender.sendCommand("getState",
        // StateBundle.class);
        // FocalPlaneState state = result.getState(FocalPlaneState.class);
        // translateFocalPlaneStateToRaftsState(state);
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        // CCSTimeStamp ts = statusChange.getCCSTimeStamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle changedStates = newStates.diffState(oldStates);
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue())
                .forEachOrdered((value) -> {
                    if (value instanceof FcsEnumerations.FilterState) {
                        //translateFcsStateToFilterState(when, (FcsEnumerations.FilterState) value);
                    } else if (value instanceof FcsEnumerations.McmState) {
                        LOG.log(Level.INFO, "Got McmState {0}", value);
                        translateFcsStateToFilterState(when, (FcsEnumerations.McmState) value);
                    } else if (value instanceof FcsEnumerations.AutochangerInclination) {
                        LOG.log(Level.INFO, "Got AutochangerInclination {0}", value);
                    }
                });
    }

    private void waitUntilMoveComplete(FilterChanger.FilterState finalState, Duration estimatedDurationForFilterChange)
            throws ExecutionException {
        // TOOD: Do we need to wait for it to start moving first?
        // yes of course and we should also wait for a possible unloaded end state if
        // set no filter.
        Future<Void> waitForStatus = ccs.waitForStatus(finalState);
        try {
            waitForStatus.get(estimatedDurationForFilterChange.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException | TimeoutException ex) {
            throw new ExecutionException("Timeout waiting for filter change to complete", ex);
        }
    }

    private void translateFcsStateToFilterState(CCSTimeStamp when, FcsEnumerations.McmState value) {
        LOG.log(Level.INFO, "Got filter changer state {0} ", value);
        if (value == null) {
            value = FcsEnumerations.McmState.LOADED;
        }
        FilterChanger.FilterState converted = convertState(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted));
        }
    }

    @Override
    protected void onEvent(StatusMessage msg) {
        Object data = msg.getObject();
        if (data instanceof KeyValueDataList) {
            KeyValueDataList kvdl = (KeyValueDataList) data;
            //Object dataObject = kvdl.getListOfKeyValueData().get(0).getValue();
            String dataKey = ((KeyValueDataList) data).getKey();
            LOG.log(Level.INFO, "Got kvdl {0} {1}", new Object[]{dataKey, kvdl.getListOfKeyValueData()});
        } else if (data instanceof KeyValueData) {
            final KeyValueData sentData = (KeyValueData) data;
            String dataKey = sentData.getKey();
            LOG.log(Level.INFO, "Got kvd {0} {1}", new Object[]{dataKey, sentData.getValue()});
            if ("fcs/mcm".equals(dataKey) && sentData.getValue() instanceof KeyValueDataList) {
                //[2023-10-30T11:59:46.473PDT] INFO: Got kvd fcs/mcm filter_on_autochanger_id=0, filter_on_autochanger_name=NO FILTER, autochanger_trucks_position=3606, autochanger_trucks_location=ONLINE (org.lsst.ccs.subsystem.ocsbridge.sim.MainCameraFilterChangerSubsystemLayer onEvent)
                //[2023-10-26T12:54:37.976PDT] INFO: Got kvd fcs/mcm filter_on_autochanger_id=43, filter_on_autochanger_name=ef, autochanger_trucks_position=3104, autochanger_trucks_location=ONLINE (org.lsst.ccs.subsystem.ocsbridge.sim.MainCameraFilterChangerSubsystemLayer onEvent) 
                KeyValueDataList kvdl = (KeyValueDataList) sentData.getValue();
                int id = 0;
                int slot = 0;  // Currently not provided by FCS
                int position = 0;
                String name = null;
                double proximity = 0;
                for (KeyValueData d : kvdl) {
                    Object key = d.getKey();
                    Object value = d.getValue();
                    if ("filter_on_autochanger_id".equals(key)) {
                        id = ((Number) value).intValue();
                    } else if ("filter_on_autochanger_name".equals(key)) {
                        name = value.toString();
                    } else if ("autochanger_trucks_position".equals(key)) {
                        position = ((Number) value).intValue();
                    } else if ("proximity".equals(key)) {
                        proximity = ((Number) value).doubleValue();
                    } else if ("filter_previous_socketID".equals(key)) {
                        slot = ((Number) value).intValue();
                    }
                }
                //Example of what is sent by ComCamFCS
                //[2023-08-02T03:43:17.049+0000] INFO: Sent: CCSSetFilterEvent{filterName=r_03, start=true, position=-1.0, slot=-1, filterType=r} (org.lsst.ccs.subsystem.ocsbridge.sim.FilterChangerComCamSubsystemLayer onEvent)
                String fullName = String.format("%s_%d", name, id);
                String type = name == null || name.length() == 1 ? name : String.format("other:%s", name);
                if ("NO FILTER".equals(name)) {
                    fullName = "NONE";
                    type = "other:NONE";
                }
                ccsSetFilterEvent = new CCSSetFilterEvent(fullName, type, slot, proximity == 0 ? position : proximity);
                ccs.fireEvent(ccsSetFilterEvent);
                LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
            }
        }
    }

    @Override
    protected void onRepublishServiceData() {
        if (ccsSetFilterEvent != null) {
            ccs.fireEvent(ccsSetFilterEvent);
            LOG.log(Level.INFO, "Resent: {0}", ccsSetFilterEvent);
        }
    }

}
