package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.camera.Camera;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;

/**
 * The filter changer used by the MCM
 *
 * @author tonyj
 */
public class FilterChanger {

    private static final Logger LOG = Logger.getLogger(FilterChanger.class.getName());

    public enum FilterState {

        ROTATING, UNLOADED, LOADED, UNLOADING, LOADING, NOFILTER
    }

    private List<String> installedFilters;
    private final CCS ccs;
    private FilterChangerInterface filterChanger;
    private final MCMConfig config;

    FilterChanger(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        this.filterChanger = new FilterChangerSimulation(ccs, config);
    }

    public void start(String configName) throws ExecutionException {
        filterChanger.start(configName);
        // filterChanger is locked at this point, so installed filters cannot change
        installedFilters = filterChanger.getInstalledFilters();
        LOG.log(Level.INFO, "Installed filters {0}", installedFilters);
        ccs.fireEvent(new CCSAvailableFiltersEvent(installedFilters));
    }

    boolean filterIsAvailable(String filter) {
        return installedFilters.contains(filter);
    }

    List<String> getAvailableFilters() {
        return installedFilters;
    }

    void setFilter(String filter) throws ExecutionException {
        filterChanger.setFilter(filter);
    }

    Duration getEstimatedTimeForChange(String filterName) {
        return filterChanger.getEstimatedDurationForFilterChange(filterName);
    }

    ControlledSubsystem registerMCMSubsystem(MCMSubsystem mcm) {
        final FilterChangerInterface controlledSubsystem = config.getCameraType() == Camera.MAIN_CAMERA ? 
                new MainCameraFilterChangerSubsystemLayer(mcm, ccs, config) : new ComCamFilterChangerSubsystemLayer(mcm, ccs, config);
        filterChanger = controlledSubsystem;
        return (ControlledSubsystem) controlledSubsystem;
    }

}
