package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.Serializable;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.OperationalState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLockService.AgentLockUpdateListener;
import org.lsst.ccs.services.UnauthorizedLevelException;
import org.lsst.ccs.services.UnauthorizedLockException;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;

/**
 * A base class for subsystems which are actively controlled by the MCM
 *
 * @author tonyj
 */
class ControlledSubsystem {

    private static final Logger LOG = Logger.getLogger(ControlledSubsystem.class.getName());

    protected final MCMConfig config;
    protected final MCMCommandSender commandSender;
    protected final CCS ccs;
    private final String targetSubsystem;
    private final AgentLockService agentLockService;
    private final AlertService alertService;
    private final Alert goneAlert;
    private final Alert faultAlert;
    private final Alert lockAlert;
    private final Subsystem mcm;
    private volatile boolean isConnected = false;
    
    volatile private AgentLock targetLock; // null if the controlled subsystem is not currently locked my MCM
    
    public ControlledSubsystem(Subsystem mcm, String targetSubsystem, CCS ccs, MCMConfig config) {
        this.config = config;
        this.mcm = mcm;
        this.agentLockService = mcm.getAgentService(AgentLockService.class);
        this.alertService = mcm.getAgentService(AlertService.class);
        String goneAlertId = targetSubsystem+"_gone";
        this.goneAlert = new Alert(goneAlertId, "Alert raised when controlled subsystem disappears.");
        alertService.registerAlert(goneAlert); 

        String faultAlertId = targetSubsystem+"_fault";
        this.faultAlert = new Alert(faultAlertId, "Alert raised when controlled subsystem goes into fault state.");
        alertService.registerAlert(faultAlert);
        
        String lockAlertId = targetSubsystem+"_lock";
        this.lockAlert = new Alert(lockAlertId, "Alert raised when a lock on a controlled subsystem is lost.");
        alertService.registerAlert(lockAlert); 
        
        this.commandSender = new MCMCommandSender(targetSubsystem, mcm.getMessagingAccess());
        this.ccs = ccs;
        this.targetSubsystem = targetSubsystem;

        mcm.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                // This is now dealt with in the status message listener
            }

            @Override
            public void disconnected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (agent.getName().equals(targetSubsystem)) {
                        if (agentLockService.getExistingLockForAgent(targetSubsystem) != null) {
                            alertService.raiseAlert(goneAlert, AlertState.ALARM, String.format("Controlled subsystem %s has disconnected", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);                            
                        }
                        onDisconnect(agent);
                        isConnected = false;
                    }
                }
            }
        });

        Predicate<BusMessage<? extends Serializable, ?>> targetSubsystemFilter = BusMessageFilterFactory.messageOrigin(targetSubsystem);
        mcm.getMessagingAccess().addStatusMessageListener((msg) -> {
            if (!isConnected) {
                StateBundle initialState = msg.getState();
                AgentInfo agent = msg.getOriginAgentInfo();
                alertService.raiseAlert(goneAlert, AlertState.NOMINAL, String.format("Controlled subsystem %s has connected", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
                onConnect(agent, initialState);
                isConnected = true;
            }
            if (msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification scn = (StatusStateChangeNotification) msg;
                boolean isInFault = scn.getNewState().isInState(OperationalState.ENGINEERING_FAULT);
                if (isInFault && targetLock != null) {
                    String cause = scn.getCause();
                    StringBuilder sb = new StringBuilder();
                    sb.append("Controlled subsystem ").append(targetSubsystem).append(" has gone into FAULT.");
                    if (cause != null) {
                        sb.append(" Cause: ").append(cause);
                    }
                    alertService.raiseAlert(faultAlert, AlertState.ALARM, sb.toString());
                    // Unlock the faulty subsystem to make it easier for people to fix it
                    try {
                        unlock();
                    } catch (ExecutionException x) {
                        LOG.log(Level.WARNING, "Error while unlocking "+targetSubsystem+" after fault", x.getCause());
                    }
                } else if (!isInFault) {
                    alertService.raiseAlert(faultAlert, AlertState.NOMINAL, String.format("Controlled subsystem %s is no longer in FAULT", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
                }
                onStateChange(scn);
            } else if (msg instanceof StatusHeartBeat) {
                // Ignored
            } else {
                onEvent(msg);
            }
        }, targetSubsystemFilter);
        
        mcm.getAgentService(AgentLockService.class).addAgentLockUpdateListener(new AgentLockUpdateListener() {
            @Override
            public void onAgentHeldLockUpdate(String agentName, AgentLock lock) {
                if (agentName.equals(targetSubsystem) && targetLock != null) {
                    if (lock == null) { // unlocked or detached
                        alertService.raiseAlert(lockAlert, AlertState.ALARM, String.format("Lost lock on controlled subsystem %s", targetSubsystem));
                    } else { // attached; caveat: with current lock service implementation, this may also happen if userid changes, but MCM is not supposed to do that
                        alertService.raiseAlert(lockAlert, AlertState.NOMINAL, String.format("Controlled subsystem %s is now locked", targetSubsystem), RaiseAlertStrategy.ON_SEVERITY_CHANGE);
                    }
                    LOG.log(Level.INFO, "Held Lock Update: {0}", asString(lock)); // FIXME: temporary diagnostics
                }
            }

            @Override
            public void onGlobalLockUpdate(String agentName, String owner, AgentLock lock) { // kludge to work around LSSTCCS-2713
                if (agentName.equals(targetSubsystem)) {
                    LOG.log(Level.INFO, "Global Lock Update: {0}", asString(lock)); // FIXME: temporary diagnostics
                    try {
                        AgentLockInfo theLock = (AgentLockInfo) lock;
                        if (theLock.getStatus() == AgentLockInfo.Status.RELEASED) {
                            onAgentHeldLockUpdate(agentName, null);
                        }
                    } catch (ClassCastException x) {
                    }
                }
            }

            @Override
            public void onAgentLevelChange(String agentName, int level) {
                // not watching this now since any level works; revisit if non-zero level becomes required for NORMAL mode operation
            }
        });
    }

    public AgentLock lockAndSwitchToNormal(int level) throws ExecutionException {
        try {
            LOG.log(Level.INFO, "Locking agent {0}", targetSubsystem);
            agentLockService.setLevelForAgent(targetSubsystem, level);
            targetLock = agentLockService.getLockForAgent(targetSubsystem); // FIXME: it would be better if setLevelForAgent(...) returned a lock (atomic)
            if (targetLock == null) {
                throw new UnauthorizedLockException("setLevelForAgent(...) succeeded yet there is no held lock");
            }
            try {
                commandSender.sendCommand("switchToNormalMode");
            } catch (ExecutionException x) {
                targetLock = null;
                agentLockService.unlockAgent(targetSubsystem); // No point leaving it locked in this case
                throw x;
            }
            return targetLock;
        } catch (RuntimeException | UnauthorizedLevelException | UnauthorizedLockException x) {
            throw new ExecutionException("Failed to lock subsystem " + targetSubsystem, x);
        }
    }
    
    public void unlock() throws ExecutionException {
        targetLock = null;
        alertService.raiseAlert(lockAlert, AlertState.NOMINAL, 
                String.format("Controlled subsystem %s is no longer expected to be locked", targetSubsystem),
                RaiseAlertStrategy.ON_SEVERITY_CHANGE);
        try {
            agentLockService.unlockAgent(targetSubsystem);
        } catch (UnauthorizedLockException | RuntimeException x) {
            if (agentLockService.getLockForAgent(targetSubsystem) != null) { // FIXME: it would be better to have unlockAgent() succeed if the target is already unlocked
                throw new ExecutionException("Failed to unlock subsystem " + targetSubsystem, x);
            }
        }
    }
    
    public void start(String configName) throws ExecutionException {
        commandSender.sendCommand("publishConfigurationInfo");
    }

    /**
     * This method is called whenever the target subsystems connects, including
     * initially when the the MCM itself starts up if the target subsystem is
     * already running.
     *
     * @param agent
     */
    protected void onConnect(AgentInfo agent, StateBundle initialState) {

    }

    protected void onDisconnect(AgentInfo agent) {

    }

    protected void onStateChange(StatusStateChangeNotification statusChange) {

    }

    protected void onEvent(StatusMessage msg) {
    }

    String getAgentName() {
        return targetSubsystem;
    }
    
    AgentLock getLock() {
        return targetLock;
    }
    
    void sendEvent(String key, Serializable object) {
        KeyValueData kvd = new KeyValueData(key, object);
        mcm.publishSubsystemDataOnStatusBus(kvd);
        LOG.log(Level.INFO, "Sent: {0}={1}",new Object[]{key, object});
    }
    
    /**
     * Temporary local method, remove once depending on toolkit 6.1.6
     */
    static private String asString(AgentLock lock) {
        if (lock == null) return "null";
        StringBuilder sb = new StringBuilder();
        sb.append("[Agent: ").append(lock.getAgentName()).append(", owner: ").append(lock.getOwner());
        try {
            AgentLockInfo theLock = (AgentLockInfo) lock;
            sb.append(", ").append(theLock.getStatus());
            sb.append(", current: ").append(theLock.getCurrentAgent()).append(", origin: ").append(theLock.getOriginatingAgent());
        } catch (ClassCastException x) {
            sb.append(", class: ").append(lock.getClass().getName());
        }
        return sb.append("]").toString();
    }
     
}
