package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.lsst.ccs.subsystem.ocsbridge.xml.MakeXMLConfiguration.DictionaryConfiguration;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription.LocationVariable;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription.PatternMatchedSALVariable;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * A version of XMLMaker which uses SALClasDescriptionMaker instead of
 * ProcessedInfoMaker
 *
 * @author farrukh
 */
public class XMLMaker2 {

    private final boolean verbose;
    private final UnitConverter unitConverter = new UnitConverter();
    private final TypeConverter typeConverter = new TypeConverter();

    private static final Logger LOG = Logger.getLogger(XMLMaker2.class.getName());
    
    public enum SALType {
        TELEMETRY("Telemetry", "_Telemetry.xml"), SETTINGS_APPLIED("Event", "_Events.xml");

        private final String type;
        private final String xmlFileSuffix;

        SALType(String type, String xmlFileSuffix) {
            this.type = type;
            this.xmlFileSuffix = xmlFileSuffix;
        }

        String getSchema() {
            return type;
        }
        
        public String getXMLFileSuffix() {
            return xmlFileSuffix;
        }

        public String getClassNameAppendage() {
            return this == SETTINGS_APPLIED ? "Configuration" : "";
        }

    }

    public XMLMaker2(boolean verbose) {
        this.verbose = verbose;
    }
    
    public Document createXML(MakeXMLConfiguration config) throws ParserConfigurationException, TransformerException  {
        Document document = createDocument(config.getSALType());
        List<MakeXMLConfiguration.DictionaryConfiguration> dictConfigs = config.getOrderedListOfDictionaryConfigurations();
        for (MakeXMLConfiguration.DictionaryConfiguration dictConfig : dictConfigs) {
            SALClassDescriptionMaker descriptionMaker = new SALClassDescriptionMaker(dictConfig, config.getMapping());
            generateXML(document, descriptionMaker);
        }
        return document;
    }

    public Document createDocument(SALType salType) throws ParserConfigurationException {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = dbf.newDocumentBuilder();
        Document dom = db.newDocument();
        Element rootEle = dom.createElement("SAL" + salType.getSchema() + "Set");
        rootEle.setAttributeNS("http://www.w3.org/2001/XMLSchema-instance",
                "xsi:noNamespaceSchemaLocation", "https://raw.githubusercontent.com/lsst-ts/ts_xml/develop/python/lsst/ts/xml/data/schema/SAL" + salType.getSchema() + "Set.xsd");
        dom.appendChild(rootEle);
        return dom;
    }

    public void addXML(Document document, DictionaryConfiguration dictConfig, Mapping mapping) throws IOException, ParserConfigurationException, TransformerException, ClassNotFoundException {
        SALClassDescriptionMaker infoMaker = new SALClassDescriptionMaker(dictConfig, mapping);
        generateXML(document, infoMaker);
    }

    Document makeXMLFromDictionary(DictionaryConfiguration dictConfig, Mapping mapping) throws ParserConfigurationException, TransformerException, ClassNotFoundException, IOException {
        return makeXMLFromDictionary(dictConfig, mapping, dictConfig.getLevel());
        
    }
    Document makeXMLFromDictionary(DictionaryConfiguration dictConfig, Mapping mapping, int level) throws ParserConfigurationException, TransformerException, ClassNotFoundException, IOException {
        SALType type = dictConfig.getSALType();
        Document document = createDocument(type);
        SALClassDescriptionMaker infoMaker = new SALClassDescriptionMaker(dictConfig, mapping, level);
        generateXML(document, infoMaker);
        return document;
    }
    
    
    void generateXML(Document dom, SALClassDescriptionMaker descMaker) throws ParserConfigurationException, TransformerConfigurationException, TransformerException {
        generateXML(dom, descMaker, descMaker.level);
    }
    void generateXML(Document dom, SALClassDescriptionMaker descMaker, int level) throws ParserConfigurationException, TransformerConfigurationException, TransformerException {
        final SALType salType = descMaker.getDictionaryConfiguration().getSALType();
        final Map<String,SALClassDescription> salClassDescriptions = descMaker.getSALClassDescriptions();        
        String subsystemName = descMaker.getDictionaryConfiguration().getCSCName();
        
        Element rootEle = (Element) dom.getChildNodes().item(0);
        // process salType - this is telemetry or event for now - 
        // 
        String classNameAppendage = salType.getClassNameAppendage();

        // finalReprocessedClassInfo was produced in the loop above.
        if (verbose) {
            for (Map.Entry<String, SALClassDescription> key : salClassDescriptions.entrySet()) {
                System.out.println(" Class Name " + key.getKey());
                System.out.println(key.getValue());
            }
        }


        StringBuilder summaryOfMissingDescriptions = new StringBuilder("Topics with missing descriptions:\n");        
        boolean missingDescriptions = false;

        // ******* So if everything is moved to ProcessedInfoMaker or a similar class - then we need it to return finalReprocessedClassInfo which is directly below ... April 14. 2021 ****
        for (Map.Entry<String, SALClassDescription> entry : salClassDescriptions.entrySet()) {

            //System.out.println("<" + "SAL" + salType + ">");
            // really need subsystem name AND key.getkey() ? I think if you are making the classname you can get rid of one 
            //      System.out.println("<EFDB_Topic>" + key.getKey().trim() + "_" + salType + "</EFDB_Topic>");
            // this is being fixed elsewhere and so following line is redundant 
            if (subsystemName.contains("comcam")) {
                subsystemName = "CCCamera";
            }
            
            String topicName = entry.getValue().getTopicName();
            topicName += classNameAppendage;

            String text = "Created by xmlmaker2 for subsystem " + subsystemName + " class " + topicName + " using level " + level;
            final String category = entry.getValue().getCategory();
            if (category != null && !category.isEmpty()) {
                text = text + " for category " + category;
            }
            
            if ( "InfluxDb".equals(category) ) 
                continue;            
            Comment comment = dom.createComment(text);
            rootEle.appendChild(comment);
            Comment comment2 = dom.createComment(String.join("", Collections.nCopies(text.length(), "=")));
            rootEle.appendChild(comment2);

            Element salElement = dom.createElement("SAL" + salType.getSchema());
            rootEle.appendChild(salElement);

            Element subsystemElement = dom.createElement("Subsystem");
            subsystemElement.setTextContent(subsystemName);
            salElement.appendChild(subsystemElement);

            Element topicElement = dom.createElement("EFDB_Topic");
            topicElement.setTextContent(topicName);
            salElement.appendChild(topicElement);
            String checksumText = "CCS: Dictionary_Checksum: "+Long.toString(entry.getValue().computeChecksum())+"L ";
            Comment checksumComment = dom.createComment(checksumText);
            salElement.appendChild(checksumComment);
            //just for Configuration
            if (classNameAppendage.contains("Configuration")) {
                // add version                
                Element version = dom.createElement("item");
                salElement.appendChild(version);
                Element efdbName = dom.createElement("EFDB_Name");
                efdbName.setTextContent("version");
                version.appendChild(efdbName);
                Element desc = dom.createElement("Description");
                desc.setTextContent("Version of the settings, formatted using CCS conventions");
                version.appendChild(desc);
                Element type = dom.createElement("IDL_Type");
                type.setTextContent("string");
                version.appendChild(type);
                Element units = dom.createElement("Units");
                units.setTextContent("unitless");
                version.appendChild(units);
                Element count = dom.createElement("Count");
                count.setTextContent("1");
                version.appendChild(count);
            }

            // location 
            for (SALClassDescription.SALVariable var : entry.getValue().getVariables()) {
                int counts = entry.getValue().getSharedLocationCount(var);

                String type = typeConverter.xmlType(var.getType());

                if (var.getUnits() != null) {
                    // TODO: LCOBM-39 -- Remove this
                    if (var.getUnits().equalsIgnoreCase("DAC Counts")) {
                        continue;
                    }
                }

                String variableName = var.getVariableName();
                if (variableName.contains("_")) {

                    String temp = "";
                    String[] splitVariable = variableName.split("_");
                    splitVariable[0] = splitVariable[0].toLowerCase();

                    for (int i = 0; i <= splitVariable.length - 1; i++) {
                        temp += splitVariable[i] + "_";
                    }

                    if (temp.charAt(temp.length() - 1) == '_') {
                        temp = temp.substring(0, temp.length() - 1);
                    }

                    variableName = temp;
                } else if (variableName.length() > 1) {
                    variableName = variableName.substring(0, 1).toLowerCase() + variableName.substring(1);
                } else {
                    variableName = variableName.toLowerCase();
                }

                // TODO: LCOBM-39 More bogus rafts settings
                if ("Rafts".equals(category)) {
                    if (variableName.equals("id") || variableName.equals("useScienceCCD")) {
                        continue;
                    }
                }
                
                // Workaround for LCOBM-69
                if (variableName.startsWith("rebstate") || variableName.startsWith("tripstate") || variableName.equals("rebState") || variableName.equals("tripState")) {
                    continue;
                }

                // TODO: LCOBM-40 Temporary special handling of dbandHi, dbandLo
                if (variableName.endsWith("dbandHi")) {
                    variableName = variableName.replace("dbandHi", "warnHi");
                } else if (variableName.endsWith("dbandLo")) {
                    variableName = variableName.replace("dbandLo", "warnLo");
                }
                
                // Some variables contain illegal character -, change to _
                variableName = variableName.replace('-', '_');
                
                // Variable name cannot begin with digit
                if (Character.isDigit(variableName.charAt(0))) {
                    variableName = "n_"+variableName;
                }

                if (var instanceof PatternMatchedSALVariable) {
                    PatternMatchedSALVariable pmsv = (PatternMatchedSALVariable) var;
                    LocationVariable locationVariable = pmsv.getLocationVariable();
                    Comment locationComment = dom.createComment("CCSMETA: Array "+variableName+" indexed by "+locationVariable.getVariableName());
                    salElement.appendChild(locationComment);
                }    
                
                String description = null;
                if ( var.getDescription() == null ) {
                    description = "MISSING";
                    summaryOfMissingDescriptions.append(topicName).append("_").append(variableName).append("\n");
                    missingDescriptions = true;
                } else {
                    description = var.getDescription();
                }
                int backslash = description.indexOf('\\');
                if (backslash > 0) {
                    description = description.substring(backslash + 1);
                }
                
                
                Element item = dom.createElement("item");
                salElement.appendChild(item);

                Element efdbName = dom.createElement("EFDB_Name");
                efdbName.setTextContent(variableName);
                item.appendChild(efdbName);

                if (var.getUnits() != null) {
                    if (var.getUnits().toLowerCase().contains("rpm")) {
                        description = description + " (RPM)";
                    }
                }
                
                // SAL does not support string arrays
                boolean isStringArray = counts > 1 && "string".equals(type);
                if (isStringArray) {
                    description += " (colon delimited array)";
                }
                
                Element desc = dom.createElement("Description");
                desc.setTextContent(description);
                item.appendChild(desc);

                Element idlType = dom.createElement("IDL_Type");
                idlType.setTextContent(type);
                item.appendChild(idlType);
                String ccsUnits = var.getUnits();
                // Temporary workaround for incorrect units in timers
                String salUnits = unitConverter.convertCCSUnitToSALUnit(ccsUnits);
                if ("unitless".equals(salUnits) && variableName.endsWith("taskPeriodMillis")) {
                    salUnits = "millisecond";
                }   
                Element unitsElement = dom.createElement("Units");
                unitsElement.setTextContent(salUnits);
                item.appendChild(unitsElement);
                Element countElement = dom.createElement("Count");
                countElement.setTextContent(String.valueOf(isStringArray ? 1 : counts));
                item.appendChild(countElement);
                if (var instanceof PatternMatchedSALVariable && counts==1) {
                    Element isJavaArray = dom.createElement("IsJavaArray");
                    isJavaArray.setTextContent("yes");
                    item.appendChild(isJavaArray);                    
                }
            }

        }
        if ( missingDescriptions ) {
            LOG.log(Level.WARNING, "Missing descriptions\n {0}",summaryOfMissingDescriptions.toString());
        }
    }

    private String cleanCapitalize(String inString) {
        String output = inString;
        output = output.toLowerCase();
        output = output.substring(0, 1).toUpperCase() + output.substring(1);
        output = output.trim();
        return output;
    }

    /**
     * Write the XML document to a file
     *
     * @param output The location to write the output
     * @param dom The XML document
     * @throws TransformerConfigurationException
     * @throws TransformerException
     */
    public void writeXML(StreamResult output, Document dom) throws TransformerConfigurationException, TransformerException {
        final TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer tr = transformerFactory.newTransformer();
        tr.setOutputProperty(OutputKeys.INDENT, "yes");
        tr.setOutputProperty(OutputKeys.METHOD, "xml");
        tr.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
        // This works with jdk8, but perhaps not later JDK versions?
        tr.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
        // send DOM to file
        tr.transform(new DOMSource(dom), output);
    }

}
