package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.IOException;
import static java.lang.System.exit;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.regex.Pattern;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.subsystem.ocsbridge.config.Camera;
import org.lsst.ccs.subsystem.ocsbridge.xml.MakeXMLConfiguration.DictionaryConfiguration;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription.SALVariable;
import org.lsst.ccs.subsystem.ocsbridge.xml.XMLMaker2.SALType;

/**
 * Given a data provider dictionary create a map of SALClassDescriptors for a
 * given subsystem.
 *
 * @author farrukh, tonyj
 */
public class SALClassDescriptionMaker {

    private final Map<String, SALClassDescription> SALClassDescriptionMap = new TreeMap<>();
    private final Map<String, Class> SALClassMap = new TreeMap<>();
    private static final boolean SALCLASS_DESCRIPTION_MAKER_DEBUG = false;
    private static final Set<String> CONFIGURATION_ATTRIBUTES = Collections.singleton(DataProviderInfo.Type.CONFIGURATION.name());
    private static final Set<String> TELEMETRY_ATTRIBUTES = new HashSet<>();
    static {
        TELEMETRY_ATTRIBUTES.add(DataProviderInfo.Type.MONITORING.name());
        TELEMETRY_ATTRIBUTES.add(DataProviderInfo.Type.TRENDING.name());
    }
    // TODO: Move somewhere better
    private static final Set<Pattern> trendingExclusions = new HashSet<>();
    static {
        trendingExclusions.add(Pattern.compile("ccsVersions/.*"));
        trendingExclusions.add(Pattern.compile("runtimeInfo/.*"));
        trendingExclusions.add(Pattern.compile("config/.*")); // Configuration 
        trendingExclusions.add(Pattern.compile("ImageMetaData/.*")); // Configuration checksums
        trendingExclusions.add(Pattern.compile(".*/sequencerChecksum")); // sequencer checksums
        trendingExclusions.add(Pattern.compile(".*/sequencerFile")); // sequencer file name
        trendingExclusions.add(Pattern.compile(".*endSetFilter.*")); // comcam fcs
        trendingExclusions.add(Pattern.compile(".*startSetFilter.*")); // comcam fcs
    }

    private static final Logger LOG = Logger.getLogger(SALClassDescriptionMaker.class.getName());

    private final DictionaryConfiguration dictConfig;
    final int level;
    
    public SALClassDescriptionMaker(DictionaryConfiguration dictConfig, Mapping mapping) {
        this(dictConfig, mapping, dictConfig.getLevel());
    }

    //For tests only
    public SALClassDescriptionMaker(DictionaryConfiguration dictConfig, Mapping mapping, int level) {
        
        this.level = level;
        this.dictConfig = dictConfig;

        SALType salType = dictConfig.getSALType();
        String cscName = dictConfig.getCSCName();
        Camera camera = dictConfig.getCamera();
        String subsystemName = dictConfig.getComponentName();
        
        
        DataProviderDictionary dataProviderDictionary = null;
        try {
            dataProviderDictionary = dictConfig.getDictionary();
        } catch (ClassNotFoundException|IOException ex) {
            throw new RuntimeException("Could not fetch the dictionary", ex);
        } 
        
        Set<String> attributeTypes = salType == SALType.SETTINGS_APPLIED ? CONFIGURATION_ATTRIBUTES : TELEMETRY_ATTRIBUTES ;
        List<DataProviderInfo> dpInfoList = dataProviderDictionary.getDataProviderInfos();
        
        outer: for (DataProviderInfo dpi : dpInfoList) {
            final String dataType = dpi.getAttributeValue(DataProviderInfo.Attribute.DATA_TYPE);
            if (attributeTypes.contains(dataType)) {
                String type = dpi.getAttributeValue(DataProviderInfo.Attribute.TYPE);
                String description = dpi.getAttributeValue(DataProviderInfo.Attribute.DESCRIPTION);
                String units = dpi.getAttributeValue(DataProviderInfo.Attribute.UNITS);
                String cat = dpi.getAttributeValue(DataProviderInfo.Attribute.CONFIG_CATEGORY);
                cat = (cat == null) ? "" : cat;
                if ("TRENDING".equals(dataType)) cat = "Trending";
                String category = cat;

                // Still needed? It appears so, although I am not sure why
                // It appears that type is set to null for MONITORING data, not sure why
                // TODO: Keep track of which subsystems have type == null, after updating dictionary .ser files
                if (type == null) {
                    type = "double";
                }

                // I've changed getPath() below to getFullPath() because I noticed some of the values were missing when 
                // access was attempted via getPath()..
                String originalPathName = dpi.getFullPath().replace("main/", "").trim();
                // Restore old behaviour where items defined at the base level had a leading slash
                if (salType == SALType.SETTINGS_APPLIED && !originalPathName.contains("/")) {
                    originalPathName = "/"+originalPathName;
                }
                
                if ("TRENDING".equals(dataType)) {
//                    System.out.println(originalPathName);
                    for (Pattern exclude : trendingExclusions) {
                        if (exclude.matcher(originalPathName).matches()) continue outer; 
                    }
                }

                Mapping.Match match = mapping.match(originalPathName);
                if (match != null) {
                    String rawSALClassParameterName = match.getPatternName() + "/" + match.getPathAfterMatch();
                    String salClassParameterName = makeSALClassParameterName(rawSALClassParameterName.trim(), level);

                    if (SALCLASS_DESCRIPTION_MAKER_DEBUG) {
                        System.out.println(" **** SALClass Description Maker ****");
                        System.out.println(" Original path name " + originalPathName);
                        System.out.println(" Location " + match.getLocation());
                        System.out.println(" Path after Pattern match " + match.getPathAfterMatch());
                        System.out.println(" Pattern Name " + match.getPatternName());
                        System.out.println(" Raw Class Parameter Name " + rawSALClassParameterName);
                        System.out.println(" Level " + level);
                        System.out.println(" SALClassParameter Name " + salClassParameterName);
                        System.out.println(" ");
                    }
                    
                    String topicName = makeSALTopicName(level, cscName, subsystemName, salType, rawSALClassParameterName, category);
                    Class simpleSalClass = makeSALClasseName(topicName, salType, camera);
                    
                    SALClassDescription salClassDescription = SALClassDescriptionMap.computeIfAbsent(topicName, (name) -> new SALClassDescription(topicName, simpleSalClass, level, category));
                    SALClassDescription.BusVariable busVariable = new SALClassDescription.BusVariable(originalPathName);
                    
                    if ( simpleSalClass != null ) {
                        SALClassMap.put(topicName, simpleSalClass);
                    }

                    SALClassDescription.SALVariable existingVariable = salClassDescription.getVariable(salClassParameterName);
                    if (existingVariable instanceof SALClassDescription.PatternMatchedSALVariable) {
                        ((SALClassDescription.PatternMatchedSALVariable) existingVariable).addBusVariable(busVariable);
                    } else {
                        String locationVariableName = match.getPatternName() + "Location";
                        SALClassDescription.LocationVariable locationVariable = (SALClassDescription.LocationVariable) salClassDescription.getVariable(locationVariableName);
                        if (locationVariable == null) {
                            locationVariable = new SALClassDescription.LocationVariable(locationVariableName, match.getPatternName() + " location");
                            salClassDescription.add(locationVariable);
                        }
                        SALClassDescription.PatternMatchedSALVariable var = new SALClassDescription.PatternMatchedSALVariable(match.getPatternName(), match.getLocation(), locationVariable, salClassParameterName, type, units, description);
                        var.addBusVariable(busVariable);
                        salClassDescription.add(var);
                    }
                } else {
                    String topicName = makeSALTopicName(level, cscName, subsystemName, salType, originalPathName, category);
                    Class simpleSalClass = makeSALClasseName(topicName, salType, camera);
                    SALClassDescription salClassDescription = SALClassDescriptionMap.computeIfAbsent(topicName, (name) -> new SALClassDescription(topicName, simpleSalClass, level, category));
                    SALClassDescription.BusVariable busVariable = new SALClassDescription.BusVariable(originalPathName);
                    SALClassDescription.SimpleSALVariable var = new SALClassDescription.SimpleSALVariable(makeSALClassParameterName(originalPathName, level), type, units, description, busVariable);
                    salClassDescription.add(var);
                    if ( simpleSalClass != null ) {
                        SALClassMap.put(topicName, simpleSalClass);
                    }
                }
            }
        }

        // If there is only one location variable for a given class, then call it location unless it is already called that.
        for (SALClassDescription classDescription : SALClassDescriptionMap.values()) {

            // Find all location variables
            List<SALVariable> locationVariables = classDescription.getVariables().stream().filter(v -> v instanceof SALClassDescription.LocationVariable).collect(Collectors.toList());
            if (locationVariables.size() == 1) {
                SALVariable theLocationVariable = locationVariables.get(0);
                classDescription.renameVariable(theLocationVariable, "location");
            }
        }        
    }

    public DictionaryConfiguration getDictionaryConfiguration() {
        return dictConfig;
    }

    public Map<String, SALClassDescription> getSALClassDescriptions() {
        return SALClassDescriptionMap;
    }

    public Map<String, Class> getSALClasses() {
        return SALClassMap;
    }
    
    private Class makeSALClasseName(String topicName, SALType salType, Camera camera) {
        
        String[] packages = null;
        switch (camera) {
            case COMCAM:
                packages = new String[]{"org.lsst.sal.cccamera.event", "org.lsst.sal.camera.event", "org.lsst.sal.cccamera.telemetry", "org.lsst.sal.camera.telemetry"};
                break;
            case AUXTEL:
                packages = new String[]{"org.lsst.sal.atcamera.event", "org.lsst.sal.camera.event", "org.lsst.sal.atcamera.telemetry", "org.lsst.sal.camera.telemetry"};
                break;
            case MAIN_CAMERA:
                packages = new String[]{"org.lsst.sal.camera.event", "org.lsst.sal.camera.telemetry"};
                break;
        }

        String tmpTopicName = topicName;

        tmpTopicName = tmpTopicName.replace("-", "_");
        if (salType == SALType.SETTINGS_APPLIED) {
            tmpTopicName = tmpTopicName.replace(camera.getCscName() + "_logevent_", "");
        } else {
            tmpTopicName = tmpTopicName.replace(camera.getCscName() + "_", "");
        }
        tmpTopicName = capitalize(tmpTopicName);
        //className = "org.lsst.sal.cccamera.event." + className;

        //SALClassDescription scd = entry.getValue();
        //String category = scd.getCategory();
        //if (category != null && !category.isEmpty()) className += "_"+category;
        if (salType == SALType.SETTINGS_APPLIED) {
            tmpTopicName += "ConfigurationEvent";
        } else {
            tmpTopicName += "Telemetry";
        }
        Class<?> classForName = null;
        for (String pkg : packages) {
            try {
                //System.out.println("Trying "+pkg + "." + className);
                classForName = Class.forName(pkg + "." + tmpTopicName);
                break;
            } catch (ClassNotFoundException x) {
            }
        }
        if (classForName == null) {
            // TODO: Ignore MT_CAMERA vacuum at the moment, since the xml 20 vacuum sections is out-of-date
            Level messageLevel = Level.WARNING;
            if (camera == Camera.MAIN_CAMERA && (topicName.contains("shutter") || topicName.contains("fcs") || topicName.contains("vacuum") || topicName.contains("StatusAggregator"))) messageLevel = Level.FINE;
            if (topicName.contains("InfluxDb")) messageLevel = Level.FINE;
            LOG.log(messageLevel, "For {0} {1} {2} Class not found {3}", new Object[]{topicName, salType, camera, tmpTopicName});
        }
        return classForName;
    }
    
    /**
     * Construct the SALClassName
     * @param level The level at which the conversion is being made
     * @param cscName The CSCName
     * @param subsystemName The CCS subsystem name
     * @param salType The SALType 
     * @param path The path
     * @param category The category
     * @return 
     */
    private String makeSALTopicName(int level, String cscName, String subsystemName, SALType salType, String path, String category) {

        if (cscName.toLowerCase().contains("comcam")) {
            cscName = "CCCamera";
        }

        // debug
        String subsystemAndComponent = "";
        if (!cscName.isEmpty()) {

            subsystemAndComponent = cscName.substring(0, 1).toUpperCase() + cscName.substring(1);
        } // there should always be a subsystem 
        subsystemName = subsystemName.trim();

        if (!subsystemName.isEmpty()) {
            subsystemAndComponent += "_" + subsystemName.substring(0, 1).toLowerCase() + subsystemName.substring(1);
        }

        String className;

        if (salType == SALType.SETTINGS_APPLIED) {
            className = cscName + "_" + "logevent" + "_" + subsystemName.toLowerCase();
        } else {
            className = subsystemAndComponent;
        }

        // Obsolete?
        if (path.contains("main/")) {
            path = path.replace("main/", "");
            className = subsystemAndComponent;// and thats it - 

        } else {// go upto level - 

            for (int i = 0; i < level; i++) {
                final String pathComponentI = path.split("/")[i];
                if (pathComponentI.length() > 0) {
                    className = className + "_" + capitalize(pathComponentI);
                }
            }

        }

        if (!(category == null)) {
            if (!category.isEmpty()) {
                className = className + "_" + category;
            }

        }

        // check to see if last character is underscore if so remove it 
        if (className.endsWith("_")) {
            className = className.substring(0, className.length() - 1);
        }

        return className;
    }

    final String makeSALClassParameterName(String path, int level) {
        path = path.replace("main/", "");
        int pathLength = path.split("/").length;
        String variableName = "";
        path = path.replaceAll("-","_");
       
        // Fix forward slash problem Farrukh August 9, 2022
        if(path.startsWith("/"))path = path.replace("/", "");

        if (path.contains("/")) {
            // level better be less than or at most equal to path-length 
            if (level <= pathLength) {
                for (int i = level; i < pathLength; i++) {

                    variableName = variableName + "_" + path.split("/")[i];
                }

                if (variableName.startsWith("_")) {
                    variableName = variableName.substring(1);

                }
            } else {
                System.out.println("!!You have specified level to be longer than the variable path + name!!");
                exit(-1);

            }
        } else {
            variableName = path;
        }
 // Finally remove illegal character "-" from path Farrukh April 8 2022
//         variableName = variableName.replace('-', '_');

        return uncapitalize(variableName).trim();
    }

    public static String capitalize(String in) {
        if (in.length() > 1) {
            return in.substring(0, 1).toUpperCase() + in.substring(1);
        } else {
            return in.toUpperCase();
        }
    }

    public static String uncapitalize(String in) {
        if (in.length() > 1) {
            return in.substring(0, 1).toLowerCase() + in.substring(1);
        } else {
            return in.toLowerCase();
        }
    }

}
