package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.subsystem.ocsbridge.config.Camera;
import org.lsst.ccs.subsystem.ocsbridge.util.SerializationUtils;
import org.lsst.ccs.subsystem.ocsbridge.xml.XMLMaker2.SALType;

/**
 * Encapsulate the configuration needed to write XML files.
 *
 * @author tonyj
 */
public class MakeXMLConfiguration {

    /**
     * Create a MakeXMLConfiguration suitable for all subsystems for a given
     * camera and SALtype.
     *
     * @param camera The type of camera
     * @param salType The salType
     * @return A configuration suitable for use with XMLMaker2
     */
    public static MakeXMLConfiguration getInstance(Camera camera, SALType salType) {
        return new MakeXMLConfiguration(camera, salType);
    }

    /**
     * Create a MakeXMLConfiguration suitable for use with a single agentName
     * with a corresponding dictionary
     *
     * @param camera The type of camera
     * @param salType The salType
     * @param agentName The agent name
     * @param dictionary The dictionary corresponding to the agent name
     * @return A configuration suitable for use with XMLMaker2
     */
    public static MakeXMLConfiguration getInstance(Camera camera, SALType salType, String agentName, DataProviderDictionary dictionary) {
        return new MakeXMLConfiguration(camera, salType, agentName, dictionary);
    }

    // For each agent/SALType we need, level, componentName, ser file name
    private enum AgentConversionInfo {
        AUXTEL_TELEMETRY_ATS_FP(Camera.AUXTEL, SALType.TELEMETRY, "ats-fp", 1, "focal_plane", "ats-fp_dataDictionary.ser"),
        AUXTEL_TELEMETRY_ATS_DAQ_MONITOR(Camera.AUXTEL, SALType.TELEMETRY, "ats-daq-monitor", 1, "daq_monitor", "ats-daq-monitor_dataDictionary.ser"),
        AUXTEL_TELEMETRY_ATS_POWER(Camera.AUXTEL, SALType.TELEMETRY, "ats-power", 0, "power", "ats-power-status-dictionary.ser"),
        AUXTEL_TELEMETRY_ATS(Camera.AUXTEL, SALType.TELEMETRY, "ats", 0, "vacuum", "ats-status-dictionary.ser"),
        AUXTEL_TELEMETRY_ATS_BONN_SHUTTER(Camera.AUXTEL, SALType.TELEMETRY, "bonn-shutter", 1, "bonn_shutter", "bonn-shutter-status-dictionary.ser"),
        AUXTEL_TELEMETRY_ATS_IH(Camera.AUXTEL, SALType.TELEMETRY, "ats-ih", 1, "image_handling", "ats-ih_dataDictionary.ser"),
        // 
        AUXTEL_EVENT_ATS_FP(Camera.AUXTEL, SALType.SETTINGS_APPLIED, "ats-fp", 1, "focal_plane", "ats-fp_dataDictionary.ser"),
        AUXTEL_EVENT_ATS_DAQ_MONITOR(Camera.AUXTEL, SALType.SETTINGS_APPLIED, "ats-daq-monitor", 1, "daq_monitor", "ats-daq-monitor_dataDictionary.ser"),
        AUXTEL_EVENT_ATS_POWER(Camera.AUXTEL, SALType.SETTINGS_APPLIED, "ats-power", 1, "ats_power", "ats-power-status-dictionary.ser"),
        AUXTEL_EVENT_ATS(Camera.AUXTEL, SALType.SETTINGS_APPLIED, "ats", 1, "ats", "ats-status-dictionary.ser"),
        AUXTEL_EVENT_ATS_BONN_SHUTTER(Camera.AUXTEL, SALType.SETTINGS_APPLIED, "bonn-shutter", 1, "bonn_shutter", "bonn-shutter-status-dictionary.ser"),
        AUXTEL_EVENT_ATS_IH(Camera.AUXTEL, SALType.SETTINGS_APPLIED, "ats-ih", 1, "image_handling", "ats-ih_dataDictionary.ser"),
        //    
        COMCAM_TELEMETRY_COMCAM_FCS(Camera.COMCAM, SALType.TELEMETRY, "comcam-fcs", 0, "fcs", "comcam-fcs-status-dictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_BONN_SHUTTER(Camera.COMCAM, SALType.TELEMETRY, "bonn-shutter", 1, "bonn_shutter", "comcam-bonn-shutter-status-dictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_DAQ_MONITOR(Camera.COMCAM, SALType.TELEMETRY, "comcam-daq-monitor", 1, "daq_monitor", "comcam-daq-monitor_dataDictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_REBPOWER(Camera.COMCAM, SALType.TELEMETRY, "comcam-rebpower", 1, "rebpower", "comcam-rebpower-status-dictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_VACUUM(Camera.COMCAM, SALType.TELEMETRY, "comcam-vacuum", 1, "vacuum", "comcam-vacuum-status-dictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_QUADBOX(Camera.COMCAM, SALType.TELEMETRY, "comcam-quadbox", 1, "quadbox", "comcam-quadbox-status-dictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_FP(Camera.COMCAM, SALType.TELEMETRY, "comcam-fp", 1, "focal_plane", "comcam-fp_dataDictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_IH(Camera.COMCAM, SALType.TELEMETRY, "comcam-ih", 1, "image_handling", "comcam-ih_dataDictionary.ser"),
        COMCAM_TELEMETRY_COMCAM_MPM(Camera.COMCAM, SALType.TELEMETRY, "comcam-mpm", 1, "mpm", "comcam-mpm-status-dictionary.ser"),
        //
        COMCAM_EVENT_COMCAM_FCS(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-fcs", 1, "fcs", "comcam-fcs-status-dictionary.ser"),
        COMCAM_EVENT_COMCAM_BONN_SHUTTER(Camera.COMCAM, SALType.SETTINGS_APPLIED, "bonn-shutter", 1, "bonn_shutter", "comcam-bonn-shutter-status-dictionary.ser"),
        COMCAM_EVENT_COMCAM_DAQ_MONITOR(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-daq-monitor", 1, "daq_monitor", "comcam-daq-monitor_dataDictionary.ser"),
        COMCAM_EVENT_COMCAM_REBPOWER(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-rebpower", 1, "rebpower", "comcam-rebpower-status-dictionary.ser"),
        COMCAM_EVENT_COMCAM_VACUUM(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-vacuum", 1, "vacuum", "comcam-vacuum-status-dictionary.ser"),
        COMCAM_EVENT_COMCAM_QUADBOX(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-quadbox", 1, "quadbox", "comcam-quadbox-status-dictionary.ser"),
        COMCAM_EVENT_COMCAM_FP(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-fp", 1, "focal_plane", "comcam-fp_dataDictionary.ser"),
        COMCAM_EVENT_COMCAM_IH(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-ih", 1, "image_handling", "comcam-ih_dataDictionary.ser"),
        COMCAM_EVENT_COMCAM_MPM(Camera.COMCAM, SALType.SETTINGS_APPLIED, "comcam-mpm", 1, "mpm", "comcam-mpm-status-dictionary.ser"),
        //
        CAMERA_TELEMETRY_QUADBOX(Camera.MAIN_CAMERA, SALType.TELEMETRY, "quadbox", 1, "quadbox", "quadbox-status-dictionary.ser"),
        CAMERA_TELEMETRY_REBPOWER(Camera.MAIN_CAMERA, SALType.TELEMETRY, "rebpower", 1, "rebpower", "rebpower-status-dictionary.ser"),
        CAMERA_TELEMETRY_HEX(Camera.MAIN_CAMERA, SALType.TELEMETRY, "hex", 1, "hex", "hex_dataDictionary.ser"),
        CAMERA_TELEMETRY_REFRIG(Camera.MAIN_CAMERA, SALType.TELEMETRY, "refrig", 1, "refrig", "refrig_dataDictionary.ser"),
        CAMERA_TELEMETRY_VACUUM(Camera.MAIN_CAMERA, SALType.TELEMETRY, "vacuum", 1, "vacuum", "vacuum_dataDictionary.ser"),
        CAMERA_TELEMETRY_DAQ_MONITOR(Camera.MAIN_CAMERA, SALType.TELEMETRY, "daq-monitor", 1, "daq_monitor", "daq-monitor_dataDictionary.ser"),
        CAMERA_TELEMETRY_FOCAL_PLANE(Camera.MAIN_CAMERA, SALType.TELEMETRY, "focal-plane", 1, "focal_plane", "focal-plane_dataDictionary.ser"),
        CAMERA_TELEMETRY_FCS(Camera.MAIN_CAMERA, SALType.TELEMETRY, "fcs", 0, "fcs", "fcs-status-dictionary.ser"),
        CAMERA_TELEMETRY_SHUTTER(Camera.MAIN_CAMERA, SALType.TELEMETRY, "cam-shutter", 0, "shutter", "cam-shutter-status-dictionary.ser"),
        CAMERA_TELEMETRY_CHILLER(Camera.MAIN_CAMERA, SALType.TELEMETRY, "chiller", 1, "chiller", "chiller_dataDictionary.ser"),
        //
        CAMERA_EVENT_QUADBOX(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "quadbox", 1, "quadbox", "quadbox-status-dictionary.ser"),
        CAMERA_EVENT_REBPOWER(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "rebpower", 1, "rebpower", "rebpower-status-dictionary.ser"),
        CAMERA_EVENT_HEX(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "hex", 1, "hex", "hex_dataDictionary.ser"),
        CAMERA_EVENT_REFRIG(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "refrig", 1, "refrig", "refrig_dataDictionary.ser"),
        CAMERA_EVENT_VACUUM(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "vacuum", 1, "vacuum", "vacuum_dataDictionary.ser"),
        CAMERA_EVENT_DAQ_MONITOR(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "daq-monitor", 1, "daq_monitor", "daq-monitor_dataDictionary.ser"),
        CAMERA_EVENT_FOCAL_PLANE(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "focal-plane", 1, "focal_plane", "focal-plane_dataDictionary.ser"),
        // TODO: Deal with multiple instances
        CAMERA_EVENT_IMAGE_HANDLING(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "image-handling", 1, "image-handling", "image-handling-lsst-dc01_dataDictionary.ser"),
        CAMERA_EVENT_FCS(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "fcs", 0, "fcs", "fcs-status-dictionary.ser"),
        CAMERA_EVENT_SHUTTER(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "cam-shutter", 0, "shutter", "cam-shutter-status-dictionary.ser"),
        CAMERA_EVENT_CHILLER(Camera.MAIN_CAMERA, SALType.SETTINGS_APPLIED, "chiller", 0, "chiller", "chiller_dataDictionary.ser");

        private Camera camera;
        private SALType salType;
        private String agentName;
        private int level;
        private String componentName;
        private String dictionaryFile;

        AgentConversionInfo(Camera camera, SALType salType, String agentName, int level, String componentName, String dictionaryFile) {
            this.camera = camera;
            this.salType = salType;
            this.agentName = agentName;
            this.level = level;
            this.componentName = componentName;
            this.dictionaryFile = dictionaryFile;
        }

        public int getLevel() {
            return level;
        }

        public String getComponentName() {
            return componentName;
        }

        public String getDictionaryFile() {
            return camera.cameraName()+"/"+dictionaryFile;
        }

        public SALType getSALType() {
            return salType;
        }

        public String getAgentName() {
            return agentName;
        }

        public Camera getCamera() {
            return camera;
        }
        
        public String getCSCName() {
            return camera.getCscName();
        }
        
    }

    private final Camera camera;
    private final SALType salType;
    private final Mapping mapping;
    private final List<DictionaryConfiguration> dictionaries;

    private MakeXMLConfiguration(Camera camera, SALType salType) {
        this.salType = salType;
        this.camera = camera;
        mapping = Mapping.defaultMapping();

        dictionaries = initDictionaries(Arrays.stream(AgentConversionInfo.values()).filter(info -> info.getCamera() == camera).filter(info -> info.getSALType() == salType).collect(Collectors.toList()));
    }

    private MakeXMLConfiguration(Camera camera, SALType salType, String agentName, DataProviderDictionary dictionary) {
        this.salType = salType;
        this.camera = camera;

        mapping = Mapping.defaultMapping();

        List<DictionaryConfiguration> dicts = Collections.emptyList();
        for (AgentConversionInfo info : AgentConversionInfo.values()) {
            if (info.getSALType() == salType && info.getAgentName().equals(agentName)) {
                dicts = Collections.singletonList(new DictionaryConfiguration(info, dictionary));
                break;
            }
        }
        dictionaries = dicts;
    }

    private List<DictionaryConfiguration> initDictionaries(List<AgentConversionInfo> infos) {
        List<DictionaryConfiguration> dicts = new ArrayList<>();
        for (AgentConversionInfo info : infos) {
            dicts.add(new DictionaryConfiguration(info));
        }
        return dicts;
    }

    public String getXMLFileName() {
        return camera.getCscName() + salType.getXMLFileSuffix();
    }

    public String getXMLFileNameForAgent(String agentName) {
        return camera.getCscName() + "_" + agentName + salType.getXMLFileSuffix();
    }

    public String getCSCName() {
        return camera.getCscName();
    }

    public List<DictionaryConfiguration> getOrderedListOfDictionaryConfigurations() {
        return dictionaries;
    }
    
    public SALType getSALType() {
        return salType;
    }

    public Mapping getMapping() {
        return mapping;
    }

    public DictionaryConfiguration getDictionaryConfigurationForAgent(String agentName) {
        for ( DictionaryConfiguration dc : dictionaries ) {
            if ( dc.getAgentName().equals(agentName) ) {
                return dc;
            }
        }
        return null;
    }
    
    public static class DictionaryConfiguration {

        private DataProviderDictionary dictionary;
        private final AgentConversionInfo info;

        DictionaryConfiguration(AgentConversionInfo info, DataProviderDictionary dictionary) {
            this.info = info;
            this.dictionary = dictionary;
        }

        DictionaryConfiguration(AgentConversionInfo info) {
            this.info = info;
        }

        public DataProviderDictionary getDictionary() throws IOException, ClassNotFoundException {
            if ( dictionary == null ) {
                this.dictionary = SerializationUtils.readDictionaryFromFile(info.getDictionaryFile());                
            }
            return dictionary;
        }

        public int getLevel() {
            return info.getLevel();
        }

        public String getComponentName() {
            return info.getComponentName();
        }

        public String getAgentName() {
            return info.getAgentName();
        }
        
        public String getCSCName() {
            return info.getCSCName();
        }
        
        public SALType getSALType() {
            return info.getSALType();
        }
        
        public Camera getCamera() {
            return info.getCamera();
        }
    }
}
