package org.lsst.ccs.subsystem.ocsbridge.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;

/**
 * Utility class to read/write serialized files.
 * 
 * @author The LSST CCS Team 
 */
public class SerializationUtils {

    private static final Logger LOG = Logger.getLogger(SerializationUtils.class.getName());
    
    
    
    
    public static DataProviderDictionary readDictionaryFromFile(String dictionaryFileName) {
        Object obj = null;
        try ( ObjectInputStream ois = new ObjectInputStream(getInputStreamFor(dictionaryFileName))) {
            obj = ois.readObject();
        } catch (IOException | ClassNotFoundException ex) {
            throw new RuntimeException("Failed to read " + dictionaryFileName, ex);
        }
        
        DataProviderDictionary dict = null;
        if ( obj instanceof DataProviderDictionary ) {
            dict = (DataProviderDictionary)obj;
        } else if ( obj instanceof StatusDataProviderDictionary ) {
            dict = ((StatusDataProviderDictionary) obj).getDataProviderDictionary();
        } else if ( obj instanceof byte[] ) {
            StatusDataProviderDictionary sdpd = new StatusDataProviderDictionary((byte[])obj, "gzip");
            dict = sdpd.getDataProviderDictionary();
        } else {
            throw new RuntimeException("Cannot process serialized object of type "+obj.getClass());            
        }
        
        return dict;
        
    }
    
    
    public static StatusConfigurationInfo readStatusConfigurationInfoFromFile(String fileName) throws IOException, ClassNotFoundException {
        try ( InputStream in = getInputStreamFor(fileName);  ObjectInputStream ois = new ObjectInputStream(in)) {
            Object readObject = ois.readObject();
            return (StatusConfigurationInfo) readObject;
        }
    }

    
    
    public static StatusSubsystemData readStatusSubsystemData(String fileName) throws IOException, ClassNotFoundException {
        try (InputStream in = getInputStreamFor(fileName); ObjectInputStream ois = new ObjectInputStream(in)) {
            Object readObject = ois.readObject();
            return (StatusSubsystemData) readObject;
        }
    }
    

    private static InputStream getInputStreamFor(String fileName) {
        String tmpFullPath = fileName;
        InputStream is = BootstrapResourceUtils.getBootstrapResource(fileName,MCM.class);
        if ( is == null ) {
            File inputFile = new File(fileName);
            if (!inputFile.exists()) {
                throw new RuntimeException("File " + fileName + " does not exist.");
            }
            if (!inputFile.canRead()) {
                throw new RuntimeException("File " + fileName + " cannot be read.");
            }
            tmpFullPath = inputFile.getAbsolutePath();
            try {
                is = new FileInputStream(inputFile);
            } catch (FileNotFoundException ex) {
                throw new RuntimeException(ex);
            }
        } else {
            String filePath = BootstrapResourceUtils.getPathOfResourceInUserResourceDirectories(fileName);
            if (filePath != null) {
                tmpFullPath = filePath;
            }
        }
        LOG.log(Level.INFO, "Reading file {0}", tmpFullPath);            
        return is;
    }
    
}
