package org.lsst.ccs.subsystem.ocsbridge.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.subsystem.ocsbridge.config.Camera;
import org.lsst.ccs.subsystem.ocsbridge.xml.MakeXMLConfiguration;
import org.lsst.ccs.subsystem.ocsbridge.xml.MakeXMLConfiguration.DictionaryConfiguration;
import org.lsst.ccs.subsystem.ocsbridge.xml.Mapping;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescriptionMaker;
import org.lsst.ccs.subsystem.ocsbridge.xml.XMLMaker2.SALType;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Utility class to extract checksum information from either a
 * SAL XML file or a CCS DataProviderDictionary.
 *
 * @author tonyj
 */
public class ChecksumExtractorUtils {

    private final Pattern commentPattern = Pattern.compile("CCS: Dictionary_Checksum: (\\d+)L\\s+");
    private final XPathExpression commentExpression;
    private final XPathExpression topicExpression;
    
    public ChecksumExtractorUtils() throws XPathExpressionException {
        // We are looking for items of the form
        //        <SALEventSet xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="https://raw.githubusercontent.com/lsst-ts/ts_xml/develop/python/lsst/ts/xml/data/schema/SALEventSet.xsd">
        //            <SALEvent>
        //                <Subsystem>CCCamera</Subsystem>
        //                <EFDB_Topic>CCCamera_logevent_mpm_Pluto_DevicesConfiguration</EFDB_Topic>
        //                <!--CCS: Dictionary_Checksum: 1617359185L -->
        final XPathFactory xPathFactory = XPathFactory.newInstance();
        final XPath xPath = xPathFactory.newXPath();
        // OK to compile two expressions from a single XPath? Seems to work.
        topicExpression = xPath.compile("//EFDB_Topic");
        commentExpression = xPath.compile("following-sibling::comment()[1]");
    }
    /**
     * Search for and return topics with associated check sums.
     * @param in An stream which will be parsed to extract the XML document 
     * @return A map of Topic->Checksum
     * @throws org.xml.sax.SAXException
     * @throws XPathExpressionException 
     * @throws java.io.IOException 
     * @throws javax.xml.parsers.ParserConfigurationException 
     */
    public Map<String, Long> extractChecksumsFromSALXMLInputStream(InputStream in) throws SAXException, XPathExpressionException, IOException, ParserConfigurationException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        DocumentBuilder builder = factory.newDocumentBuilder();
        Document document = builder.parse(in);
        return extractChecksums(document);
    }
    
    /**
     * Search for and return topics with associated check sums.
     * @param document The parsed XML document from which the checksum should be extracted
     * @return A map of Topic->Checksum
     * @throws XPathExpressionException 
     */
    public Map<String, Long> extractChecksums(Document document) throws XPathExpressionException {
        Map<String, Long> result = new LinkedHashMap<>();

        NodeList topicNodes = (NodeList) topicExpression.evaluate(document, XPathConstants.NODESET);
        for (int i = 0; i < topicNodes.getLength(); i++) {
            Node topicNode = topicNodes.item(i);
            String topic = topicNode.getTextContent();
            Comment commentNode = (Comment) commentExpression.evaluate(topicNode, XPathConstants.NODE);
            if (commentNode != null) {
                String comment = commentNode.getNodeValue();
                Matcher matcher = commentPattern.matcher(comment);
                if (matcher.matches()) {
                    long checksum = Long.parseLong(matcher.group(1));
                    result.put(topic, checksum);
                }
            }
        }
        return result;
    }
    
    /**
     * Extract all the checksums for a CCS DataProvideDictionary
     * @param camera The camera corresponding to the dictionary
     * @param agentName The agent name
     * @param dataProviderDictionary The data provider dictionary
     * @return A map of Topic -> Checksum
     */
    public Map<String, Long> extractChecksumsFromDataProviderDictionaryForAgent(Camera camera, String agentName, DataProviderDictionary dataProviderDictionary) {
        Map<String, Long> result = new LinkedHashMap<>();
        
        result.putAll(extractChecksumsFromDataProviderDictionaryForSALType(camera, SALType.TELEMETRY, agentName, dataProviderDictionary));
        result.putAll(extractChecksumsFromDataProviderDictionaryForSALType(camera, SALType.SETTINGS_APPLIED, agentName, dataProviderDictionary));
        return result;
    }
    
    private Map<String, Long> extractChecksumsFromDataProviderDictionaryForSALType(Camera camera, SALType salType, String agentName, DataProviderDictionary dataProviderDictionary) {
        Map<String, Long> result = new LinkedHashMap<>();
        MakeXMLConfiguration config = MakeXMLConfiguration.getInstance(camera, salType, agentName, dataProviderDictionary);
        if ( config.getOrderedListOfDictionaryConfigurations().size() > 1 ) {
            throw new RuntimeException("Only one dictionary configuration expected here.");
        }
        for (MakeXMLConfiguration.DictionaryConfiguration dc : config.getOrderedListOfDictionaryConfigurations()) {
            result.putAll(extractChecksumsFromDictionaryConfiguration(dc, config.getMapping()));        
        }
        return result;
    }    
    
    /**
     * Extract checksums from a DictionaryConfiguration
     * @param dc the dictionary
     * @param mapping the mapping to use
     * @return A Map of <String, Long> giving the checksum for each topic
     */
    public Map<String, Long> extractChecksumsFromDictionaryConfiguration(DictionaryConfiguration dc, Mapping mapping) {
        Map<String, Long> result = new LinkedHashMap<>();

        SALClassDescriptionMaker infoMaker = new SALClassDescriptionMaker(dc, mapping);
        final Map<String, SALClassDescription> salClassDescriptions = infoMaker.getSALClassDescriptions();

        for (Map.Entry<String, SALClassDescription> entry : salClassDescriptions.entrySet()) {
            String name = entry.getKey();
            SALClassDescription scd = entry.getValue();
            result.put(name, scd.computeChecksum());
        }
        return result;
    }


    
}
