package org.lsst.ccs.subsystem.ocsbridge;

import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.subsystem.ocsbridge.config.Camera;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescriptionMaker;
import org.lsst.ccs.subsystem.ocsbridge.xml.XMLMaker2.SALType;
import org.lsst.sal.camera.CameraEvent;

/**
 * Handler receipt of configuration from CCS, conversion into equivalent SAL
 * messages, and send to SAL.
 *
 * @author tonyj
 */
public abstract class ConfigurationSender extends GenericSender {


    private static final Logger LOG = Logger.getLogger(ConfigurationSender.class.getName());
    private OCSEventSender sender;

    static ConfigurationSender create(Camera device, OCSEventSender sender) {
        switch (device) {
            case COMCAM:
                return new ComCamConfigurationSender(sender);
            case AUXTEL:
                return new AuxTelConfigurationSender(sender);
            case MAIN_CAMERA:
                return new CameraConfigurationSender(sender);
            default:
                throw new IllegalArgumentException("Unsupported device: " + device);
        }

    }

    protected ConfigurationSender(OCSEventSender sender, Camera camera) {
        super(camera,SALType.SETTINGS_APPLIED);
        this.sender = sender;
    }

    public void send(StatusConfigurationInfo data) {
        try {
            List<CameraEvent> converted = getConverter().settingsAppliedEventConverter(data);
            if (converted.isEmpty()) {
                LOG.log(Level.INFO, "No converted configuration {0}", data.getOriginAgentInfo().getName());
            } else {
                for (CameraEvent t : converted) {
                    t = applyAfterBurner(t);
                    LOG.log(Level.INFO, "Sending {0} {1}", new Object[]{data.getOriginAgentInfo().getName(), t});
                    sender.sendEvent(t);
                }
            }
        } catch (ReflectiveOperationException ex) {
            LOG.log(Level.WARNING, String.format("Problem converting configuration subsytem: %s key: %s", data.getOriginAgentInfo().getName(), data.getConfigurationInfo()));
        }
    }

    /**
     * Allow subclasses to modify sent events if necessary
     * @param t The event to be sent.
     * @return The input event or modified event
     */
    protected CameraEvent applyAfterBurner(CameraEvent t) {
        return t;
    }
    
    void setSender(OCSCommandExecutor ocs) {
        this.sender = ocs;
    }


    String getConfigurationEvents() {
        return getConverter().getClassMap().values().stream().map(c -> c.getSimpleName()).map(name -> name.replaceAll("Event$", "")).map(name -> SALClassDescriptionMaker.uncapitalize(name)).collect(Collectors.joining(","));
    }
}
