package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.ocsbridge.config.Camera;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListenerList;
import org.lsst.sal.SAL;
import org.lsst.sal.SALCommandResponse;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.sal.SALException;
import org.lsst.sal.camera.CameraStateChangeEvent;
import org.lsst.sal.camera.event.HeartbeatEvent;


/**
 * A standalone version of the OCSGui, which communicates with the camera using
 * SAL
 *
 * @author tonyj
 */
public class StandaloneOCSGUI {

    private final static Logger logger = Logger.getLogger(StandaloneOCSGUI.class.getName());

    public static void main(String[] args) {
        
        Camera device = Camera.MAIN_CAMERA;
        if (args.length > 0) {
            device = Camera.valueOf(args[0]);
        }
        
        SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr = OCSBridgeConfig.createSALManager(device);
        final GUISALLayer guisalLayer = new GUISALLayer(mgr);
        ToyOCSGUI gui = new ToyOCSGUI(guisalLayer, device);
        gui.setVisible(true);
        guisalLayer.start();
    }

    private static class GUISALLayer implements GUILayer {

        private final SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr;
        private final EventListenerList<CameraEvent> cell = new EventListenerList<>();
        private final EventListenerList<CameraStateChangeEvent> cscel = new EventListenerList<>();

        public GUISALLayer(SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr) {
            this.mgr = mgr;
        }

        public void start() {

            Thread eventThread = new Thread("SALEventReceiver") {

                @Override
                public void run() {
                    try {
                        runEventLoop();
                    } catch (Exception x) {
                        logger.log(Level.WARNING, "Failed to initialize SAL communication layer for events, "
                                + "check that SAL has been setup correctly.", x);
                    }
                }
            };
            eventThread.start();

            Thread telemetryThread = new Thread("SALTelemetryReceiver") {

                @Override
                public void run() {
                    try {
                        runTelemetryLoop();
                    } catch (Exception x) {
                        logger.log(Level.WARNING, "Failed to initialize SAL communication layer for telemetry, "
                                + "check that SAL has been setup correctly.", x);
                    }
                }
            };
            telemetryThread.start();
        }

        
        
        @Override
        public void execute(CameraCommand cmd) {
            try {
                logger.log(Level.INFO, "Sending {0}", cmd);
                SALCommandResponse response = mgr.issueCommand(cmd);
                Duration estimatedTime = response.waitForAck(Duration.ofSeconds(10));
                if (estimatedTime != Duration.ZERO) {
                    logger.log(Level.INFO, "Expected command duration {0}", estimatedTime);
                }
                int rc = response.waitForCompletion(estimatedTime.plus(Duration.ofSeconds(10)));
                logger.log(Level.INFO, "Command complete ack={0}", rc);
            }  catch (SALCommandResponse.CommandFailedException ex) {
                logger.log(Level.SEVERE, "Command failed: errorCode={1}, ack={2}, message={0}", new Object[]{ex.getMessage(), ex.getErrorCode(), ex.getAck()});
            } catch (TimeoutException ex) {
                logger.log(Level.WARNING, "Timeout while waiting for ack from: {0}", cmd);
            } catch (SALException ex) {
                logger.log(Level.SEVERE, "Exception executing SAL command: {0}", ex.getMessage());
            }
        }

        private void runEventLoop() {
            try {
                for (;;) {
                    CameraEvent event = mgr.getNextEvent(Duration.ofMinutes(1));
                    if (event == null) {
                        logger.fine("Still waiting for an event");
                    } else {
                        Level level = event instanceof HeartbeatEvent ? Level.FINE : Level.INFO;
                        logger.log(level, "Received {0}", event);
                        if (event instanceof CameraStateChangeEvent) {
                            cscel.fireEvent((CameraStateChangeEvent) event);
                        } else {
                            cell.fireEvent(event);
                        }
                    }
                }
            } catch (SALException ex) {
                logger.log(Level.SEVERE, "Unexpected exception while waiting for events", ex);
            }
        }

        private void runTelemetryLoop() {
            try {
                for (;;) {
                    CameraTelemetry telemetry = mgr.getTelemetry(Duration.ofMinutes(1));
                    if (telemetry != null) {
                        logger.log(Level.INFO, "Received {0}", telemetry);
                    }
                }
            } catch (SALException ex) {
                logger.log(Level.SEVERE, "Unexpected exception while waiting for events", ex);
            }
        }
        

        @Override
        public void execute(CCSCommand command) {
            // NOOP - not supported in standalone mode
        }

        @Override
        public boolean supportsCCSCommands() {
            return false;
        }

        @Override
        public void addEventListener(EventListener<CameraEvent> cel) {
            cell.addEventListener(cel);
        }

        @Override
        public void removeEventListener(EventListener<CameraEvent> cel) {
            cell.removeEventListener(cel);
        }

        @Override
        public void addStateChangeListener(EventListener<CameraStateChangeEvent> cscl) {
            cscel.addEventListener(cscl);
        }

        @Override
        public void removeStateChangeListener(EventListener<CameraStateChangeEvent> cscl) {
            cscel.removeEventListener(cscl);
        }
    }
}
