package org.lsst.ccs.subsystem.ocsbridge;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.DataProviderDictionaryService.DataProviderDictionaryEvent.EventType;
import org.lsst.ccs.services.DataProviderDictionaryService.DataProviderDictionaryListener;
import org.lsst.ccs.subsystem.ocsbridge.xml.Mapping;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription;

/**
 * DictioanaryProvider is placed on the tree in the OCSBridge groovy file.
 * Its purpose is to listen for dictionaries published on the buses and keep a list of them which 
 * are used for converting telemetry and configuration to be sent to the 
 * OCS.
 * @author tonyj
 */
public class DictionaryProvider implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(DictionaryProvider.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private DataProviderDictionaryService dpdService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private OCSBridgeConfig config;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private OCSSubsystem ocsSubsystem;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private OCSBridgeConfig ocsBridgeConfig;
    
    private final Listener listener = new Listener();
    private final Map<String, SALClassDescription> salMonitoringClassDescriptions = new HashMap<>();
    private final Map<String, SALClassDescription> salConfigurationClassDescriptions = new HashMap<>();

    
    @Override
    public void start() {
        dpdService.addDataProviderDictionaryListener(listener);
    }

    Map<String, SALClassDescription> getStatusClassDescriptions() {
        return salMonitoringClassDescriptions;
    }
    
    Map<String, SALClassDescription> getConfigurationClassDescriptions() {
        return salConfigurationClassDescriptions;
    }

    private class Listener implements DataProviderDictionaryListener {

        Mapping mapping = Mapping.defaultMapping();
        
        @Override
        public void dataProviderDictionaryUpdate(DataProviderDictionaryService.DataProviderDictionaryEvent evt) {
            try {
                if (evt.getEventType() == EventType.ADDED) {
                    AgentInfo.AgentType type = evt.getAgentInfo().getType();
                    if (type == AgentInfo.AgentType.WORKER || type == AgentInfo.AgentType.SERVICE) {
                        String name = evt.getAgentInfo().getName();
                        for (String l : ocsBridgeConfig.getListenSubsystems()) {
                            if (l.equals(name) ) {
                                DataProviderDictionary dictionary = evt.getDictionary();
                                ocsSubsystem.addSubsystem(name, dictionary);
                                LOG.log(Level.INFO, "Added dictionary for {0}", new Object[]{name});
                                break;
                            }
                        }
                    }
                } else if ( evt.getEventType() == EventType.REMOVED ) {
                    ocsSubsystem.removeSubsystem(evt.getAgentInfo().getName());
                }
            } catch (Throwable t) {
                LOG.log(Level.SEVERE, "Exception adding dictionary", t);
            }
        }

    }
}

