package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Maintains a list of patterns to match. These patterns are applied to incoming
 * paths to tell if the corresponding CCS values should be condensed into arrays in
 * the SAL classes.
 *
 * @author tonyj
 */
public class Mapping {

    private final List<PatternPlus> patterns = new ArrayList<>();

    /**
     * Create the default Mapping
     *
     * @return The default Mapping
     */
    public static Mapping defaultMapping() {
        try ( InputStream in = Mapping.class.getResourceAsStream("pattern.map")) {
            return new Mapping(in);
        } catch (IOException x) {
            throw new RuntimeException("Failed to load default pattern.map", x);
        }
    }

    public Mapping(InputStream in) throws IOException {
        Pattern legacyPattern = Pattern.compile("(\\w+)\\s*:\\s(.*)");
        Pattern flexiblePattern = Pattern.compile("(\\w+)\\s*:\\s*\\|\\s*(.+)\\s*\\|\\s*(.+)\\s*\\|\\s*(.+)\\s*\\|");
        try ( BufferedReader reader = new BufferedReader(new InputStreamReader(in))) {
            for (;;) {
                /**
                 * We support two types of patterns 
                 * Legacy: <pattern> : <name>
                 * Flexible: |<pattern>|<location>|<after>|: <name>
                 */
                String line = reader.readLine();
                if (line == null) {
                    break;
                } else if (line.startsWith("#")) {
                    continue;
                }
                Matcher flexibleMatcher = flexiblePattern.matcher(line);
                Matcher legacyMatcher = legacyPattern.matcher(line);

                if (flexibleMatcher.matches()) {
                    String key = flexibleMatcher.group(1);
                    String value = flexibleMatcher.group(2);
                    Pattern pattern = Pattern.compile(value, Pattern.CASE_INSENSITIVE);
                    patterns.add(new PatternPlus(pattern, key, flexibleMatcher.group(3), flexibleMatcher.group(4)));
                }
                else if (legacyMatcher.matches()) {
                    String key = legacyMatcher.group(1);
                    String value = legacyMatcher.group(2);
                    Pattern pattern = Pattern.compile(value, Pattern.CASE_INSENSITIVE);
                    patterns.add(new PatternPlus(pattern, key));
                } else {
                    throw new IOException("Invalid mapping file line: " + line);
                }
            }
        }
    }

    public List<PatternPlus> getPatternList() {
        return Collections.unmodifiableList(patterns);
    }

    /**
     * Test the given path for a match with the patterns
     *
     * @param path The path to test
     * @return A Match object or <code>null</code> if no match
     */
    public Match match(String path) {
        for (PatternPlus patternPlus : patterns) {
            Matcher matcher = patternPlus.getPattern().matcher(path);
            if (matcher.matches()) {
                return new Match(matcher, patternPlus);
            }
        }
        return null;
    }

    public Map<Pattern, String> getPatterns() {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    public static class PatternPlus {
        private final Pattern pattern;
        private final String patternName;
        private final String location;
        private final String after;
        
        /**
         * Legacy constructor
         * @param pattern
         * @param patternName 
         */
        PatternPlus(Pattern pattern, String patternName) {
            this.pattern = pattern;
            this.patternName = patternName;
            this.location = null;
            this.after = null;
        }
        /** 
         * Flexible constructor
         */
        private PatternPlus(Pattern pattern, String patternName, String location, String after) {
            this.pattern = pattern;
            this.patternName = patternName;
            this.location = location;
            this.after = after;
        }

        public Pattern getPattern() {
            return pattern;
        }

        public String getPatternName() {
            return patternName;
        }

        public String getLocation() {
            return location;
        }

        public String getAfter() {
            return after;
        }
    }

    public static class Match {

        private final Matcher matcher;
        private final PatternPlus patternPlus;

        private Match(Matcher matcher, PatternPlus patternPlus) {
            this.patternPlus = patternPlus;
            this.matcher = matcher;
        }

        /**
         * The pattern name that was matched
         *
         * @return
         */
        public String getPatternName() {
            return patternPlus.getPatternName();
        }

        /**
         * The part of the path that represents the matched location
         *
         * @return
         */
        public String getLocation() {
            if (patternPlus.getLocation() == null) {
                StringBuilder builder = new StringBuilder();
                int groupCount = matcher.groupCount();
                for (int g = 1; g < groupCount; g++) {
                    builder.append(matcher.group(g));
                }
                return builder.toString();
            } else {
                return matcher.replaceAll(patternPlus.getLocation());
            }
        }

        /**
         * The path after it has had the location removed
         *
         * @return
         */
        public String getPathAfterMatch() {
            if (patternPlus.getAfter() == null) {
                int groupCount = matcher.groupCount();
                return matcher.group(groupCount);
            } else {
                return matcher.replaceAll(patternPlus.getAfter());
            }
        }
    }
}
