package org.lsst.ccs.subsystem.ocsbridge.util;

import org.lsst.ccs.subsystem.ocsbridge.events.EventListenerList;
import org.lsst.ccs.subsystem.ocsbridge.events.EventListener;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSEvent;
import java.time.Duration;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSEvent.CCSEventListenerList;
import org.lsst.ccs.subsystem.ocsbridge.util.State.StateChangeListener;
import org.lsst.ccs.subsystem.ocsbridge.events.CCSEvent.CCSEventListener;

/**
 * Abstraction of subset of CCS toolkit. This class allows the MCM + OCSBriudge to be run in simulation/test
 * mode with no real CCS. 
 * This class deals with routing status messages and scheduling actions. it also allows status listeners 
 * to be added which will receive notification of any status change events.
 *
 * @author tonyj
 */
public class CCS {

    private final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(4);
    private final CCSEventListenerList eventListenerList = new CCSEventListenerList();
    private final EventListenerList<StatusMessage> statusMessageListenerList = new EventListenerList<>();
    private static final Logger LOG = Logger.getLogger(CCS.class.getName());

    private final AggregateStatus as = new AggregateStatus();

    <T> ScheduledFuture<T> schedule(Duration when, Callable<T> callable) {
        return scheduler.schedule(callable, when.toMillis(), TimeUnit.MILLISECONDS);
    }

    public ScheduledFuture<?> schedule(Duration when, Runnable runnable) {
        return scheduler.schedule(new RunnableWrapper(runnable), when.toMillis(), TimeUnit.MILLISECONDS);
    }
    
    public CompletableFuture<Void> scheduleCompletable(Duration when) {
        // TODO: This does not support cancelling the completable future.
        CompletableFuture<Void> future = new CompletableFuture<>();
        scheduler.schedule(()-> { future.complete(null); }, when.toMillis(), TimeUnit.MILLISECONDS);
        return future;
    }

    public void addStateChangeListener(StateChangeListener<? extends Enum> listener) {
        as.addStateChangeListener(listener);
    }

    public void removeStateChangeListener(StateChangeListener<? extends Enum> listener) {
        as.removeStateChangeListener(listener);
    }

    public void addEventListener(CCSEventListener listener) {
        eventListenerList.addEventListener(listener);
    }

    public void removeEventListener(CCSEventListener listener) {
        eventListenerList.removeEventListener(listener);
    }

    public void addStatusMessageListener(EventListener<StatusMessage> listener) {
        statusMessageListenerList.addEventListener(listener);
    }

    public void removeStatusMessageListener(EventListener<StatusMessage> listener) {
        statusMessageListenerList.removeEventListener(listener);
    }
    
    public void runInBackground(Runnable r) {
        scheduler.execute(r);
    }

    public void shutdown() throws InterruptedException {
        scheduler.shutdownNow();
        scheduler.awaitTermination(10, TimeUnit.SECONDS);
        eventListenerList.clear();
        statusMessageListenerList.clear();
    }

    public AggregateStatus getAggregateStatus() {
        return as;
    }

    public CompletableFuture<Void> waitForStatus(Enum state) {
        return as.waitForStatus(state);
    }

    public void fireEvent(CCSEvent event) {
        eventListenerList.fireEvent(event);
    }
    
    public void fireEvent(StatusMessage event) {
        statusMessageListenerList.fireEvent(event);
    }

    public void schduleAtFixedRate(int i, TimeUnit timeUnit, Runnable runnable) {
        scheduler.scheduleAtFixedRate(new RunnableWrapper(runnable), i, i, timeUnit);
    }

    public ScheduledExecutorService getScheduler() {
        return scheduler;
    }
    /**
     * A wrapper for scheduled runnables. Captures exceptions and logs them
     */
    private static class RunnableWrapper implements Runnable {

        private final Runnable runnable;

        public RunnableWrapper(Runnable runnable) {
            this.runnable = runnable;
        }

        @Override
        public void run() {
            try {
                runnable.run();
            } catch (Throwable t) {
                LOG.log(Level.SEVERE, "Error running scheduled runnable", t);
            }
        }
    }
}
