package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An interface for talking to a real comcam filter changer subsystem.
 *
 * @author tonyj
 */
public class MainCameraFilterChangerSubsystemLayer extends ControlledSubsystem implements FilterChangerInterface {

    private static final Logger LOG = Logger.getLogger(ComCamFilterChangerSubsystemLayer.class.getName());
    private final static Map<FcsEnumerations.McmState, FilterChanger.FilterState> FCS_TO_FILTER_STATE = new HashMap<>();

    static {
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.LOADING, FilterChanger.FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.UNLOADING, FilterChanger.FilterState.UNLOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.LOADING, FilterChanger.FilterState.LOADING);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.NO_FILTER, FilterChanger.FilterState.NOFILTER);
        FCS_TO_FILTER_STATE.put(FcsEnumerations.McmState.ROTATING, FilterChanger.FilterState.ROTATING);
    }

    class FilterStateConverter {

        public FilterStateConverter() {
            //System.out.println("building FSConverter");
        }

        FilterChanger.FilterState prevStaticState = null;

        private FilterChanger.FilterState convertStateInstantaneous(FcsEnumerations.McmState state) {
            FilterChanger.FilterState s = FCS_TO_FILTER_STATE.get(state);
            if (s == null) {
                s = FilterChanger.FilterState.LOADED;
            }
            return s;
        }

        FilterChanger.FilterState convertState(FcsEnumerations.McmState state) {
            FilterChanger.FilterState s = convertStateInstantaneous(state);
            return s;
        }
    }

    // Issue there?
    volatile FilterStateConverter converter; // = new FilterStateConverter();

    FilterChanger.FilterState convertState(FcsEnumerations.McmState state) {
        //System.out.println("converting state "+state+" with "+converter);
        return converter.convertState(state);
    }

    public MainCameraFilterChangerSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFilterChangerSubsystemName(), ccs, config);
        converter = new FilterStateConverter();
        //System.out.println("MCF built with converter "+converter);
    }

    // those commands were defined mostly in accordance with the ComCam filter
    // changer system
    // adapting here to final FCS logics
    @Override
    public void setFilter(String filter) throws ExecutionException {
        if ("NONE".equals(filter)) {
            commandSender.sendCommand(Void.TYPE, getEstimatedDurationForFilterChange(filter), "setNoFilter");
            waitUntilMoveComplete(FilterChanger.FilterState.NOFILTER, getEstimatedDurationForFilterChange(filter));
        } else {
            commandSender.sendCommand(Void.TYPE, getEstimatedDurationForFilterChange(filter), "setFilterByName", filter);
            waitUntilMoveComplete(FilterChanger.FilterState.LOADED, getEstimatedDurationForFilterChange(filter));
        }
    }

    @Override
    public Map<String, String> getAvailableFilters() throws ExecutionException {
        // map ? Why ? string keys and names, this is very specific of comcam implementation
        // let's retune name, name for FCS
        @SuppressWarnings("unchecked")
        List<String> l = commandSender.sendCommand(List.class, "listAllFilters");
        Map<String, String> m = new HashMap<>();
        l.forEach(s -> m.put(s, s));
        return m;
    }

    @Override
    public List<String> getInstalledFilters() throws ExecutionException {
        @SuppressWarnings("unchecked")
        Map<Integer, String> m = commandSender.sendCommand(Map.class, "listFiltersOnChanger");
        return new ArrayList<>(m.values());
    }

    @Override
    public String getCurrentFilter() throws ExecutionException {
        // Temporarily change to using printFilterONLINEName. Note this returns "NONE" rather than null.
        return commandSender.sendCommand(String.class, "printFilterONLINEName");
    }

    @Override
    public Duration getEstimatedDurationForFilterChange(String filterName) {
        // TODO: Would be good if coult ask the filter changer how long it expects
        // the move to take. Currently we can only ask the estimated time to a
        // known position.
        return Duration.ofSeconds(240);
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Filter changer connected");
        FcsEnumerations.FilterState state = initialState.getState(FcsEnumerations.FilterState.class);
        FcsEnumerations.McmState mcmState = initialState.getState(FcsEnumerations.McmState.class);
        LOG.log(Level.INFO, "Got initial McmState {0}", mcmState);
        FcsEnumerations.AutochangerInclination inclinationState = initialState.getState(FcsEnumerations.AutochangerInclination.class);
        LOG.log(Level.INFO, "Got initial AutochangerInclination {0}", inclinationState);
        translateFcsStateToFilterState(CCSTimeStamp.currentTime(), mcmState);
    }

    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Filter changer disconnected");
        // TOOD: Deal with initial state
        // StateBundle result = commandSender.sendCommand("getState",
        // StateBundle.class);
        // FocalPlaneState state = result.getState(FocalPlaneState.class);
        // translateFocalPlaneStateToRaftsState(state);
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        // CCSTimeStamp ts = statusChange.getCCSTimeStamp();
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        StateBundle changedStates = newStates.diffState(oldStates);
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue())
                .forEachOrdered((value) -> {
                    if (value instanceof FcsEnumerations.FilterState) {
                        //translateFcsStateToFilterState(when, (FcsEnumerations.FilterState) value);
                    } else if (value instanceof FcsEnumerations.McmState) {
                        LOG.log(Level.INFO, "Got McmState {0}", value);
                        translateFcsStateToFilterState(when, (FcsEnumerations.McmState) value);
                    } else if (value instanceof FcsEnumerations.AutochangerInclination) {
                        LOG.log(Level.INFO, "Got AutochangerInclination {0}", value);
                    }
                });
    }

    private void waitUntilMoveComplete(FilterChanger.FilterState finalState, Duration estimatedDurationForFilterChange)
            throws ExecutionException {
        // TOOD: Do we need to wait for it to start moving first?
        // yes of course and we should also wait for a possible unloaded end state if
        // set no filter.
        Future<Void> waitForStatus = ccs.waitForStatus(finalState);
        try {
            waitForStatus.get(estimatedDurationForFilterChange.toMillis(), TimeUnit.MILLISECONDS);
        } catch (InterruptedException | TimeoutException ex) {
            throw new ExecutionException("Timeout waiting for filter change to complete", ex);
        }
    }

    private void translateFcsStateToFilterState(CCSTimeStamp when, FcsEnumerations.McmState value) {
        LOG.log(Level.INFO, "Got filter changer state {0} ", value);
        if (value == null) value = FcsEnumerations.McmState.LOADED;
        FilterChanger.FilterState converted = convertState(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(when, new State(converted));
        }
    }

    @Override
    protected void onEvent(StatusMessage msg) {
        Object data = msg.getObject();
        if (data instanceof KeyValueDataList) {
            KeyValueDataList kvdl = (KeyValueDataList) data;
            //Object dataObject = kvdl.getListOfKeyValueData().get(0).getValue();
            String dataKey = ((KeyValueDataList) data).getKey();
            LOG.log(Level.INFO, "Got kvdl {0} {1}", new Object[]{dataKey, kvdl.getListOfKeyValueData()});

//            if (dataObject instanceof EndSetFilterData) {
//                EndSetFilterData efd = (EndSetFilterData) dataObject;
//                final CCSSetFilterEvent ccsSetFilterEvent = new CCSSetFilterEvent(efd.getFilterName(),
//                        efd.getFilterType(), efd.getFilterSlot(), efd.getFilterPosition());
//                ccs.fireEvent(ccsSetFilterEvent);
//                LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
//            } else if (dataObject instanceof SetFilterData) {
//                SetFilterData sfd = (SetFilterData) dataObject;
//                final CCSSetFilterEvent ccsSetFilterEvent = new CCSSetFilterEvent(sfd.getFilterName(),
//                        sfd.getFilterType());
//                ccs.fireEvent(ccsSetFilterEvent);
//                LOG.log(Level.INFO, "Sent: {0}", ccsSetFilterEvent);
//            }
        } else if (data instanceof KeyValueData) {
            String dataKey = ((KeyValueData) data).getKey();
            LOG.log(Level.INFO, "Got kvd {0} {1}", new Object[]{dataKey, ((KeyValueData) data).getValue()});
        }
    }
}
