package org.lsst.ccs.subsystem.ocsbridge.util;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystem.ocsbridge.xml.Mapping;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription.SALVariable;

/**
 *
 * @author Farrukh Azfar
 */
public class GenericConverter {

    private static final Logger LOG = Logger.getLogger(GenericConverter.class.getName());

    private final Map<String, String> nameChangeTelemetry;
    private final Map<String, String> nameChangeSettingsApplied;

    //keep a record of when data is missing on the bus ...
    private final List<String> mia = new ArrayList<>();
    private final Map<Class, List<String>> alreadyWarned = new HashMap<>();
    private final Map<String, Class> classMap;
    private final List<String> subsystemsToConvert;
    private final Mapping mapping;
    // print telemetry data processing
    private static final boolean DEBUG_PRINT_TRENDING = false;

    // print settings applied processing 
    private static final boolean DEBUG_PRINT_SETTINGS = false;

    // print reflection processing 
    private static final boolean DEBUG_PRINT_REFLECTION = false;

    // print location value processing 
    private static final boolean DEBUG_PRINT_BADVALUES = false;

    // returnBusData is an important function where we want to catch mistakes
    private static final boolean DEBUG_PRINT_RETURNBUSDATA = false;

    // change following to false for debugging 
    private static final boolean logAndContinueOnConversionError = true;
    boolean badBValue = false;
    double badDValue = Double.NaN;
    int badIValue = -111111111;
    float badFValue = Float.NaN;
    String badStringValue = "NOTFOUND";
    long badLValue = -111111111;
    short badSValue = (short) -11111;

    /**
     * Create a new generic converter for converting CCS classes to SAL classes
     *
     * @param subsystemsToConvert The list of subsystem this generic converter
     * should handle
     * @param classMap A map to be used in case the class name as created by
     * XMLMaker/ProcessedInfo to the actual SAL generated class
     * @param nameChangeTelemetry A map to be used in the case that telemetry
     * variable names have changed
     * @param nameChangeSettingsApplied A map to be used in the case that
     * settings applied variable names have changed.
     */
    public GenericConverter(List<String> subsystemsToConvert, Map<String, Class> classMap, Mapping mapping, Map<String, String> nameChangeTelemetry, Map<String, String> nameChangeSettingsApplied) {
        this.subsystemsToConvert = subsystemsToConvert;
        this.classMap = classMap;
        this.nameChangeTelemetry = nameChangeTelemetry;
        this.nameChangeSettingsApplied = nameChangeSettingsApplied;
        this.mapping = mapping;
    }

    public List<CameraTelemetry> telemetryConverter(Map<String, SALClassDescription> givenSalClassMapInfo, StatusSubsystemData data) throws ReflectiveOperationException {

        ArrayList<CameraTelemetry> salTelemetry = new ArrayList<>();

        // extract subsystem name  and get data
        String subsystemName = data.getOriginAgentInfo().getName();
        KeyValueData subsystemData = data.getSubsystemData();
        List<KeyValueData> values = (List<KeyValueData>) subsystemData.getValue();
        // data is contained in the list directly above ...

        if (!subsystemsToConvert.contains(subsystemName)) {
            System.out.println("********************************************** ");
            System.out.println("Warning: returning Empty List of CameraTelemetry ");
            System.out.println("Current susbsystem = " + subsystemName);
            System.out.println("Contents of subsystemsToConvert :" + subsystemsToConvert.toString());
            System.out.println("********************************************** ");
            return Collections.<CameraTelemetry>emptyList();
        }

        // identify this class's printout
        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ");
            System.out.println("***** Start Printout : GenericConverter3:telemetryConverter ******");
        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ");
            System.out.println("****** GenericConverter3:telemetryConverter : Start Filling Received Bus Data *******");
        }

        Map<String, Object> dataNamesAndValues = new HashMap<>();

        for (KeyValueData kvd : values) {
            String pathFromBus = kvd.getKey();

            // For now we ignore the states in the passed in telemetry
            if (!pathFromBus.contains("/state")) {

                // fill in bus data ...
                dataNamesAndValues.put(pathFromBus, kvd.getValue());
                if (DEBUG_PRINT_TRENDING) {
                    System.out.println(" Variable name from bus " + kvd.getKey());
                    System.out.println(" Variable type from bus " + kvd.getType());
                    System.out.println(" Variable value from bus " + kvd.getValue());
                }
            }

        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println("****** GenericConverter3:telemetryConverter: End Filling Received Bus Data *******");
        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ********GenericConverter3:telemetryConverter SalClassMap ************");
            for (Map.Entry<String, SALClassDescription> scd : givenSalClassMapInfo.entrySet()) {
                System.out.println(" Class Name from SALClassDescription " + scd.getKey());
                System.out.println(" Complete SAL Class Description " + scd.getValue().toString());
            }
        }

        for (String subsystem : subsystemsToConvert) {
            for (Map.Entry<String, SALClassDescription> scd : givenSalClassMapInfo.entrySet()) {
                String scdSALClassName = scd.getKey();

                if (DEBUG_PRINT_TRENDING) {

                    System.out.println(" Class Name from SALClassDescription " + scdSALClassName);

                }

                // actual SAL Class ...
                Class realSALClass = classMap.get(scdSALClassName);

                if (realSALClass == null) {

                    //System.out.println(" Printout of sal class name change follows : ");
                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" No entry found for " + scdSALClassName);
                        for (Map.Entry<String, Class> ck : classMap.entrySet()) {
                            System.out.println(" SalClassDescriptionName " + scdSALClassName + " Entry Key " + ck.getKey() + " class entry " + ck.getValue().getName());
                        }

                    }

                }

                // if the subsystemName from this batch of telemetry data is one of the subsystems we've been requested and if
                // the real class is not null we proceed. 
                if (subsystemName.equals(subsystem) && !(realSALClass == null)) {

                    // get the parameters of the first constructor - (there could be more so potentially this is something to look out for)    
                    Parameter[] parameters = realSALClass.getConstructors()[0].getParameters();

                    CameraTelemetry ct = (CameraTelemetry) reflectedClass(realSALClass, dataNamesAndValues, parameters, scd.getValue(), null);
                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" ");
                        System.out.println(" GenericConverter3:telemetryConverter: Class Received ");
                        System.out.println(ct.toString());

                    }

                    int check[] = checkClassCompleteness(ct);
                    int numMissingValues = check[0];
                    int numGetMethods = check[1];

                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" ");
                        System.out.println(" GenericConverter3:telemetryConverter: Class quality check ");
                        System.out.println(" Number of missing values " + check[0]);
                        System.out.println(" Number of get functions " + check[1]);
                    }

                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" ");
                        System.out.println("**** GenericConverter3:telemetryConverter: Missing values from reflected class ******");
                        System.out.println("Class name " + realSALClass.getName() + " Number of missing values " + numMissingValues + " Number of methods " + numGetMethods);
                    }

                    if (!(numMissingValues == numGetMethods)) {
                        salTelemetry.add(ct);
                    }
                }

            }

        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ");
            System.out.println(" Number of classes in SAL Telemetry " + salTelemetry.size());
            System.out.println("****************** End of Printout from GenericConverter3:telemetryConverter");
        }

        return salTelemetry;
    }

    public List<CameraEvent> settingsAppliedEventConverter(Map<String, SALClassDescription> givenSalClassMapInfo, StatusConfigurationInfo sdata) throws ReflectiveOperationException {

        ConfigurationInfo data = sdata.getConfigurationInfo();

        // allegedly sdata.getOriginAgentInfo().getName() returns a subsystemName from Status Configuration
        String subsystemFromStatusConfiguration = sdata.getOriginAgentInfo().getName();

        // local copy of salclassmapinfo passed in constructor not sure why I'm doing this 
        Map<String, SALClassDescription> salClassMapInfo = givenSalClassMapInfo;

        // return a list containing all the classes created here 
        ArrayList<CameraEvent> salSettings = new ArrayList<>();

        List<ConfigurationParameterInfo> cpinfo = data.getAllParameterInfo();

        if (DEBUG_PRINT_SETTINGS) {
            System.out.println(" ");
            System.out.println("**** GenericConverter3:settingsAppliedConverter Data from Bus Message ****");

            for (ConfigurationParameterInfo dinfo : data.getAllParameterInfo()) {
                System.out.println(" Category from Bus : " + dinfo.getCategoryName() + " ,Data from bus: " + dinfo.getPathName() + " = " + dinfo.getConfiguredValueObject());
            }
        }

        if (DEBUG_PRINT_SETTINGS) {
            System.out.println(" ");
            System.out.println("**** GenericConverter3:settingsAppliedConverter SALClassDescription information ****");
            for (Map.Entry<String, SALClassDescription> scd : salClassMapInfo.entrySet()) {
                String scdSALClassName = scd.getKey();
                System.out.println(" Class Name from SALClassDescription " + scdSALClassName);
                System.out.println(" Printout of SALClassDescription " + scd.getValue().toString());
            }
        }

        // subsystemsToConvert doesn't contain the given subsystem 
        if (!subsystemsToConvert.contains(sdata.getOriginAgentInfo().getName())) {
            System.out.println("********************************************** ");
            System.out.println("Warning: returning Empty List of CameraEvents ");
            System.out.println("Current susbsystem = " + sdata.getOriginAgentInfo().getName());
            System.out.println("Contents of subsystemsToConvert :" + subsystemsToConvert.toString());
            System.out.println("********************************************** ");
            return Collections.<CameraEvent>emptyList();
        }


        /*
         * Loop over all of the entries in the salClassMapInfo to see which categories they contain
         */
        for (Map.Entry<String, SALClassDescription> sc : salClassMapInfo.entrySet()) {
            Map<String, Object> thisCategoriesDataNew = new HashMap<>();
            String scCategory = sc.getValue().getCategory();
            // Check that the subsystem on the buses matches the subsystem from salClassMapInfo
            // This code accepts too much for subsystems like ats
            if (!sc.getKey().contains("_" + subsystemFromStatusConfiguration + "_")) {
                continue;
            }

            // I've apparently filled in scCategory 
            if (scCategory != null && scCategory != "null" && !scCategory.isEmpty()) {
                Map<String, ConfigurationParameterInfo> thisCategoriesCPI = alternateCategoryData(data, sc.getValue().getCategory());
                // now get the sal class name
                Class realSALClass = classMap.get(sc.getValue().getClassName());
                // the check on realSALClass seems to be sufficient 
                if (realSALClass != null) {
                    for (Map.Entry<String, ConfigurationParameterInfo> entry : thisCategoriesCPI.entrySet()) {
                        thisCategoriesDataNew.put(entry.getValue().getPathName(), entry.getValue().getConfiguredValueObject());
                    }

                    if (thisCategoriesDataNew.isEmpty()) {
                        continue;
                    }
                    // thisCategoriesDataNew is the same as local settings data 
                    Parameter[] parameters = realSALClass.getConstructors()[0].getParameters();

                    if (DEBUG_PRINT_SETTINGS) {
                        Method[] premethods = realSALClass.getMethods();
                        System.out.println(" ");
                        System.out.println("**** GenericConverter3:settingsAppliedConverter Preliminary printout of methods and parameter names ****");

                        for (Method premethod : premethods) {
                            System.out.println(" Method name of reflected class before constructor invocation- compare to whats on the bus " + premethod.getName());
                        }
                        for (Parameter parameter : parameters) {
                            System.out.println(" Parameter name of reflected class before constructor invocation- compare to whats on the bus " + parameter.getName());
                        }
                    }
                    String categoryVersion = data.getConfigurationDescriptionObject().getCategoryTag(scCategory).toString();

                    try {
                        CameraEvent ce = (CameraEvent) reflectedClass(realSALClass, thisCategoriesDataNew, parameters, sc.getValue(), categoryVersion);

                        if (DEBUG_PRINT_SETTINGS) {
                            System.out.println(" ");
                            System.out.println(" GenericConverter3:settingsAppliedConverter: Class Received ");
                            System.out.println(" Forcing recompilation ");
                            System.out.println(ce.toString());
                            System.out.println("***********(3)***************\n");
                        }

                        int check[] = checkClassCompleteness(ce);
                        int numMissingValues = check[0];
                        int numGetMethods = check[1];

                        if (DEBUG_PRINT_SETTINGS) {
                            System.out.println(" ");
                            System.out.println(" GenericConverter3:settingsAppliedConverter: Class quality check ");
                            System.out.println(" Number of missing values " + check[0]);
                            System.out.println(" Number of get functions " + check[1]);
                        }

                        if (!(numMissingValues == numGetMethods)) {
                            salSettings.add(ce);
                        }

                        if ((numMissingValues == numGetMethods)) {
                            if (DEBUG_PRINT_SETTINGS) {
                                System.out.println(" ");
                                System.out.println(" GenericConverter2:SettingsAppliedConverter");
                                System.out.println(" no values have been found on the bus");
                                System.out.println(" Values in the following have been extracted from ConfigInfo ");
                                System.out.println(" and inserted into the Map dataNamesAndValues ");
                                System.out.println(ce.toString());

                                for (String tospath : thisCategoriesDataNew.keySet()) {

                                    // For now we ignore the states in the passed in telemetry
                                    if (!tospath.contains("/state")) {

                                        System.out.println(tospath);
                                        System.out.println(thisCategoriesDataNew.get(tospath));
                                    }

                                }
                                System.out.println(" *********************** ");
                            }

                        }
                    } catch (ReflectiveOperationException | RuntimeException x) {
                        if (logAndContinueOnConversionError) {
                            LOG.log(Level.WARNING, "Error converting configuration, continuing", x);
                        } else {
                            throw x;
                        }
                    }

                }
            }

        }

        // still needs to be assigned a value - return SalSettings
        return salSettings;
    }

    public Object reflectedClass(Class cls, Map<String, Object> dataNamesAndValues, Parameter[] parameters, SALClassDescription salClassDescriptionList, String categoryVersion) throws ReflectiveOperationException {

     
        // arguments for this class's constructor     
        ArrayList<Object> constructorArgs = new ArrayList<>();
        Map<String, List<SALClassDescription.BusVariable>> locationBusVariableCorrespondence = new HashMap<>();

        for (SALVariable sv : salClassDescriptionList.getVariables()) {
            if (sv instanceof SALClassDescription.PatternMatchedSALVariable) {
                SALClassDescription.PatternMatchedSALVariable pv = (SALClassDescription.PatternMatchedSALVariable) sv;
                if (!pv.getBusVariables().isEmpty()) {
                    locationBusVariableCorrespondence.put(pv.getLocationVariable().getVariableName(), pv.getBusVariables());
                }
            }
        }

        if (DEBUG_PRINT_REFLECTION) {
            System.out.println("");
            System.out.println("****** Generic Converter 3 : SAL Class Description Information ****");
            for (SALVariable psv : salClassDescriptionList.getVariables()) {
                System.out.println(" SAL Variable " + psv.toString());
            }
            System.out.println(" ");
        }

        boolean found = false;

        // loop over received constructor parameters
        for (Parameter parameter : parameters) {

            found = false;

            if (DEBUG_PRINT_REFLECTION) {
                System.out.println(" ");
                System.out.println(" ***** START GenericConverter3:reflectedClass called by both telemetryConverter or settingsAppliedConverter ******");
                System.out.println(" Looping over SAL class parameters and dealing with parameter " + parameter.getName() + " type " + parameter.getType());
            }

            if (parameter.getName().trim().equals("priority")) {
                constructorArgs.add(1);
                if (DEBUG_PRINT_REFLECTION) {
                    System.out.println(" Priority found and set ");
                }
                found = true;

            }

            if (parameter.getName().trim().equals("version")) {
                constructorArgs.add(categoryVersion);
                if (DEBUG_PRINT_REFLECTION) {
                    System.out.println(" Version found and set ");
                }
                found = true;

            }
            // if this variable is not priority nor version which are present in all SettingsApplied classes(but not telemetry) then we loop
            // over all SALClassDescriptionList to find a match

            if (!found) {
                for (SALVariable sv : salClassDescriptionList.getVariables()) {

                    // what's the parameter Type ? Here we are most interested in Strings lists and arrays which are treated differently
                    String parameterType = parameter.getType().toString();
                    boolean parameterIsArray = parameter.getType().isArray();

                    // do I need to check any of the following or am I ok  ?
                    boolean isAnArrayList = parameterType.contains("List") || parameterType.contains("List");
                    boolean isAnArray = (!isAnArrayList) && parameterType.contains("[");
                    boolean isAString = parameterType.contains("String") || parameterType.contains("String");
                    boolean isAMap = parameterType.contains("Map");

                    if (DEBUG_PRINT_REFLECTION) {
                        System.out.println(" ");
                        System.out.println(" ***** GenericConverter3:reflectedClass Comparing ******");
                        System.out.println(" Comparing SAL Class parameter " + parameter.getName() + " ,with SAL class description " + sv.getVariableName());
                        System.out.println(" SAL Class Parameter type " + parameter.getType() + " ,SAL Class description parameter type " + sv.getType());
                    }

                    // we're checking to see if the SALClassDescription variable has used patternMatching - ie been through the string replacement                    
                    boolean involvesPatternMatching = sv instanceof SALClassDescription.PatternMatchedSALVariable;
                    if (involvesPatternMatching) {
                        if (DEBUG_PRINT_REFLECTION) {
                            System.out.println(" ");
                            System.out.println(" ***** GenericConverter3:reflectedClass involved pattern matching  ******");
                            System.out.println(" Comparing SAL Class parameter " + parameter.getName() + " ,with SAL class description " + sv.getVariableName());
                            System.out.println(" SAL Class Parameter type " + parameter.getType() + " ,SAL Class description parameter type " + sv.getType());
                        }
                    }

                    String svVariableName = "";
                    // check to see if the current SALClassDescription variable is a location variable
                    if (sv instanceof SALClassDescription.LocationVariable) {
                        svVariableName = ((SALClassDescription.LocationVariable) sv).getVariableName();

                    }

                    // check that the variable name on the SALClassDescription is the same as the parameter name in the class constructore
                    if (sv.getVariableName().toLowerCase().trim().equals(parameter.getName().toLowerCase().trim())) {

                        if (DEBUG_PRINT_REFLECTION) {
                            System.out.println(" ");
                            System.out.println(" ***** GenericConverter3:reflectedClass Match found ******");
                            System.out.println(" SAL Class parameter " + parameter.getName() + " SAL variable description " + sv.getVariableName());
                            System.out.println(" SAL Class parameter type " + parameter.getType() + " SAL variable description type " + sv.getType());
                        }

                        // the current parameter is an array and was created using pattern matching and it is not a location variable
                        if (parameterIsArray && involvesPatternMatching && !parameter.getName().trim().toLowerCase().contains("location")) {

                            // get the size of array
                            Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType(),
                                    ((SALClassDescription.PatternMatchedSALVariable) sv).getCount());

                            // get the associated busvariables so that the data can be extracted
                            for (int i = 0; i < ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().size(); i++) {

                                SALClassDescription.BusVariable busVarName = ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(i);

                                if (!(null == returnBusData(cls, parameter, busVarName.getPathAndNameOnBuses(), dataNamesAndValues))) {
                                    if (DEBUG_PRINT_REFLECTION) {
                                        System.out.println(" ");
                                        System.out.println("***** GenericConverter3:reflectedClass Filling an array ***");
                                        System.out.println(" Array SAL Class Name " + parameter.getName() + " SAL Description Name " + sv.getVariableName());
                                        System.out.println(" Bus type " + returnBusData(cls, parameter, busVarName.getPathAndNameOnBuses(), dataNamesAndValues).getClass().getTypeName());
                                        System.out.println(" Bus variable name " + busVarName.getPathAndNameOnBuses());
                                        System.out.println(" Array element index " + i + " Array element value from bus " + returnBusData(cls, parameter, busVarName.getPathAndNameOnBuses(), dataNamesAndValues));
                                    }
                                    Object returnedData = returnBusData(cls, parameter, busVarName.getPathAndNameOnBuses(), dataNamesAndValues);
                                    if (returnedData.getClass().isArray()) {
                                        arrayArg = returnedData;
                                    } else {
                                        // Farrukh April 15 2022 - so here I am assuming thatn
                                        // returnedData is a single element of an array 
                                        java.lang.reflect.Array.set(arrayArg, i, returnedData);
                                    }
                                } else {
                                    // Farrukh/Tony April 15 2022 - set array element to the correct bad value given the type of
                                    // the parameter - 
                                    java.lang.reflect.Array.set(arrayArg, i, AssignValuesForMissingBusData(parameter));
                                    // was   java.lang.reflect.Array.set(arrayArg, i, -11111111)); Now type is taken 
                                    // care of by passing parameter to AssignValuesForMissingBusData ..

                                }
                            }

                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println(" ***** GenericConverter3:reflectedClass arrayArg **** ");
                                System.out.println(" Adding  argument to constructor list : SAL class parameter  name " + parameter.getName() + " ,SAL class parameter type " + parameter.getClass().getTypeName());
                                System.out.println(" SAL class description parameter name " + sv.getVariableName() + " SAL class description parameter type " + sv.getType());
                                System.out.println(" Constructor array argument value " + arrayArg.toString() + " constructor array argument type " + arrayArg.getClass().getTypeName());
                            }
                            constructorArgs.add(arrayArg);
                            found = true;

                        }

                        // Farrukh April 15 2022 - should we be assuming that location will always be filled in ? 
                        // fill in the names of locations - these will be a String variable in the class constructor 
                        // deal with locations - we have to be sure that SAL classes NEVER will have a string array in the constructor.
                        // Containing location is not good enough, we have to make sure the variable 
                        if (!parameterIsArray && sv instanceof SALClassDescription.LocationVariable) {
                            // we've already checked that the names of the parameters in the class and the ProcessedInfo are the same 
                            // here we have created the value of the location variable inside the processed Info so no issues.

                            List<String> locationVals = new ArrayList<>();

                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("***** GenericConverter3:reflectedClass Location variables ***");
                                System.out.println("  SAL Class parameter name Name " + parameter.getName() + " SAL Description Name " + sv.getVariableName());
                                System.out.println(" SAL Class Description printout " + sv.toString());
                            }

                            if (locationBusVariableCorrespondence.get(sv.getVariableName()) == null) {
                                if (DEBUG_PRINT_REFLECTION) {
                                    System.out.println(" Null from locationBusVariableCorrespondence " + sv.toString());
                                }

                            }

                            for (SALClassDescription.BusVariable bv : locationBusVariableCorrespondence.get(sv.getVariableName())) {
                                locationVals.add((getLocationValue(bv.getPathAndNameOnBuses()).replace("/", "")));
                                if (DEBUG_PRINT_REFLECTION) {
                                    System.out.println(" Added location value " + (getLocationValue(bv.getPathAndNameOnBuses()).replace("/", "")));
                                }
                            }

                            // we should already have the location variables and they should always be present 
                            constructorArgs.add(convertStringList(locationVals));// has to be a string so I am not checking 
                            // if its not a string something is seriously wrong ...
                            found = true;

                        }

                        //
                        // one of the unusual cases where the bus should have a an array published on it or we didn't add it here .... 
                        if (parameterIsArray && !involvesPatternMatching) {

                            boolean notmatched = (sv instanceof SALClassDescription.SimpleSALVariable);
                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("****GenericConverter3:reflectedClass : An array not from pattern matching (unusual)");
                                System.out.println(" Should not be match - is it (false is trouble) ? " + notmatched);
                                System.out.println(" SAL parameter name " + parameter.getName() + " ,SALClassDescription parameter " + sv.getVariableName());
                                System.out.println(" SAL parameter type " + parameter.getType() + " ,SALClassDescription parameter type " + sv.getType());
                                System.out.println(" There should really be a Bus variable associated with this and that should be the next print is it ? ");
                                if (notmatched) {
                                    System.out.println(" Good news (see previous line) " + ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses());
                                }
                            }

                            // get bus variable name
                            String busVariableName = ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses();
                            if (!(null == returnBusData(cls, parameter, busVariableName, dataNamesAndValues))) {
                                constructorArgs.add(returnBusData(cls, parameter, busVariableName, dataNamesAndValues));
                            } else {
                                // Bus didn't carry a value for this parameter, its size is unknown, we create an array of one
                                // element and set it equal to the bad value for this particular type, Farrukh 16 April 2022. 
                                Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType());
                                java.lang.reflect.Array.set(arrayArg, 0, AssignValuesForMissingBusData(parameter));
                                constructorArgs.add(arrayArg);
                            }
                            found = true;
                        }

                        // pattern matching has occured and the parameter is not an array
                        if (involvesPatternMatching && !parameterIsArray && !parameter.getName().toLowerCase().contains("location")) {

                            boolean matched = (sv instanceof SALClassDescription.PatternMatchedSALVariable);
                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("****GenericConverter3:reflectedClass : Not an array but pattern matched (unusual)");
                                System.out.println(" Should not be match  ? " + matched);
                                System.out.println(" SAL parameter name " + parameter.getName() + " ,SALClassDescription parameter " + sv.getVariableName());
                                System.out.println(" SAL parameter type " + parameter.getType() + " ,SALClassDescription parameter type " + sv.getType());
                                System.out.println(" There should really be a Bus variable associated with this and that should be the next print is it ? ");
                                if (matched) {
                                    System.out.println(" Good news (see previous line) " + ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(0).getPathAndNameOnBuses());
                                }
                            }

                            if (!(null == returnBusData(cls, parameter, ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(0).getPathAndNameOnBuses(), dataNamesAndValues))) {
                                constructorArgs.add(returnBusData(cls, parameter, ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(0).getPathAndNameOnBuses(), dataNamesAndValues));
                            } else {
                                constructorArgs.add(AssignValuesForMissingBusData(parameter));
                            }
                            found = true;
                        }

                        if (!involvesPatternMatching && !parameterIsArray && !parameter.getName().toLowerCase().contains("location")) {

                            boolean notmatched = (sv instanceof SALClassDescription.SimpleSALVariable);
                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("****GenericConverter3:reflectedClass : Not an array not pattern matched - a vanilla parameter");
                                System.out.println(" Should not be match - is it (false is trouble) ? " + notmatched);
                                System.out.println(" SAL parameter name " + parameter.getName() + " ,SALClassDescription parameter " + sv.getVariableName());
                                System.out.println(" SAL parameter type " + parameter.getType() + " ,SALClassDescription parameter type " + sv.getType());
                                System.out.println(" There should really be a Bus variable associated with this and that should be the next print is it ? ");
                                if (notmatched) {
                                    System.out.println(" Good news (see previous line) " + ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses());
                                }
                            }

                            // proposed change April 9 2022. Also duplicate this in other places - Farrukh  
                            if (!(null == returnBusData(cls, parameter, ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses(),
                                    dataNamesAndValues))) {
                                Object valuevalue = returnBusData(cls, parameter, ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses(), dataNamesAndValues);

                                constructorArgs.add(returnBusData(cls, parameter, ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses(), dataNamesAndValues));
                            } else {

                                constructorArgs.add(AssignValuesForMissingBusData(parameter));

                            }

                            found = true;

                        }

                    }
                }

            }

            // If we are here then we have not derived the appropriate parameter name from the dictionary
            // so SALClassDescriptionInfo which extracts info from the dictionary and makes all the parameter
            // names for the actual SAL class constructor was unable to derive the name of this parameter
            // in this case it is difficult to know what to do except to take the parameter from the actual SAL
            // class and find its type and stick a value - we also won't know the size for any array
            // Farrukh April 15 2022
            if (!found) {

                if (DEBUG_PRINT_REFLECTION) {
                    System.out.println(" **** GenericConverter3:reflectedClass Not found parameter information ");
                    System.out.println(" Class Name " + cls.getCanonicalName());
                    System.out.println(" Didn't find " + parameter.getName());
                    System.out.println(" reflectedClass : Printing from stored information Is it here ? ");

                    for (String tospath : dataNamesAndValues.keySet()) {
                        System.out.println(" path name " + tospath);
                        System.out.println(" value " + dataNamesAndValues.get(tospath) + " type " + dataNamesAndValues.get(tospath).getClass().getTypeName());
                    }

                }

                mia.add(parameter.getName());
                alreadyWarned.put(cls, mia);
                int jj = Collections.frequency(alreadyWarned.get(cls), parameter.getName());
                // we should be able to set the number of warnings we want to tolerate
                // if you want to test for arrays and the type use parameter.getParameterizedType().getTypeName()
                // this will give you double[], long[], int[] etc...
                if (parameter.getType().isArray()) {
                    Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType(), 0);
                    constructorArgs.add(arrayArg);
                    // keeping older block around in case we want to revert                    
/*                    if (parameter.getParameterizedType().getTypeName().toLowerCase().contains("double")) {
                        constructorArgs.add(new double[0]);
                    }
                    if (parameter.getParameterizedType().getTypeName().toLowerCase().contains("long")) {
                        constructorArgs.add(new long[0]);
                    }
                    if (parameter.getParameterizedType().getTypeName().toLowerCase().contains("int")) {
                        constructorArgs.add(new int[0]);
                    } */
                } else {
                    constructorArgs.add(AssignValuesForMissingBusData(parameter));
                }

            }

        }

        if (DEBUG_PRINT_REFLECTION) {
            System.out.println(" ");
            System.out.println("***** GenericConverter3.settingsAppliedConverter: cross check just before reflection ******");
            System.out.println(" Constructor Argugments and types about to pass these on to constructor ");
            for (int i = 0; i < constructorArgs.size(); i++) {
                System.out.println(" Value and Type " + constructorArgs.get(i) + " " + constructorArgs.get(i).getClass().getTypeName());
                System.out.println(" Value and Type 2 " + constructorArgs.get(i).getClass().getTypeParameters());

            }

            System.out.println(" Please compare the with the constructor of the class " + cls.getConstructors()[0].toString());
        }
        // so now constructor args must be full.....or not let's check ....
        try {
            // potentially useful debug statement 
          //  System.out.println("printing the class out " + cls.toString());
            return cls.getConstructors()[0].newInstance(constructorArgs.toArray());
        } catch (ReflectiveOperationException | RuntimeException x) {
            throw new RuntimeException(String.format("Error creating SAL event for class %s with args %s", cls.getCanonicalName(), betterToString(parameters, constructorArgs)), x);
        }

    }

    // given the bus path name and the mapping this function will generate the value of the location 
    String getLocationValue(String busPathName) {

        Mapping.Match match = mapping.match(busPathName);
        if (match != null) {
            return match.getLocation();
        } else {
            return "NOTFOUND";
        }
    }

    // return data 
    Object returnBusData(Class cls,java.lang.reflect.Parameter parameter, String pathName, Map<String, Object> dataNamesAndValues) { //List<KeyValueData> values) {

     
        Object returnValue = null;
        boolean classTypeIsConfig = false;
        
        if(cls.getCanonicalName().contains("Event") && cls.getCanonicalName().contains("org.lsst.sal.cccamera.event")){
           classTypeIsConfig = true;
        } 
        
        // We are currently converting Durations to strings, here is check against that 
        // if however encounter a duration that should not be a string we should not convert it
        // to a string ..rather to a double 
         
        if (DEBUG_PRINT_RETURNBUSDATA) {
            System.out.println("*** Entering returnBusData ***");
            System.out.print(" Parameter name " + parameter.getName() + ", Parameter type " + parameter.getParameterizedType().getTypeName());

            System.out.println(" Given path name " + pathName);
            System.out.println(" Stored path names and values " + dataNamesAndValues.toString());
        }

        for (String tospath : dataNamesAndValues.keySet()) {

            String key = tospath;

            // For now we ignore the states in the passed in telemetry
            if (!key.contains("/state") && pathName.toLowerCase().trim().equals(tospath.toLowerCase().trim())) {

                
                // change doubles to floats, remove this once xml is fixed 
                returnValue = dataNamesAndValues.get(tospath);
                if (returnValue instanceof Double) {
                    returnValue = ((Double) returnValue).floatValue();
                    if (DEBUG_PRINT_RETURNBUSDATA) {
                        System.out.println(" instance of double " + returnValue);
                    }

                }
                
//     double interval = java.time.Duration.ofMillis(BUSVALUE);
//     double interval = interval/1000.0; // convert to seconds ...                
            if( returnValue instanceof java.time.Duration && !parameter.getType().toString().toLowerCase().contains("string")) {
            // change to just double once xml is fixed 
                    returnValue = ((Double) returnValue).floatValue();
                    if (DEBUG_PRINT_RETURNBUSDATA) {
                        System.out.println(" instance of double " + returnValue);
                    }

                }
           

                if (returnValue instanceof HashMap) {
                    returnValue = ((HashMap) returnValue).toString();

                    if (DEBUG_PRINT_RETURNBUSDATA) {
                        System.out.println(" instance of hash map ");
                        System.out.println(" Hash Map printout " + returnValue + " " + returnValue.getClass().getTypeName());

                        if (returnValue.getClass().getTypeName().contains("[")) {
                            System.out.println(" apparrently is a hash map and a string ");
                            System.out.println(" Is a string array/list " + returnValue);
                        }
                    }
                }

                if (returnValue.getClass().getTypeName().toLowerCase().contains("string") && returnValue instanceof String[]) {
                    if (DEBUG_PRINT_RETURNBUSDATA) {
                        System.out.println(" instance of String [] " + returnValue);
                    }
                    returnValue = convertStringArray((String[]) returnValue);
                }

                // Maybe here check for arrayList and containing a string - though I am not sure this will work. Farrukh April 9 2022
                // TODO: We should use toString on any unsupported Object, not an explicit list
 // old  : if (returnValue instanceof List || returnValue instanceof Map || returnValue instanceof Enum || returnValue instanceof Duration) {

                if (returnValue instanceof List || returnValue instanceof Map || returnValue instanceof Enum ) {

                    // Can find out what the first element in the array list is with the following 
                    //  System.out.println("INSTANCE OF " + ((ArrayList)returnValue).get(0).getClass().getTypeName());
                    returnValue = returnValue.toString();
                }
                
                if(returnValue instanceof java.time.Duration && parameter.getType().toString().toLowerCase().contains("string")){
                   returnValue = returnValue.toString();
                }

                if (returnValue.getClass().getTypeName().toLowerCase().contains("string") && (returnValue instanceof List)) {

                    returnValue = convertStringList((List<String>) returnValue);
                }

            }
        }

        // I think Here we should be returning badDValue badIValue etc - depending on whether we find a value or not - Farrukh April 8 2022. 
        if (null == returnValue) {
            if (DEBUG_PRINT_RETURNBUSDATA) {
                System.out.println(" From returnBusData : return value is null " + returnValue);
            }
            // the following warning is for the case when a configuration event class has a null 
            // value for any pathName - this shouldn't happen thus it is outside the DEBUG_PRINT 
            if(classTypeIsConfig){
             System.out.println(" ====================================================================");
             System.out.println(" Warning ! Bus data contains null for pathname : " + pathName);
             System.out.println(" this corresponds to class name  " + cls.getTypeName() + ", parameter name " + parameter.getName() );
             System.out.println(" ====================================================================");
            }
            // will pick up parameter as a double or a float and not as java.time.Duration
            // since the java.time.Duration will (should!) only be the data type on the bus
            returnValue = AssignValuesForMissingBusData(parameter);
        }

        if (DEBUG_PRINT_RETURNBUSDATA) {
            System.out.println(" *** End returnBusData ***");
        }
        return returnValue;
    }

    // clean up the class name used for xmls
    String cleanupXMLClassName(String xmlClassName) {

        xmlClassName = xmlClassName.replace("comcam-", "");
        String[] blahs = xmlClassName.split("_");
        String cleanedName = "";

        for (int j = 1; j < blahs.length; j++) {
            cleanedName += blahs[j] + "_";
        }

        int lastIndex = cleanedName.lastIndexOf("_");
        cleanedName = cleanedName.substring(0, lastIndex);

        return cleanedName.trim();
    }

    // clean up the SAL class Name for comparision
    String cleanupSALClassName(String salClassName) {

        String cleanedName = salClassName.split("\\.")[salClassName.split("\\.").length - 1];

        cleanedName = cleanedName.replace("ATCamera", "").trim();
        cleanedName = cleanedName.replace("MTCamera", "").trim();
        cleanedName = cleanedName.replace("CCCamera", "").trim();
        cleanedName = cleanedName.replace("Telemetry", "").trim();
        cleanedName = cleanedName.replace("ConfigurationAppliedEvent", "").trim();

        if (cleanedName.startsWith("_")) {
            cleanedName = cleanedName.replaceFirst("[_]", "").trim();
        }

        return cleanedName.trim();
    }

    boolean badValueOrValues(Object invokedValue, Object badValue, String className, String methodName) {
        boolean bad = true;

        boolean isanArray = invokedValue.getClass().isArray();
        String type = invokedValue.getClass().getTypeName();
        boolean isDouble = false;
        boolean isString = false;
        boolean isInt = false;
        boolean isFloat = false;
        boolean isLong = false;
        boolean isBool = false;
        boolean isShort = false;

        if (type.toLowerCase().contains("double")) {
            isDouble = true;
            badValue = badDValue;
        } else if (type.toLowerCase().contains("string")) {
            isString = true;
            badValue = badStringValue;
        } else if (type.toLowerCase().contains("int")) {
            isInt = true;
            badValue = badIValue;
        } else if (type.toLowerCase().contains("float")) {
            isFloat = true;
            badValue = badFValue;
        } else if (type.toLowerCase().contains("long")) {
            isLong = true;
            badValue = badLValue;
        } else if (type.toLowerCase().contains("short")) {
            badValue = badSValue;
        } // Comment added April 14 2022, Farrukh Azfar : Not sure what to do about booleans - we usually add 
        // false for a bad value and there are only two values available and it is conceivable that false is also the 
        // correct value to return ! 

        if (isanArray && isDouble) {

            double[] doubleArray = (double[]) invokedValue;
            for (int i = 0; i < doubleArray.length; i++) {
                // even if one value is good the array is passable        
                if (!Double.isNaN(doubleArray[i])) {
                    bad = false;
                }
            }
        }
        if (isanArray && isInt) {

            int[] intArray = (int[]) invokedValue;
            for (int i = 0; i < intArray.length; i++) {
                // even if one value is good the array is passable        
                if (!(intArray[i] == (int) badValue)) {
                    bad = false;
                }
            }
        }
        if (isanArray && isFloat) {

            float[] floatArray = (float[]) invokedValue;
            for (int i = 0; i < floatArray.length; i++) {
                // even if one value is good the array is passable        
                if (!Float.isNaN(floatArray[i])) {
                    bad = false;
                }
            }
        }

        if (!isanArray) {
            if (!invokedValue.equals(badValue)) {
                bad = false;
            }
        }

        if (DEBUG_PRINT_BADVALUES) {
            System.out.println(" ******* GenericConverter3:badValueOrValues ******* ");
            System.out.println(" Class name " + className + " method name " + methodName);
            System.out.println(" Array ? " + isanArray + " type " + invokedValue.getClass().getTypeName());
            System.out.println(" Bad value check " + invokedValue + " " + badValue);
        }
        return bad;
    }

    Map<String, Object> copyStringObjectMap(Map<String, Object> inMap) {

        Map<String, Object> copy = new HashMap<>();

        copy.putAll(inMap);

        return copy;
    }

    private static String betterToString(Parameter[] parameters, ArrayList<Object> data) {
        List<String> result = new ArrayList<>();
        for (int i = 0; i < data.size(); i++) {
            boolean classesMatch = ClassUtils.isAssignable(data.get(i).getClass(), parameters[i].getType(), true);
            String argString = parameters[i].getName() + "=" + (data.get(i).getClass().isArray() ? arrayToString(data.get(i)) : data.get(i).toString());
            if (!classesMatch) {
                argString += String.format(" WARNING: classMismatch %s != %s", parameters[i].getType(), data.get(i).getClass());
            }
            result.add(argString);
        }
        return "[" + String.join(", ", result) + "]";
    }

    private static String convertStringList(List<String> stringList) {

        String[] localArray = new String[stringList.size()];
        stringList.toArray(localArray);
        return manipulateStringArray(localArray);
    }

    private static String convertStringArray(String[] stringArray) {

        String[] localArray = null;

        return manipulateStringArray(stringArray);

    }

    private static String manipulateStringArray(String[] inputArray) {
        String returnStringI = "";
        String returnStringF = "";
        int lengthOfStringArray = inputArray.length;
        if (!(lengthOfStringArray > 0)) {
            System.out.println("Warning from GenericConverter3:manipulateStringArray : Length of string Array is 0 !");
            returnStringF = "NOTFOUND";
        } else {
            for (int i = 0; i < lengthOfStringArray; i++) {
                returnStringI += inputArray[i] + ":";
                if (i == lengthOfStringArray - 1) {
                    int target = returnStringI.lastIndexOf(":");
                    if (target == -1) {
                        returnStringF = returnStringI;
                    } else {
                        returnStringF = returnStringI.substring(0, target);
                    }
                }
            }
        }
        return returnStringF;
    }

    private static String arrayToString(Object array) {
        List<String> result = new ArrayList<>();
        int l = Array.getLength(array);
        for (int i = 0; i < l; i++) {
            result.add(Array.get(array, i).toString());
        }
        return "[" + String.join(", ", result) + "]";
    }

    // Comment added April 14 2022 - Farrukh Azfar : this program calls badValueOrValues
    // to check if a class has a certain number of bad values set for its constructor 
    // parameters - the purpose here is to determine if a class should be returned if
    // it has greater than a total number of bad arguments ...
    int[] checkClassCompleteness(Object cls) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException {

        Parameter[] thisClassParameters = cls.getClass().getConstructors()[0].getParameters();
        Method[] methods = cls.getClass().getDeclaredMethods();
        ArrayList<Method> getMethods = new ArrayList<>();
        int numMissingValues = 0;
        int numGetMethods = 0;
        int[] returnValues = new int[2];

        for (int i = 0; i < methods.length; i++) {
            String methodName = methods[i].getName();

            String[] parts = methodName.split("\\s");
            if (!parts[parts.length - 1].contains("toString") && parts[parts.length - 1].contains("get")) {
                getMethods.add(methods[i]);
                numGetMethods++;
            }
        }

        boolean checkbadvalue = false;

        for (Method method : getMethods) {

            if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("double")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badDValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("string")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badStringValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("float")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badFValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("int")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badIValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("long")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badLValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("short")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badSValue, cls.getClass().getName(), method.getName());
            }
            if (checkbadvalue) {
                numMissingValues++;
            }

            if (!checkbadvalue && method.getName().toLowerCase().contains("location")) {
                numMissingValues++;
            }

        }

        returnValues[0] = numMissingValues;
        returnValues[1] = numGetMethods;
        return returnValues;
    }

    Map<String, ConfigurationParameterInfo> alternateCategoryData(ConfigurationInfo data, String category) {
        Map<String, ConfigurationParameterInfo> returnMap = new HashMap<>();
        System.out.println(" AlternateCategoryData requested for category " + category);
        try {
            returnMap = data.getCurrentParameterInfoForCategory(category);
        } catch (NullPointerException e) {
            System.out.println(" Null pointer exception caught " + e.toString() + " for category " + category);

        }

        return returnMap;
    }

    Object AssignValuesForMissingBusData(java.lang.reflect.Parameter parameter) {

        Object badValue = null;
        String type = parameter.getParameterizedType().getTypeName().toLowerCase();

        if (type.contains("float")) {
            badValue = (float) badFValue;
        }
        if (type.contains("int")) {
            badValue = (int) badIValue;
        }
        if (type.contains("long")) {
            badValue = (long) badLValue;
        }
        if (type.contains("boolean")) {
            badValue = (boolean) badBValue;
        }
        if (type.contains("double")) {
            badValue = (double) badDValue;
        }
        if (type.contains("short")) {
            badValue = (short) badSValue;
        }
        if (type.contains("string")) {
            badValue = (String) badStringValue;
        }
        return badValue;
    }

}
